<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Session;

class ConferenceAdminController extends Controller
{
    /**
     * 🔹 Return site name
     */
    public function sitename()
    {
        $site = DB::table('configurations')
            ->where('configTitle', 'sitename')
            ->first();

        return response()->json([
            'status'   => true,
            'sitename' => $site->configValue ?? 'Scisynopsis Conferences',
        ]);
    }

    /**
     * 🔹 Login (replaces CI3 verify)
     */
    public function login(Request $request)
    {
        $request->validate([
            'email'    => 'required|email',
            'password' => 'required',
        ]);

        $user = DB::table('users')->where('email', $request->email)->first();

        if (!$user || !Hash::check($request->password, $user->password)) {
            return response()->json([
                'status'  => false,
                'message' => 'Invalid Username / Password',
            ]);
        }

        // ✅ Same behavior as CI3 session->set_userdata()
        Session::put([
            'is_logged_in' => 1,
            'admin_id'     => $user->id,
            'admin_email'  => $user->email,
            'usertype'     => $user->usertype ?? 1,
        ]);

        return response()->json([
            'status'  => true,
            'message' => 'Login successful',
            'user'    => [
                'id'    => $user->id,
                'email' => $user->email,
            ],
        ]);
    }

    /**
     * 🔹 Dashboard data
     */
    public function dashboard()
    {
        if (!Session::get('is_logged_in')) {
            return response()->json([
                'status'  => false,
                'message' => 'Not logged in',
            ], 401);
        }

        return response()->json([
            'status'  => true,
            'message' => 'Welcome To Scisynopsis Conferences Admin Panel',
            'user'    => [
                'id'    => Session::get('admin_id'),
                'email' => Session::get('admin_email'),
            ],
        ]);
    }

    /**
     * 🔹 Switch current conference
     */
    public function setConference(Request $request)
    {
        $request->validate(['conference_id' => 'required|integer']);
        $conf = DB::table('conferences')->find($request->conference_id);

        if (!$conf) {
            return response()->json([
                'status'  => false,
                'message' => 'Conference not found',
            ]);
        }

        Session::put([
            'conferenceId'       => $conf->id,
            'adminconferenceId'  => $conf->id,
            'folderName'         => $conf->unique_id,
            'prev_conference_id' => $conf->prev_conference_id,
        ]);

        return response()->json([
            'status'     => true,
            'message'    => 'Conference switched successfully',
            'conference' => $conf,
        ]);
    }

    /**
     * 🔹 Logout
     */
    public function logout()
    {
        Session::flush();

        return response()->json([
            'status'  => true,
            'message' => 'Logged out successfully',
        ]);
    }
}
