<?php

namespace App\Http\Controllers\Company;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Schema;

// === COMPANY MODELS (Main DB) ===
use App\Models\Company\{
    CompanyMetaData,
    CompanyHomepageScript,
    CompanyConfiguration,
    CompanyWebCms,
    CompanyWhatsApp,
    CompanyChatScript,
    CompanyPageNavigation,
    CompanyConference,
    CompanySpeaker,
    CompanyCollaboration,
    CompanyPresentation,
    CompanyEbook,
    CompanyCountry,
    CompanyCity,
    JournalCountry,
    JournalCity,
    CompanyProgramSpeaker
};

// === JOURNAL MODELS (journal DB) ===
use App\Models\Company\{
    CompanyArticle,
    CompanyAuthor,
    CompanyJournal
};

class CompanySearchController extends Controller
{
    public function index(Request $request)
    {
        $keyword = trim((string) $request->query('keyword', ''));

        if ($keyword === '') {
            return response()->json(['error' => 'Keyword parameter is required'], 400);
        }

        /* ====================== HEADER ====================== */
        $meta = CompanyMetaData::where('page_type', 27)->latest('id')->first();
        $homepage_script = CompanyHomepageScript::where('status', 1)->latest('id')->first();

        /* ====================== ARTICLES ====================== */
        $articles = CompanyArticle::with(['journal', 'authors.author'])
            ->searchByKeyword($keyword)
            ->get()
            ->map(function ($article) {
                return [
                    'id' => $article->id,
                    'article_title' => $article->article_title,
                    'alias_title' => $article->alias_title,
                    'author_names' => $article->authors->pluck('author.author_name')->implode(', '),
                    'journal_id' => $article->journal_id,
                    'journalInfo' => $article->journal,
                    'doi' => $article->doi,
                    'doi_url' => $article->doi_url,
                    'authorsInfo' => $article->authors->map(function ($a) {
                        return [
                            'id' => $a->author->id ?? null,
                            'author_name' => $a->author->author_name ?? '',
                        ];
                    }),
                ];
            });

        /* ====================== JOURNALS ====================== */
        $journals = CompanyJournal::where('status', 1)
            ->where(function ($q) use ($keyword) {
                if (Schema::connection('journal')->hasColumn('journals', 'journal_title')) {
                    $q->orWhere('journal_title', 'like', "%$keyword%");
                }
                if (Schema::connection('journal')->hasColumn('journals', 'journal_name')) {
                    $q->orWhere('journal_name', 'like', "%$keyword%");
                }
                if (Schema::connection('journal')->hasColumn('journals', 'title')) {
                    $q->orWhere('title', 'like', "%$keyword%");
                }
                if (Schema::connection('journal')->hasColumn('journals', 'short_name')) {
                    $q->orWhere('short_name', 'like', "%$keyword%");
                }
            })
            ->get();

        /* ====================== CONFERENCES ====================== */
        $conferences = CompanyConference::where('status', 1)
            ->where('publish_status', 1)
            ->where(function ($q) use ($keyword) {
                $q->where('conference_name', 'like', "%$keyword%")
                    ->orWhere('short_name', 'like', "%$keyword%")
                    ->orWhere('conference_first_title', 'like', "%$keyword%")
                    ->orWhere('conference_theme', 'like', "%$keyword%")
                    ->orWhere('conference_location', 'like', "%$keyword%")
                    ->orWhere('conference_year', 'like', "%$keyword%");
            })
            ->orderBy('conference_name', 'ASC')
            ->get()
            ->map(function ($conf) {
                return [
                    'id' => $conf->id,
                    'conference_name' => $conf->conference_name,
                    'short_name' => $conf->short_name,
                    'conference_url' => $conf->conference_url,
                    'from_date' => $conf->from_date,
                    'to_date' => $conf->to_date,
                    // ✅ FIX: Use Journal DB for location + null fallback
                    'countryInfo' => JournalCountry::find($conf->country_id),
                    'cityInfo' => JournalCity::find($conf->city_id) ?? (object)['city_name' => ''],
                ];
            });

        /* ====================== CONFERENCE SPEAKERS (keynoteArray logic) ====================== */
        $keynoteArray = [];

        if ($conferences->count() > 0) {
            $firstConference = $conferences->first();

            $aspeakers = CompanySpeaker::where('status', 1)
                ->where('conference_id', $firstConference['id'])
                ->orderBy('speaker_title', 'ASC')
                ->get();

            foreach ($aspeakers as $sp) {
                $keynoteArray[] = [
                    'id' => $sp->id,
                    'speaker_name' => $sp->speaker_name,
                    'speaker_title' => $sp->speaker_title,
                    'alias_name' => $sp->alias_name,
                    'affliation' => $sp->affliation,
                    'speaker_img' => $sp->speaker_img,
                    'country_id' => $sp->country_id,
                    // ✅ use journal db for country + city with null fallback
                    'countryInfo' => JournalCountry::where('status', 1)
                        ->where('id', $sp->country_id)
                        ->first(),
                    'cityInfo' => JournalCity::find($sp->city_id) ?? (object)['city_name' => ''],
                ];
            }
        }

        /* ====================== EBOOKS ====================== */
        $eBooks = CompanyEbook::getEbooksByCatAndTitle(0, $keyword);

        /* ====================== AUTHORS ====================== */
        $authors = CompanyAuthor::where('status', 1)
            ->where(function ($q) use ($keyword) {
                $q->where('author_name', 'like', "%$keyword%")
                    ->orWhere('alias_name', 'like', "%$keyword%");
            })
            ->get()
            ->map(function ($a) {
                return [
                    'id' => $a->id,
                    'author_name' => $a->author_name,
                    'alias_name' => $a->alias_name,
                    'authorAffs' => [],
                ];
            });

        /* ====================== COLLABORATIONS ====================== */
        $collaborations = CompanyCollaboration::getCollaborationsByKeyword($keyword);

        /* ====================== PRESENTATIONS ====================== */
        $presentations = CompanyPresentation::getPosterPresentationsByKeyword($keyword);

        /* ====================== FOOTER ====================== */
        $address = CompanyWebCms::where('status', 1)->where('page_type', 10)->latest('id')->first();
        $sociallinks = CompanyWebCms::where('status', 1)->where('page_type', 11)->latest('id')->first();
        $whatsapp = CompanyWhatsApp::where('status', 1)->latest('id')->first();
        $chatScript = CompanyChatScript::where('status', 1)->latest('id')->first();
        $journalsUrl = CompanyConfiguration::where('status', 1)->latest('id')->first();
        $conferenceUrl = CompanyConfiguration::where('status', 1)->latest('id')->first();

        /* ====================== HEADER & FOOTER NAVIGATION ====================== */
        $headerNavigation = CompanyPageNavigation::where('status', 1)
            ->where('page_type', 1)
            ->where('parent_id', 0)
            ->orderBy('orderpos', 'ASC')
            ->with('children')
            ->get();

        $footerQuickLinks = CompanyPageNavigation::where('status', 1)
            ->where('page_type', 2)
            ->where('parent_id', 0)
            ->orderBy('orderpos', 'ASC')
            ->get();

        $footerUsefulLinks = CompanyPageNavigation::where('status', 1)
            ->where('page_type', 3)
            ->where('parent_id', 0)
            ->orderBy('orderpos', 'ASC')
            ->get();

        /* ====================== FINAL JSON RESPONSE ====================== */
        return response()->json([
            'meta' => $meta,
            'homepage_script' => $homepage_script?->script_content,
            'nobanners' => 1,
            'keyword' => $keyword,
            'articles' => $articles,
            'journals' => $journals,
            'conferences' => $conferences,
            'eBooks' => $eBooks,
            'authors' => $authors,
            'collaborations' => $collaborations,
            'speakers' => $keynoteArray,
            'presentations' => $presentations,
            'address' => $address,
            'sociallinks' => $sociallinks,
            'whatsapp' => $whatsapp,
            'chatScript' => $chatScript?->script_content,
            'journalsUrl' => $journalsUrl,
            'conferenceUrl' => $conferenceUrl,
            'headerNavigation' => $headerNavigation,
            'footerQuickLinks' => $footerQuickLinks,
            'footerUsefulLinks' => $footerUsefulLinks,
        ]);
    }
}
