<?php

namespace App\Http\Controllers\Api;
use App\Http\Controllers\Controller;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;
use Mpdf\Mpdf;

class v4Controller extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | v4Index â€“ EXACT v1Index PATTERN (simplified + extend later)
    |--------------------------------------------------------------------------
    */
    public function v4Index(Request $request)
    {
        $table = $request->get('table');

        if (!$table || !Schema::hasTable($table)) {
            return response()->json(['error' => 'Invalid table'], 400);
        }

        // ðŸ”¹ Table-specific queries (add more like v1 as you migrate modules)

if ($table === 'user_confs1') {

    $rows = DB::table('user_confs1 as uc')
        ->leftJoin('conferences as c', 'c.id', '=', 'uc.conference_id')
        ->leftJoin('users as u', 'u.id', '=', 'uc.admin_id')
        ->select([
            'uc.id',
            'uc.conference_id',
            'uc.admin_id',
            'uc.created_date',
            DB::raw('COALESCE(c.conference_name, "") AS conference_name'),
            DB::raw('TRIM(CONCAT(COALESCE(u.firstname, ""), " ", COALESCE(u.lastname, ""))) AS admin_name'),
        ])
        ->orderBy('uc.id', 'DESC')
        ->get();   // 👈 returns a Collection of rows

    // IMPORTANT: return a plain array for DataTables (dataSrc: '')
    return response()->json($rows);
}

elseif ($table === 'user_confs') {

    $rows = DB::table('user_confs as uc')
        ->leftJoin('conferences as c', 'c.id', '=', 'uc.conference_id')
        ->leftJoin('users as u', 'u.id', '=', 'uc.admin_id')
        ->select([
            'uc.id',
            'uc.conference_id',
            'uc.admin_id',
            'uc.created_date',
            DB::raw('COALESCE(c.conference_name, "") AS conference_name'),
            DB::raw('TRIM(CONCAT(COALESCE(u.firstname, ""), " ", COALESCE(u.lastname, ""))) AS admin_name'),
        ])
        ->orderBy('uc.id', 'ASC')
        ->get();   // 👈 returns a Collection of rows

    // IMPORTANT: return a plain array for DataTables (dataSrc: '')
    return response()->json($rows);
}



        
        elseif ($table === 'users') {
    // Super Admin Users (admusers1) – ascending like CI3
    $query = DB::table('users')->orderBy('id', 'asc');
}
elseif ($table === 'c_invoices') {

        $query = DB::table('c_invoices as inv')
            ->leftJoin('c_reg_users as u', 'u.id', '=', 'inv.user_id')
            ->leftJoin('conferences as c', 'c.id', '=', 'inv.conference_id')
            ->select([
                'inv.id        as id',          // primary key
                'inv.id        as invoice_id',
                'inv.conference_id',

                'u.id          as user_id',
                'u.designation',
                'u.firstname',
                'u.lastname',
                'u.usr_email',
                'u.user_phone',
                'u.user_type',
                'u.country',
                'u.currency',
                'u.company',
                'u.category',
                'u.total_amount',
                'u.total_extra_amount',
                'u.payment_status',
                'u.created_date',

                'c.short_name  as conferenceName',
            ])
            ->where('inv.status', 1);

   
        $rows = $query
            ->orderBy('inv.id', 'DESC')
            ->get()
            ->map(function ($row) {
                $extra = (int) ($row->total_extra_amount ?? 0);
                $base  = (int) ($row->total_amount ?? 0);
                $total = $extra + $base;

                if ((string) $row->currency === '2') {
                    $cur = 'USD';
                } elseif ((string) $row->currency === '1') {
                    $cur = 'GBP';
                } else {
                    $cur = 'EUR';
                }

                return [
                    'id'             => $row->id,          // c_invoices.id
                    'invoice_id'     => $row->invoice_id,
                    'user_id'        => $row->user_id,

                    'designation'    => $row->designation,
                    'firstname'      => $row->firstname,
                    'lastname'       => $row->lastname,
                    'usr_email'      => $row->usr_email,
                    'user_phone'     => $row->user_phone,
                    'userType'       => $row->user_type,
                    'country'        => $row->country,
                    'currency'       => $row->currency,
                    'currency_text'  => $cur,
                    'company'        => $row->company,
                    'category'       => $row->category,
                    'created_date'   => $row->created_date,

                    'total_amount'   => $total,
                    'payment_status' => $row->payment_status,
                    'conferenceName' => $row->conferenceName,
                ];
            });

        return response()->json($rows);
    }

elseif ($table === 'meta_data') {
    return response()->json(
        DB::table('meta_data')->orderBy('id','asc')->get()
    );
}
 elseif ($table === 'user_subscriptions') {
        // CI3: getTableRowDataOrder("user_subscriptions", ["status" => 1], "id", "DESC")
        $query = DB::table('user_subscriptions')
            ->where('status', 1)
            ->orderBy('id', 'desc');
    }
    // previous_conference_gallery
elseif ($table === 'previous_conference_gallery') {

    $query = DB::table('previous_conference_gallery as g')
        ->leftJoin('conferences as c', 'c.id', '=', 'g.conference_id')
        ->select([
            'g.id',
            'g.parent_id',
            'g.conference_id',
            'g.year_id',
            'g.gallery_title',
            'g.gallery_img',
            'g.created_date',
            DB::raw('COALESCE(c.conference_name, "") as conference_name'),
            DB::raw('COALESCE(c.short_name, "")  as short_name'),
        ]);

    $parentId = (int) $request->input('parent_id', 0);
    $query->where('g.parent_id', $parentId);

    // 🔹 Get available years for this parent (like CI3: getTableYears)
    $availableYears = DB::table('previous_conference_gallery')
        ->where('parent_id', $parentId)
        ->whereNotNull('year_id')
        ->distinct()
        ->orderBy('year_id', 'ASC')
        ->pluck('year_id');

    $yearId = null;
    if ($request->filled('year_id')) {
        $yearId = (int) $request->input('year_id');
    } elseif ($parentId === 0 && $availableYears->isNotEmpty()) {
        // ✅ CI3 behavior: if no year_id & parent=0, use first (largest) year
        $yearId = $availableYears->first(); // DESC → first = latest year
    }

    // Apply year filter if resolved
    if ($yearId !== null) {
        $query->where('g.year_id', $yearId);
    }

    // Optional: conference_id filter
    if ($request->filled('conference_id')) {
        $query->where('g.conference_id', (int) $request->input('conference_id'));
    }

    $rows = $query->orderBy('g.id', 'ASC')->get();

    return response()->json($rows);
}

elseif ($table === 'previous_conference_videos') {

    $query = DB::table('previous_conference_videos as v')
        ->leftJoin('conferences as c', 'c.id', '=', 'v.conference_id')
        ->select([
            'v.id',
            'v.year_id',
            'v.conference_id',
            'v.video_url',
            'v.created_date',
            DB::raw('COALESCE(c.conference_name, "") as conference_name'),
            DB::raw('COALESCE(c.short_name, "") as short_name'),
        ]);

    $availableYears = DB::table('previous_conference_videos')
        ->whereNotNull('year_id')
        ->distinct()
        ->orderBy('year_id', 'DESC')
        ->pluck('year_id');

    $yearId = null;
    if ($request->filled('year_id')) {
        $yearId = (int) $request->input('year_id');
    } elseif ($availableYears->isNotEmpty()) {
        $yearId = $availableYears->first(); // latest year
    }

    if ($yearId !== null) {
        $query->where('v.year_id', $yearId);
    } else {
        $query->where('v.status', 1); // fallback like CI3
    }

    $rows = $query->orderBy('v.id', 'ASC')->get()
        ->map(function ($row) {
            // Extract YouTube ID safely
            $vid = '';
            if (preg_match('%(?:youtube(?:-nocookie)?\.com/(?:[^/]+/.+/|(?:v|e(?:mbed)?)/|.*[?&]v=)|youtu\.be/)([^"&?/ ]{11})%i', $row->video_url, $matches)) {
                $vid = $matches[1];
            }
            return [
                'id' => $row->id,
                'year_id' => $row->year_id,
                'conference_id' => $row->conference_id,
                'video_url' => $row->video_url,
                'youtube_id' => $vid,
                'short_name' => $row->short_name,
                'created_date' => $row->created_date,
            ];
        });

    return response()->json($rows);
}
// In v4Index(): Add this block inside the big if/elseif chain
elseif ($table === 'c_reg_users') {
    // Type from route.meta → passed via ?type= in request (frontend sets it)
    $type = (int) $request->input('type', 3);

    $conferenceId = $request->input('conference_id');

    $query = DB::table('c_reg_users as u')  // 👈 real table name
        ->leftJoin('conferences as c', 'c.id', '=', 'u.conference_id')
        ->select([
            'u.id',
            'u.conference_id',
            'u.designation',
            'u.firstname',
            'u.lastname',
            'u.usr_email',
            'u.user_phone',
            'u.country_code',
            'u.usr_whatsapp',
            'u.user_type',
            'u.country',
            'u.total_amount',
            'u.total_extra_amount',
            'u.currency',
            'u.payment_status',
            'u.payment_method',
            'u.created_date',
            'c.folder_name as conference_folder',
            'c.conference_name',
        ])
        ->where('u.user_type', $type)  // ✅ filter by type
        ->when($conferenceId, fn($q) => $q->where('u.conference_id', $conferenceId))
        ->orderBy('u.id', 'DESC');

    $rows = $query->get()->map(function ($row) {
        $curr = match ((string)$row->currency) {
            '1' => '£', '3' => '€', '4' => 'د.إ', default => '$',
        };
        $paymentMethod = $row->payment_method == 1 ? 'Stripe' : 'PayPal';
        $userTypeLabels = [
            1 => 'Brochure Users', 2 => 'Abstract Users', 3 => 'Registered Users',
            4 => 'Discount Users', 6 => 'Online Registration',
            7 => 'Manual Payments Users', 8 => 'Group Registration Users',
        ];
        $userType = $userTypeLabels[$row->user_type] ?? '—';

        $qty = max(1, (int)($row->qty ?? 1));
        $amount = $row->user_type == 7
            ? $curr . ' ' . (($row->total_amount * $qty) + $row->total_extra_amount)
            : $curr . ' ' . $row->total_amount;

        return [
            'id' => $row->id,
            'conference' => $row->conference_folder ?: $row->conference_name ?: '—',
            'name' => trim("{$row->designation} {$row->firstname} {$row->lastname}"),
            'contact' => "{$row->usr_email}\n+{$row->country_code} {$row->user_phone}\nWhatsApp: +{$row->country_code} {$row->usr_whatsapp}",
            'user_type' => $userType,
            'country' => ucwords($row->country ?? ''),
            'amount' => $amount,
            'payment_status' => $row->payment_status,
            'payment_method' => $paymentMethod,
            'created_date' => $row->created_date ? date('d-M-Y H:i:s', strtotime($row->created_date)) : '—',
        ];
    });

    return response()->json($rows);
}
// ===== web_cms (Guidelines, etc. by page_type) =====
elseif ($table === 'web_cms') {
    $pageType = (int) $request->input('page_type', 5);

    $rows = DB::table('web_cms')
        ->select([
            'id',
            'content_title',
            'short_desc',
            'country',
            'cms_img',
            'long_desc',
            'page_type',
            'order_position',
            'status',
            'created_date',
        ])
        ->where('page_type', $pageType)
        ->where('status', 1)
        ->orderBy('order_position', 'ASC')
        ->orderBy('id', 'ASC')
        ->get();

    return response()->json($rows);
}

// ... existing blocks ...

elseif ($table === 'continents') {

    $rows = DB::table('continents')
        ->select([
            'id',
            'continent_name',
            'alias_name',
            'status',
            'created_date',
        ])
        ->where('status', 1)
        ->orderBy('id', 'ASC')
        ->get();

    return response()->json($rows);
}

elseif ($table === 'countries') {

    $query = DB::table('countries')
        ->select([
            'id',
            'continent_id',
            'country_name',
            'alias_name',
            'status',
            'created_date',
        ])
        ->where('status', 1);

    if ($request->filled('continent_id')) {
        $query->where('continent_id', (int) $request->input('continent_id'));
    }

    $rows = $query->orderBy('country_name', 'ASC')->get();

    return response()->json($rows);
}

elseif ($table === 'cities') {

    $query = DB::table('cities')
        ->select([
            'id',
            'continent_id',
            'country_id',
            'city_name',
            'alias_name',
            'status',
            'created_date',
        ])
        ->where('status', 1);

    if ($request->filled('continent_id')) {
        $query->where('continent_id', (int) $request->input('continent_id'));
    }
    if ($request->filled('country_id')) {
        $query->where('country_id', (int) $request->input('country_id'));
    }

    $rows = $query->orderBy('city_name', 'ASC')->get();

    return response()->json($rows);
}

// ... else fallback remains as you already have ...
elseif ($table === 'subjects') {

    $rows = DB::table('subjects')
        ->select([
            'id',
            'subject_title',
            'alias_title',
            'status',
            'created_date',
        ])
        ->orderBy('id', 'ASC')
        ->get();

    return response()->json($rows);
}
elseif ($table === 'abstract_categories') {

    $rows = DB::table('abstract_categories')
        ->select([
            'id',
            'category_name',
            'status',
            'created_date',
        ])
        ->orderBy('id', 'ASC')
        ->get();

    return response()->json($rows);
}
elseif ($table === 'categories') {

    $rows = DB::table('categories')
        ->select([
            'id',
            'category_name',
            'status',
            'created_date',
        ])
        ->orderBy('id', 'ASC')
        ->get();

    return response()->json($rows);
}
elseif ($table === 'categories_new') {

    $rows = DB::table('categories_new')
        ->select([
            'id',
            'category_name',
            'slug',
            'status',
            'created_date',
        ])
        ->orderBy('id', 'ASC')
        ->get();

    return response()->json($rows);
}

elseif ($table === 'c_registration_categories') {

    $pageType     = (int) $request->input('page_type', 1);   // 1 = Registration, 2 = Addon
    $conferenceId = (int) $request->input('conference_id', 0);

    $query = DB::table('c_registration_categories')
        ->select([
            'id',
            'conference_id',
            'page_type',
            'category_name',
            'order_position',
            'status',
            'created_date',
        ])
        ->where('page_type', $pageType);

    if ($conferenceId > 0) {
        $query->where('conference_id', $conferenceId);
    }

    $rows = $query
        ->orderBy('order_position', 'ASC')
        ->orderBy('id', 'ASC')
        ->get();

    return response()->json($rows);
}
elseif ($table === 'universities') {

    $rows = DB::table('universities as u')
        ->leftJoin('continents as cont', 'cont.id', '=', 'u.continent_id')
        ->leftJoin('countries as co', 'co.id', '=', 'u.country_id')
        ->leftJoin('cities as ci', 'ci.id', '=', 'u.city_id')
        ->select([
            'u.id',
            'u.university_name',
            'u.alias_name',
            'u.status',
            'u.continent_id',
            'u.country_id',
            'u.city_id',
            'u.created_date',
            DB::raw('COALESCE(cont.continent_name, "") as continent_name'),
            DB::raw('COALESCE(co.country_name, "") as country_name'),
            DB::raw('COALESCE(ci.city_name, "") as city_name'),
        ])
        ->orderBy('u.id', 'ASC')
        ->get();

    return response()->json($rows);
}
elseif ($table === 'hospitals') {

    $rows = DB::table('hospitals as h')
        ->leftJoin('continents as cont', 'cont.id', '=', 'h.continent_id')
        ->leftJoin('countries as co', 'co.id', '=', 'h.country_id')
        ->leftJoin('cities as ci', 'ci.id', '=', 'h.city_id')
        ->select([
            'h.id',
            'h.hospital_name',
            'h.alias_name',
            'h.status',
            'h.continent_id',
            'h.country_id',
            'h.city_id',
            'h.created_date',
            DB::raw('COALESCE(cont.continent_name, "") as continent_name'),
            DB::raw('COALESCE(co.country_name, "") as country_name'),
            DB::raw('COALESCE(ci.city_name, "") as city_name'),
        ])
        ->orderBy('h.id', 'ASC')
        ->get();

    return response()->json($rows);
}
elseif ($table === 'societies') {

    $rows = DB::table('societies as s')
        ->leftJoin('continents as cont', 'cont.id', '=', 's.continent_id')
        ->leftJoin('countries as co', 'co.id', '=', 's.country_id')
        ->leftJoin('cities as ci', 'ci.id', '=', 's.city_id')
        ->select([
            's.id',
            's.society_name',
            's.alias_name',
            's.status',
            's.continent_id',
            's.country_id',
            's.city_id',
            's.created_date',
            DB::raw('COALESCE(cont.continent_name, "") as continent_name'),
            DB::raw('COALESCE(co.country_name, "") as country_name'),
            DB::raw('COALESCE(ci.city_name, "") as city_name'),
        ])
        ->orderBy('s.id', 'ASC')
        ->get();

    return response()->json($rows);
}
// case block inside switch ($table)
// inside v4Index(Request $request)
elseif ($table === 'conferences') {

    $query = DB::table('conferences as c')
        ->select([
            'c.id',
            'c.conference_name',
            'c.short_name',
            'c.conference_location',
            'c.conference_url',
            'c.conference_img',
            'c.from_date',
            'c.to_date',
            'c.prev_conf',
            'c.publish_status',
            'c.conferece_type',
            'c.status',
        ])
        ->where('c.status', '<', 2);

    // Filter by upcoming / previous like CI3 $type param
    // type=1 → prev_conf=0 (upcoming)
    // type=2 → prev_conf=1 (previous)
    if ($request->filled('type')) {
        $type = (int) $request->input('type');
        if ($type === 1) {
            $query->where('c.prev_conf', 0);
        } elseif ($type === 2) {
            $query->where('c.prev_conf', 1);
        }
    }

    // ❌ DO NOT filter by conference_id here – there is no conference_id column
    // so skip that part entirely for conferences table

    $rows = $query->orderBy('c.id', 'DESC')->get();

    return response()->json($rows);
}


elseif ($table === 'c_past_conference_report') {

    $query = DB::table('c_past_conference_report');

    if ($request->filled('conference_id')) {
        $query->where('conference_id', (int) $request->input('conference_id'));
    }

    $query->orderBy('id', 'ASC');
}


        // ðŸ”š default fallback (like your v1 else)
        else {
            $query = DB::table($table)->orderBy('id', 'desc');
        }

        // ðŸ”¹ Optional filters (same pattern as v1Index)
     // existing generic filter at bottom
if ($request->has('conference_id') && $table !== 'conferences') {
    if ($table === 'c_invoices') {
        $query->where('inv.conference_id', (int) $request->conference_id);
    } elseif ($table === 'user_confs1') {
        $query->where('uc.conference_id', (int) $request->conference_id);
    } else {
        $query->where('conference_id', (int) $request->conference_id);
    }
}


        if ($table === 'user_confs1' && $request->filled('admin_id')) {
        }
if ($table === 'users' && $request->filled('usertype')) {
    $query->where('usertype', (int) $request->input('usertype'));
}

        return response()->json($query->get());
    }
    /*
    |--------------------------------------------------------------------------
    | v4Show â€“ same idea as v1Show
    |--------------------------------------------------------------------------
    */
    public function v4Show(Request $request, $table, $id)
    {
        $table = trim((string) $table);

        if ($table === '' || !preg_match('/^[a-zA-Z0-9_]+$/', $table) || !Schema::hasTable($table)) {
            return response()->json(['error' => 'Invalid table'], 400);
        }

        $id = (int) $id;
        if ($id <= 0) {
            return response()->json(['error' => 'Invalid id'], 400);
        }

        $row = DB::table($table)->find($id);
        if (!$row) {
            return response()->json(['error' => 'Not found'], 404);
        }

        return response()->json($row);
    }

    /*
    |--------------------------------------------------------------------------
    | v4Store â€“ same pattern as v1Store (example: user_confs1)
    |--------------------------------------------------------------------------
    */
    public function v4Store(Request $request, $table)
    {
        $table = trim((string) $table);

        if ($table === '' || !preg_match('/^[a-zA-Z0-9_]+$/', $table) || !Schema::hasTable($table)) {
            return response()->json(['error' => 'Invalid table'], 400);
        }

        try {
            switch ($table) {

                case 'user_confs1': {
                    $conferenceId = (int) $request->input('conference_id');
                    $adminId      = (int) $request->input('admin_id');

                    if (!$conferenceId) {
                        throw new \Exception("conference_id is required");
                    }
                    if (!$adminId) {
                        throw new \Exception("admin_id is required");
                    }

                    $conf = DB::table('conferences')->find($conferenceId);
                    if (!$conf) {
                        throw new \Exception("Invalid conference_id");
                    }

                    // active Super Admin only
                    $isSuper = DB::table('users')
                        ->where('id', $adminId)
                        ->where('usertype', 1)
                        ->where('status', 1)
                        ->exists();

                    if (!$isSuper) {
                        return response()->json([
                            'status'  => false,
                            'message' => 'Selected user is not an active Super Admin.',
                        ], 422);
                    }

                    // avoid duplicates
                    $dup = DB::table('user_confs1')
                        ->where('conference_id', $conferenceId)
                        ->where('admin_id', $adminId)
                        ->exists();

                    if ($dup) {
                        return response()->json([
                            'status'  => false,
                            'message' => 'This Super Admin is already assigned to this conference.',
                        ], 422);
                    }

                    $data = [
                        'conference_id' => $conferenceId,
                        'admin_id'      => $adminId,
                        'created_date'  => now(),
                    ];

                    $id = DB::table('user_confs1')->insertGetId($data);
                    $after = DB::table('user_confs1')->find($id);

                    // if you want audit, reuse your logAudit the same way as v1Store
                    try {
                        $this->logAudit(
                            'user_confs1',
                            $id,
                            'created',
                            [],
                            (array) $after
                        );
                    } catch (\Throwable $e) {
                        \Log::error("Audit log failed in v4Store [user_confs1/{$id}]: " . $e->getMessage());
                    }

                    return response()->json([
                        'status' => true,
                        'id'     => $id,
                        'data'   => $after,
                    ]);
                }
case 'users':
{
    // Super Admin Users (usertype=4)
    $validated = $request->validate([
        'firstname' => 'required|string|max:255',
        'email'     => 'required|email|max:255|unique:users,email',
        'password'  => 'required|string|min:4',
    ]);

    $now = now();

    $id = DB::table('users')->insertGetId([
        'conference_id' => null,
        'firstname'     => $validated['firstname'],
        'lastname'      => $request->input('lastname'), // optional
        'email'         => $validated['email'],
        'password'      => sha1($validated['password']),   // CI3-compatible
        'shw_pass'      => $validated['password'],         // plain text
        'status'        => 1,
        'emailAct'      => 1,
        'usertype'      => 4,                              // SUPER ADMIN
        'regDate'       => $now,
    ]);

    return response()->json([
        'status' => true,
        'id'     => $id,
    ]);
}
 case 'users_admins':
            $data = $request->validate([
                'firstname' => ['required','string','max:255'],
                'email'     => ['required','email','max:255','unique:users,email'],
                'password'  => ['required','string','max:255'],
            ]);

            $id = DB::table('users')->insertGetId([
                'conference_id' => null,
                'firstname'     => $data['firstname'],
                'lastname'      => $request->input('lastname'), // if you later add it
                'email'         => $data['email'],
                // ✅ keep CI3 sha1 to not break logins
                'password'      => sha1($data['password']),
                'shw_pass'      => $data['password'],
                'status'        => 1,
                'emailAct'      => 1,
                'usertype'      => 2,
                'regDate'       => now(),
            ]);

            return response()->json([
                'status' => true,
                'id'     => $id,
            ]);
            
            case 'meta_data':
    $validated = $request->validate([
        'meta_title'    => 'required|string|max:500',
        'meta_desc'     => 'required|string',
        'meta_keywords' => 'required|string|max:500',
        'page_type'     => 'required|integer',
    ]);

    $id = DB::table('meta_data')->insertGetId($validated);
    return response()->json(['status'=>true,'id'=>$id]);
case 'previous_conference_gallery': {
    // validation
    $rules = [
        'conference_id' => 'required|integer|exists:conferences,id',
        'gallery_title' => 'nullable|string|max:255',
        'gallery_img'   => 'required|file|mimes:webp|max:4096', // 4MB, WEBP only
        'parent_id'     => 'nullable|integer|min:0',
    ];

    $validated = $request->validate($rules);

    $conferenceId = (int) $validated['conference_id'];
    $parentId     = (int) ($validated['parent_id'] ?? 0);

    // lookup conference_year like old CI3
    $conf = DB::table('conferences')->where('id', $conferenceId)->first();
    $yearId = $conf ? ($conf->conference_year ?? null) : null;

    // handle file upload
    $file = $request->file('gallery_img');
    $filename = null;

    if ($file && $file->isValid()) {
        // you can keep same folder as CI3: /uploads/gallery
        $filename = time() . '_' . uniqid() . '.webp';
        $file->move(public_path('uploads/gallery'), $filename);
    }

    $payload = [
        'parent_id'     => $parentId,
        'conference_id' => $conferenceId,
        'year_id'       => $yearId,
        'gallery_title' => $validated['gallery_title'] ?? null,
        'gallery_img'   => $filename,
        'created_date'  => now(),
        'status'        => 1,
    ];

    $id = DB::table('previous_conference_gallery')->insertGetId($payload);

    try {
        $this->logAudit(
            $table,
            $id,
            'created',
            [],
            $payload
        );
    } catch (\Throwable $e) {
        \Log::error("Audit log failed in v1Store [{$table}/{$id}]: " . $e->getMessage());
    }

    return response()->json([
        'status' => true,
        'id'     => $id,
        'data'   => $payload,
    ]);
}
case 'previous_conference_videos': {
    $validated = $request->validate([
        'conference_id' => 'required|integer|exists:conferences,id',
        'video_url'     => 'required|url',
    ]);

    $conf = DB::table('conferences')->find($validated['conference_id']);
    $yearId = $conf ? $conf->conference_year : null;

    $id = DB::table('previous_conference_videos')->insertGetId([
        'year_id'       => $yearId,
        'conference_id' => $validated['conference_id'],
        'video_url'     => $validated['video_url'],
        'status'        => 1,
        'created_date'  => now(),
    ]);

    return response()->json(['status' => true, 'id' => $id]);
}
case 'web_cms': {
    $pageType = (int) $request->input('page_type', 5);

    // 🔹 Advisory Board (page_type = 12) – with image
    if ($pageType === 12) {
        $validated = $request->validate([
            'content_title' => 'required|string|max:255',
            'short_desc'    => 'required|string|max:255',
            'country'       => 'required|string|max:255',
            'long_desc'     => 'nullable|string',
            'mainImage'     => 'required|file|mimes:webp|max:4096',
        ]);

        $file = $request->file('mainImage');
        $filename = null;

        if ($file && $file->isValid()) {
            $filename = time() . '_' . uniqid() . '.webp';
            $file->move(public_path('uploads/team'), $filename);
        }

        $nextOrder = (int) DB::table('web_cms')
            ->where('page_type', 12)
            ->max('order_position');
        $nextOrder = $nextOrder + 1;

        $data = [
            'content_title' => $validated['content_title'],
            'sub_title'     => Str::slug($validated['content_title']),
            'short_desc'    => $validated['short_desc'],
            'country'       => $validated['country'],
            'long_desc'     => $request->input('long_desc'),
            'cms_img'       => $filename,
            'page_type'     => 12,
            'order_position'=> $nextOrder,
            'status'        => 1,
            'created_date'  => now(),
        ];

        $id = DB::table('web_cms')->insertGetId($data);

        return response()->json([
            'status' => true,
            'id'     => $id,
        ]);
    }

    $validated = $request->validate([
        'content_title' => 'required|string|max:500',
        'long_desc'     => 'required|string',
        'page_type'     => 'required|integer',
    ]);

    $data = [
        'content_title' => $validated['content_title'],
        'sub_title'     => Str::slug($validated['content_title']),
        'long_desc'     => $validated['long_desc'],
        'page_type'     => $validated['page_type'],
        'status'        => 1,
        'created_date'  => now(),
    ];

    $id = DB::table('web_cms')->insertGetId($data);

    return response()->json(['status' => true, 'id' => $id]);
}

// ... existing cases ...

case 'continents': {
    $validated = $request->validate([
        'continent_name' => 'required|string|max:255',
    ]);

    $name  = $validated['continent_name'];
    $alias = Str::slug($name);

    $data = [
        'continent_name' => $name,
        'alias_name'     => $alias,
        'status'         => 1,
        'created_date'   => now(),
    ];

    $id = DB::table('continents')->insertGetId($data);

    return response()->json([
        'status' => true,
        'id'     => $id,
    ]);
}

case 'countries': {
    $validated = $request->validate([
        'continent_id' => 'required|integer|min:1',
        'country_name' => 'required|string|max:255',
    ]);

    $name  = $validated['country_name'];
    $alias = Str::slug($name);

    $data = [
        'continent_id'  => (int) $validated['continent_id'],
        'country_name'  => $name,
        'alias_name'    => $alias,
        'status'        => 1,
        'created_date'  => now(),
    ];

    $id = DB::table('countries')->insertGetId($data);

    return response()->json([
        'status' => true,
        'id'     => $id,
    ]);
}

case 'cities': {
    $validated = $request->validate([
        'continent_id' => 'required|integer|min:1',
        'country_id'   => 'required|integer|min:1',
        'city_name'    => 'required|string|max:255',
    ]);

    $name  = $validated['city_name'];
    $alias = Str::slug($name);

    $data = [
        'continent_id' => (int) $validated['continent_id'],
        'country_id'   => (int) $validated['country_id'],
        'city_name'    => $name,
        'alias_name'   => $alias,
        'status'       => 1,
        'created_date' => now(),
    ];

    $id = DB::table('cities')->insertGetId($data);

    return response()->json([
        'status' => true,
        'id'     => $id,
    ]);
}

// ... default ...

case 'subjects': {

    $validated = $request->validate([
        'subject_title' => 'required|string|max:255',
        'subj_desc'     => 'nullable|string',
     
    ]);

    $subjectTitle = $validated['subject_title'];
    $alias        = Str::slug($subjectTitle, '-');

    $data = [
        'subject_title'        => $subjectTitle,
        'alias_title'          => $alias,
        'subj_desc'            => $validated['subj_desc'] ?? null,
      
        'status'               => 1,
        'created_date'         => now(),
    ];

    $id = DB::table('subjects')->insertGetId($data);

    // optional audit – same pattern as other cases
    try {
        $this->logAudit(
            'subjects',
            $id,
            'created',
            [],
            $data
        );
    } catch (\Throwable $e) {
        \Log::error("Audit log failed in v4Store [subjects/{$id}]: " . $e->getMessage());
    }

    return response()->json([
        'status' => true,
        'id'     => $id,
        'data'   => $data,
    ]);
}
case 'abstract_categories': {

    $validated = $request->validate([
        'category_name' => 'required|string|max:255',
    ]);

    $data = [
        'category_name' => $validated['category_name'],
        'status'        => 1,
        'created_date'  => now(),
    ];

    $id = DB::table('abstract_categories')->insertGetId($data);

    // optional audit
    try {
        $this->logAudit(
            'abstract_categories',
            $id,
            'created',
            [],
            $data
        );
    } catch (\Throwable $e) {
        \Log::error("Audit log failed in v4Store [abstract_categories/{$id}]: " . $e->getMessage());
    }

    return response()->json([
        'status' => true,
        'id'     => $id,
        'data'   => $data,
    ]);
}
case 'categories': {

    $validated = $request->validate([
        'category_name' => 'required|string|max:255',
    ]);

    $data = [
        'category_name' => $validated['category_name'],
        'status'        => 1,
        'created_date'  => now(),
    ];

    $id = DB::table('categories')->insertGetId($data);

    try {
        $this->logAudit('categories', $id, 'created', [], $data);
    } catch (\Throwable $e) {
        \Log::error("Audit log failed in v4Store [categories/{$id}]: " . $e->getMessage());
    }

    return response()->json([
        'status' => true,
        'id'     => $id,
        'data'   => $data,
    ]);
}

case 'categories_new': {
    $validated = $request->validate([
        'category_name' => 'required|string|max:255',
        'slug'          => 'required|string|max:255',
    ]);

    $data = [
        'category_name' => $validated['category_name'],
        'slug'          => $validated['slug'],
        'status'        => 1,
        'created_date'  => now(),
    ];

    $id = DB::table('categories_new')->insertGetId($data);

    // (optional) audit log
    try {
        $this->logAudit('categories_new', $id, 'created', [], $data);
    } catch (\Throwable $e) {
        \Log::error("Audit log failed in v4Store [categories_new/{$id}]: " . $e->getMessage());
    }

    return response()->json([
        'status' => true,
        'id'     => $id,
    ]);
}
case 'c_registration_categories': {
    $validated = $request->validate([
        'conference_id' => 'required|integer|min:1',
        'page_type'     => 'required|integer',      // 1 = Registration, 2 = Addon
        'category_name' => 'required|string|max:255',
    ]);

    $conferenceId = (int) $validated['conference_id'];
    $pageType     = (int) $validated['page_type'];

    // next order_position for this conference + page_type
    $maxOrder = DB::table('c_registration_categories')
        ->where('conference_id', $conferenceId)
        ->where('page_type', $pageType)
        ->max('order_position');

    $nextOrder = ((int) $maxOrder) + 1;

    $data = [
        'conference_id' => $conferenceId,
        'page_type'     => $pageType,
        'category_name' => $validated['category_name'],
        'order_position'=> $nextOrder,
        'status'        => 1,
        'created_date'  => now(),
    ];

    $id = DB::table('c_registration_categories')->insertGetId($data);

    try {
        $this->logAudit('c_registration_categories', $id, 'created', [], $data);
    } catch (\Throwable $e) {
        \Log::error("Audit log failed in v4Store [c_registration_categories/{$id}]: " . $e->getMessage());
    }

    return response()->json([
        'status' => true,
        'id'     => $id,
    ]);
}
case 'universities': {
    $validated = $request->validate([
        'continent_id'    => 'required|integer|min:1',
        'country_id'      => 'required|integer|min:1',
        'city_id'         => 'required|integer|min:1',
        'university_name' => 'required|string|max:255',
    ]);

    $name  = $validated['university_name'];
    $alias = Str::slug($name);

    $data = [
        'continent_id'    => (int) $validated['continent_id'],
        'country_id'      => (int) $validated['country_id'],
        'city_id'         => (int) $validated['city_id'],
        'university_name' => $name,
        'alias_name'      => $alias,
        'status'          => 1,
        'created_date'    => now(),
    ];

    $id = DB::table('universities')->insertGetId($data);

    return response()->json([
        'status' => true,
        'id'     => $id,
        'data'   => $data,
    ]);
}
case 'hospitals': {
    $validated = $request->validate([
        'continent_id'  => 'required|integer|min:1',
        'country_id'    => 'required|integer|min:1',
        'city_id'       => 'required|integer|min:1',
        'hospital_name' => 'required|string|max:255',
    ]);

    $name  = $validated['hospital_name'];
    $alias = Str::slug($name);

    $data = [
        'continent_id'  => (int) $validated['continent_id'],
        'country_id'    => (int) $validated['country_id'],
        'city_id'       => (int) $validated['city_id'],
        'hospital_name' => $name,
        'alias_name'    => $alias,
        'status'        => 1,
        'created_date'  => now(),
    ];

    $id = DB::table('hospitals')->insertGetId($data);

    return response()->json([
        'status' => true,
        'id'     => $id,
        'data'   => $data,
    ]);
}
case 'societies': {
    $validated = $request->validate([
        'continent_id'  => 'required|integer|min:1',
        'country_id'    => 'required|integer|min:1',
        'city_id'       => 'required|integer|min:1',
        'society_name'  => 'required|string|max:255',
    ]);

    $name  = $validated['society_name'];
    $alias = Str::slug($name);

    $data = [
        'continent_id'  => (int) $validated['continent_id'],
        'country_id'    => (int) $validated['country_id'],
        'city_id'       => (int) $validated['city_id'],
        'society_name'  => $name,
        'alias_name'    => $alias,
        'status'        => 1,
        'created_date'  => now(),
    ];

    $id = DB::table('societies')->insertGetId($data);

    return response()->json([
        'status' => true,
        'id'     => $id,
        'data'   => $data,
    ]);
}
case 'conferences': {

    $validated = $request->validate([
        'continent_id'           => 'required|integer',
        'country_id'             => 'required|integer',
        'city_id'                => 'required|integer',
        'conference_first_title' => 'required|string|max:255',
        'conference_name'        => 'required|string|max:255',
        'short_name'             => 'required|string|max:255',
        'conference_theme'       => 'required|string|max:255',
        'conferece_type'         => 'required|in:1,2',       // 1 = Folder, 2 = Domain
        'folder_name'            => 'required|string|max:255',
        'from_date'              => 'required|date',
        'to_date'                => 'required|date',
        'conference_location'    => 'required|string|max:255',
        'subject_id'             => 'required|array|min:1',
        'subject_id.*'           => 'integer',
        'gkey'                   => 'nullable|string',
        'gse'                    => 'nullable|string',
        'mainImage'              => 'nullable|file|mimes:jpg,jpeg,png,webp',
    ]);

    $subjectIds  = $validated['subject_id'];
    $msubjectId  = $subjectIds[0] ?? null;
    $fromDate    = date('Y-m-d', strtotime($validated['from_date']));
    $toDate      = date('Y-m-d', strtotime($validated['to_date']));
    $confYear    = date('Y', strtotime($validated['from_date']));
    $shortAlias  = Str::slug($validated['short_name'], '-');

    // Banner image upload (same uploads path as CI3)
    $bannerImg = null;
    if ($request->hasFile('mainImage')) {
        $file      = $request->file('mainImage');
        $ext       = $file->getClientOriginalExtension();
        $bannerImg = time() . '.' . $ext;
        $file->move(public_path('uploads/conferences'), $bannerImg);
    }

    // Folder + URL logic (mirror CI3)
    if ((int) $validated['conferece_type'] === 1) {
        // Folder type – clean and use as folder_name
        $folderName   = $this->clean($validated['folder_name']);
        $conferenceUrl = 'https://scisynopsisconferences.com/' . $folderName . '/';
    } else {
        // Domain type – store raw & prepend https:// for URL
        $folderName   = $validated['folder_name'];
        $conferenceUrl = 'https://' . trim($folderName, '/') . '/';
    }

    // Insert main conference
    $id = DB::table('conferences')->insertGetId([
        'continent_id'           => (int) $validated['continent_id'],
        'country_id'             => (int) $validated['country_id'],
        'city_id'                => (int) $validated['city_id'],
        'folder_name'            => $folderName,
        'gkey'                   => $validated['gkey'] ?? '',
        'gse'                    => $validated['gse'] ?? '',
        'conference_first_title' => $validated['conference_first_title'],
        'conference_name'        => $validated['conference_name'],
        'short_name'             => $validated['short_name'],
        'short_alias_name'       => $shortAlias,
        'conference_theme'       => $validated['conference_theme'],
        'conference_year'        => $confYear,
        'from_date'              => $fromDate,
        'to_date'                => $toDate,
        'conference_location'    => $validated['conference_location'],
        'conferece_type'         => (int) $validated['conferece_type'],
        'conference_url'         => $conferenceUrl,
        'files_dump_status'      => 0,
        'conference_img'         => $bannerImg ?? '',
        'created_date'           => now(),
        'msubject_id'            => $msubjectId,
        'prev_conf'              => 0,
    ]);

    // unique_id like CI3: foldername_id
    $uploadsFolder = $folderName . '_' . $id;
    DB::table('conferences')->where('id', $id)->update([
        'unique_id' => $uploadsFolder,
    ]);

    // Insert conference_categories
    foreach ($subjectIds as $sid) {
        DB::table('conference_categories')->insert([
            'conference_id' => $id,
            'subject_id'    => (int) $sid,
            'status'        => 1,
            'created_date'  => now(),
        ]);
    }

    // Create physical uploads folders (mirror CI3 environment logic)
    $this->createConferenceFolders($uploadsFolder);

    // Optional audit
    try {
        $after = DB::table('conferences')->find($id);
        $this->logAudit('conferences', $id, 'created', [], (array) $after);
    } catch (\Throwable $e) {
        \Log::error("Audit log failed in v4Store [conferences/{$id}]: " . $e->getMessage());
    }

    return response()->json([
        'status' => true,
        'id'     => $id,
    ]);
}
case 'c_past_conference_report': {

    $validated = $request->validate([
        'conference_id' => 'required|integer|exists:conferences,id',
        'conf_title'    => 'required|string|max:255',
        'conf_desc'     => 'nullable|string',
    ]);

    $payload = [
        'conference_id' => (int) $validated['conference_id'],
        'conf_title'    => $validated['conf_title'],
        'conf_desc'     => $validated['conf_desc'] ?? null,
        'status'        => 1,
        'created_date'  => now(),
    ];

    $id = DB::table('c_past_conference_report')->insertGetId($payload);

    try {
        $this->logAudit('c_past_conference_report', $id, 'created', [], $payload);
    } catch (\Throwable $e) {
        \Log::error("Audit log failed in v4Store [c_past_conference_report/{$id}]: " . $e->getMessage());
    }

    return response()->json([
        'status' => true,
        'id'     => $id,
    ]);
}

                default:
                    return response()->json(['error' => 'Not supported'], 400);
            }

        } catch (\Throwable $e) {
            \Log::error("v4Store [{$table}]: " . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
            ]);

            return response()->json([
                'status'  => false,
                'message' => $e->getMessage(),
            ], 422);
        }
    }

    /*
    |--------------------------------------------------------------------------
    | v4Update â€“ same pattern as your v1Update (with guards + audit)
    |--------------------------------------------------------------------------
    */
    public function v4Update(Request $request, $table, $id)
    {
        $table = trim((string) $table);

        if ($table === '' || !preg_match('/^[a-zA-Z0-9_]+$/', $table) || !Schema::hasTable($table)) {
            return response()->json(['error' => 'Invalid table'], 400);
        }

        $id = (int) $id;
        if ($id <= 0) {
            return response()->json(['error' => 'Invalid id'], 400);
        }

        $existing = DB::table($table)->find($id);
        if (!$existing) {
            return response()->json(['error' => 'Not found'], 404);
        }

        try {
            $logUpdate = function (string $tableName, int $rowId, $beforeRow) {
                if (!$beforeRow) return;

                $afterRow = DB::table($tableName)->find($rowId);
                if (!$afterRow) return;

                try {
                    $this->logAudit(
                        $tableName,
                        $rowId,
                        'updated',
                        (array) $beforeRow,
                        (array) $afterRow
                    );
                } catch (\Throwable $e) {
                    \Log::error("Audit log failed in v4Update [{$tableName}/{$rowId}]: " . $e->getMessage());
                }
            };

            switch ($table) {

                case 'user_confs1': {
                    $before = $existing;

                    $conferenceId = (int) $request->input('conference_id', $existing->conference_id);
                    $adminId      = (int) $request->input('admin_id', $existing->admin_id);

                    if (!$conferenceId) {
                        throw new \Exception("conference_id is required");
                    }
                    if (!$adminId) {
                        throw new \Exception("admin_id is required");
                    }

                    $conf = DB::table('conferences')->find($conferenceId);
                    if (!$conf) {
                        throw new \Exception("Invalid conference_id");
                    }

                    $isSuper = DB::table('users')
                        ->where('id', $adminId)
                        ->where('usertype', 1)
                        ->where('status', 1)
                        ->exists();

                    if (!$isSuper) {
                        return response()->json([
                            'status'  => false,
                            'message' => 'Selected user is not an active Super Admin.',
                        ], 422);
                    }

                    $dup = DB::table('user_confs1')
                        ->where('conference_id', $conferenceId)
                        ->where('admin_id', $adminId)
                        ->where('id', '!=', $id)
                        ->exists();

                    if ($dup) {
                        return response()->json([
                            'status'  => false,
                            'message' => 'This Super Admin is already assigned to this conference.',
                        ], 422);
                    }

                    $data = [
                        'conference_id' => $conferenceId,
                        'admin_id'      => $adminId,
                    ];

                    DB::table('user_confs1')->where('id', $id)->update($data);
                    $logUpdate('user_confs1', $id, $before);

                    return response()->json([
                        'status'  => true,
                        'message' => 'Assigned Super Admin updated successfully',
                    ]);
                }


case 'users':
{
    $user = DB::table('users')->find($id);
    if (!$user) {
        return response()->json(['status' => false, 'message' => 'User not found'], 404);
    }

    $validated = $request->validate([
        'firstname' => 'required|string|max:255',
        'email'     => 'required|email|max:255|unique:users,email,' . $id,
        // password is optional on edit
    ]);

    $update = [
        'conference_id' => null,
        'firstname'     => $validated['firstname'],
        'lastname'      => $request->input('lastname'),
        'email'         => $validated['email'],
        'status'        => 1,
        'emailAct'      => 1,
        'usertype'      => 4,
    ];

    if ($request->filled('password')) {
        $update['password'] = sha1($request->input('password'));
        $update['shw_pass'] = $request->input('password');
    }

    DB::table('users')->where('id', $id)->update($update);

    return response()->json(['status' => true]);
}
case 'users_admins':
            $data = $request->validate([
                'firstname' => ['required','string','max:255'],
                'email'     => [
                    'required',
                    'email',
                    'max:255',
                    Rule::unique('users', 'email')->ignore($id),
                ],
                'password'  => ['nullable','string','max:255'],
            ]);

            $update = [
                'conference_id' => null,
                'firstname'     => $data['firstname'],
                'lastname'      => $request->input('lastname'),
                'email'         => $data['email'],
                'status'        => 1,
                'emailAct'      => 1,
                'usertype'      => 2,
                'regDate'       => now(), // same as CI3 (they re-set regDate)
            ];

            if (!empty($data['password'])) {
                $update['password'] = sha1($data['password']);
                $update['shw_pass'] = $data['password'];
            }

            DB::table('users')->where('id', $id)->update($update);

            return response()->json(['status' => true]);
            case 'meta_data':
    $validated = $request->validate([
        'meta_title'    => 'required|string|max:500',
        'meta_desc'     => 'required|string',
        'meta_keywords' => 'required|string|max:500',
        // page_type EDIT optional
    ]);

    DB::table('meta_data')->where('id',$id)->update($validated);
    return response()->json(['status'=>true]);
case 'previous_conference_gallery': {
    $id = (int) $id;

    $rules = [
        'conference_id' => 'required|integer|exists:conferences,id',
        'gallery_title' => 'nullable|string|max:255',
        'gallery_img'   => 'nullable|file|mimes:webp|max:4096',
        'parent_id'     => 'nullable|integer|min:0',
        'hidden_gallery_img' => 'nullable|string', // from your Vue hidden field
    ];

    $validated = $request->validate($rules);

    $before = DB::table('previous_conference_gallery')->where('id', $id)->first();
    if (!$before) {
        return response()->json(['error' => 'Record not found'], 404);
    }

    $conferenceId = (int) $validated['conference_id'];
    $parentId     = (int) ($validated['parent_id'] ?? 0);

    $conf = DB::table('conferences')->where('id', $conferenceId)->first();
    $yearId = $conf ? ($conf->conference_year ?? null) : null;

    $filename = $validated['hidden_gallery_img'] ?? $before->gallery_img;

    // if new file uploaded → replace old
    if ($request->hasFile('gallery_img')) {
        $file = $request->file('gallery_img');

        if ($file && $file->isValid()) {
            // delete old file
            if ($filename && file_exists(public_path('uploads/gallery/' . $filename))) {
                @unlink(public_path('uploads/gallery/' . $filename));
            }

            $filename = time() . '_' . uniqid() . '.webp';
            $file->move(public_path('uploads/gallery'), $filename);
        }
    }

    $payload = [
        'parent_id'     => $parentId,
        'conference_id' => $conferenceId,
        'year_id'       => $yearId,
        'gallery_title' => $validated['gallery_title'] ?? null,
        'gallery_img'   => $filename,
    ];

    DB::table('previous_conference_gallery')
        ->where('id', $id)
        ->update($payload);

    try {
        $this->logAudit(
            $table,
            $id,
            'updated',
            (array) $before,
            $payload
        );
    } catch (\Throwable $e) {
        \Log::error("Audit log failed in v1Update [{$table}/{$id}]: " . $e->getMessage());
    }

    return response()->json([
        'status' => true,
        'id'     => $id,
        'data'   => $payload,
    ]);
}

case 'previous_conference_videos': {
    $before = DB::table('previous_conference_videos')->find($id);
    if (!$before) return response()->json(['error' => 'Not found'], 404);

    $validated = $request->validate([
        'conference_id' => 'required|integer|exists:conferences,id',
        'video_url'     => 'required|url',
    ]);

    $conf = DB::table('conferences')->find($validated['conference_id']);
    $yearId = $conf ? $conf->conference_year : null;

    DB::table('previous_conference_videos')->where('id', $id)->update([
        'year_id'       => $yearId,
        'conference_id' => $validated['conference_id'],
        'video_url'     => $validated['video_url'],
    ]);

    return response()->json(['status' => true]);
}

case 'web_cms': {
    $pageType = (int) ($request->input('page_type') ?? $existing->page_type ?? 5);

    // Advisory Board edit
    if ($pageType === 12) {
        $validated = $request->validate([
            'content_title'      => 'required|string|max:255',
            'short_desc'         => 'required|string|max:255',
            'country'            => 'required|string|max:255',
            'long_desc'          => 'nullable|string',
            'mainImage'          => 'nullable|file|mimes:webp|max:4096',
            'hiddenmainImage'    => 'nullable|string',
        ]);

        $before = $existing;

        $filename = $validated['hiddenmainImage'] ?? $existing->cms_img;

        if ($request->hasFile('mainImage')) {
            $file = $request->file('mainImage');
            if ($file && $file->isValid()) {
                // delete old
                if ($filename && file_exists(public_path('uploads/team/' . $filename))) {
                    @unlink(public_path('uploads/team/' . $filename));
                }
                $filename = time() . '_' . uniqid() . '.webp';
                $file->move(public_path('uploads/team'), $filename);
            }
        }

        $payload = [
            'content_title' => $validated['content_title'],
            'sub_title'     => Str::slug($validated['content_title']),
            'short_desc'    => $validated['short_desc'],
            'country'       => $validated['country'],
            'long_desc'     => $request->input('long_desc'),
            'cms_img'       => $filename,
        ];

        DB::table('web_cms')->where('id', $id)->update($payload);

        return response()->json(['status' => true]);
    }

    // Guidelines edit (no image)
    $validated = $request->validate([
        'content_title' => 'required|string|max:500',
        'long_desc'     => 'required|string',
    ]);

    $payload = [
        'content_title' => $validated['content_title'],
        'sub_title'     => Str::slug($validated['content_title']),
        'long_desc'     => $validated['long_desc'],
    ];

    DB::table('web_cms')->where('id', $id)->update($payload);

    return response()->json(['status' => true]);
}

// ... existing cases ...

case 'continents': {
    $validated = $request->validate([
        'continent_name' => 'required|string|max:255',
    ]);

    $name  = $validated['continent_name'];
    $alias = Str::slug($name);

    DB::table('continents')
        ->where('id', $id)
        ->update([
            'continent_name' => $name,
            'alias_name'     => $alias,
        ]);

    return response()->json(['status' => true]);
}

case 'countries': {
    $validated = $request->validate([
        'country_name' => 'required|string|max:255',
    ]);

    $name  = $validated['country_name'];
    $alias = Str::slug($name);

    $update = [
        'country_name' => $name,
        'alias_name'   => $alias,
    ];

    // optionally allow moving to another continent via edit:
    if ($request->filled('continent_id')) {
        $update['continent_id'] = (int) $request->input('continent_id');
    }

    DB::table('countries')->where('id', $id)->update($update);

    return response()->json(['status' => true]);
}

case 'cities': {
    $validated = $request->validate([
        'city_name' => 'required|string|max:255',
    ]);

    $name  = $validated['city_name'];
    $alias = Str::slug($name);

    $update = [
        'city_name'  => $name,
        'alias_name' => $alias,
    ];

    DB::table('cities')->where('id', $id)->update($update);

    return response()->json(['status' => true]);
}

// ... default ...

case 'subjects': {

    $before = $existing;

    $validated = $request->validate([
        'subject_title' => 'required|string|max:255',
        'subj_desc'     => 'nullable|string',

    ]);

    $subjectTitle = $validated['subject_title'];
    $alias        = Str::slug($subjectTitle, '-');

    $payload = [
        'subject_title'        => $subjectTitle,
        'alias_title'          => $alias,
        'subj_desc'            => $validated['subj_desc'] ?? null,
        
    ];

    DB::table('subjects')
        ->where('id', $id)
        ->update($payload);

    // use your $logUpdate helper closure
    $logUpdate('subjects', $id, $before);

    return response()->json([
        'status'  => true,
        'message' => 'Subject updated successfully',
    ]);
}
case 'abstract_categories': {

    $before = $existing;

    $validated = $request->validate([
        'category_name' => 'required|string|max:255',
    ]);

    $payload = [
        'category_name' => $validated['category_name'],
    ];

    DB::table('abstract_categories')
        ->where('id', $id)
        ->update($payload);

    // use the $logUpdate closure already defined in v4Update
    $logUpdate('abstract_categories', $id, $before);

    return response()->json([
        'status'  => true,
        'message' => 'Category updated successfully',
    ]);
}
case 'categories': {

    $before = $existing;

    $validated = $request->validate([
        'category_name' => 'required|string|max:255',
    ]);

    $payload = [
        'category_name' => $validated['category_name'],
    ];

    DB::table('categories')
        ->where('id', $id)
        ->update($payload);

    $logUpdate('categories', $id, $before);

    return response()->json([
        'status'  => true,
        'message' => 'Category updated successfully',
    ]);
}
case 'categories_new': {
    $before = $existing;

    $validated = $request->validate([
        'category_name' => 'required|string|max:255',
        'slug'          => 'required|string|max:255',
    ]);

    $update = [
        'category_name' => $validated['category_name'],
        'slug'          => $validated['slug'],
    ];

    DB::table('categories_new')->where('id', $id)->update($update);

    // (optional) audit
    try {
        $this->logAudit(
            'categories_new',
            $id,
            'updated',
            (array) $before,
            $update
        );
    } catch (\Throwable $e) {
        \Log::error("Audit log failed in v4Update [categories_new/{$id}]: " . $e->getMessage());
    }

    return response()->json(['status' => true]);
}
case 'c_registration_categories': {
    $before = $existing;

    $validated = $request->validate([
        'category_name' => 'required|string|max:255',
        // page_type/order_position unchanged by this edit (same as CI3)
    ]);

    $update = [
        'category_name' => $validated['category_name'],
    ];

    DB::table('c_registration_categories')->where('id', $id)->update($update);

    try {
        $this->logAudit(
            'c_registration_categories',
            $id,
            'updated',
            (array) $before,
            $update
        );
    } catch (\Throwable $e) {
        \Log::error("Audit log failed in v4Update [c_registration_categories/{$id}]: " . $e->getMessage());
    }

    return response()->json(['status' => true]);
}
case 'universities': {
    $validated = $request->validate([
        'university_name' => 'required|string|max:255',
        // if later you want to edit continent/country/city, add them here
    ]);

    $name  = $validated['university_name'];
    $alias = Str::slug($name);

    $update = [
        'university_name' => $name,
        'alias_name'      => $alias,
    ];

    DB::table('universities')
        ->where('id', $id)
        ->update($update);

    return response()->json(['status' => true]);
}
case 'hospitals': {
    $validated = $request->validate([
        'hospital_name' => 'required|string|max:255',

        // optional: allow changing location on edit
        'continent_id'  => 'nullable|integer|min:1',
        'country_id'    => 'nullable|integer|min:1',
        'city_id'       => 'nullable|integer|min:1',
    ]);

    $update = [
        'hospital_name' => $validated['hospital_name'],
        'alias_name'    => Str::slug($validated['hospital_name']),
    ];

    if ($request->filled('continent_id')) {
        $update['continent_id'] = (int) $validated['continent_id'];
    }
    if ($request->filled('country_id')) {
        $update['country_id'] = (int) $validated['country_id'];
    }
    if ($request->filled('city_id')) {
        $update['city_id'] = (int) $validated['city_id'];
    }

    DB::table('hospitals')->where('id', $id)->update($update);

    return response()->json(['status' => true]);
}
case 'societies': {
    $validated = $request->validate([
        'society_name' => 'required|string|max:255',
        'continent_id' => 'nullable|integer|min:1',
        'country_id'   => 'nullable|integer|min:1',
        'city_id'      => 'nullable|integer|min:1',
    ]);

    $update = [
        'society_name' => $validated['society_name'],
        'alias_name'   => Str::slug($validated['society_name']),
    ];

    if ($request->filled('continent_id')) {
        $update['continent_id'] = (int) $validated['continent_id'];
    }
    if ($request->filled('country_id')) {
        $update['country_id'] = (int) $validated['country_id'];
    }
    if ($request->filled('city_id')) {
        $update['city_id'] = (int) $validated['city_id'];
    }

    DB::table('societies')->where('id', $id)->update($update);

    return response()->json(['status' => true]);
}

case 'conferences': {

    $before = $existing;

    $validated = $request->validate([
        'continent_id'           => 'required|integer',
        'country_id'             => 'required|integer',
        'city_id'                => 'required|integer',
        'conference_first_title' => 'required|string|max:255',
        'conference_name'        => 'required|string|max:255',
        'short_name'             => 'required|string|max:255',
        'conference_theme'       => 'required|string|max:255',
        'conferece_type'         => 'required|in:1,2',
        'folder_name'            => 'required|string|max:255',
        'from_date'              => 'required|date',
        'to_date'                => 'required|date',
        'conference_location'    => 'required|string|max:255',
        'subject_id'             => 'required|array|min:1',
        'subject_id.*'           => 'integer',
        'gkey'                   => 'nullable|string',
        'gse'                    => 'nullable|string',
        'mainImage'              => 'nullable|file|mimes:jpg,jpeg,png,webp',
        'hiddenmainImage'        => 'nullable|string',
    ]);

    $subjectIds  = $validated['subject_id'];
    $msubjectId  = $subjectIds[0] ?? null;
    $fromDate    = date('Y-m-d', strtotime($validated['from_date']));
    $toDate      = date('Y-m-d', strtotime($validated['to_date']));
    $confYear    = date('Y', strtotime($validated['from_date']));
    $shortAlias  = Str::slug($validated['short_name'], '-');

    // image – keep old if not replaced
    $bannerImg = $validated['hiddenmainImage'] ?? $existing->conference_img ?? null;
    if ($request->hasFile('mainImage')) {
        $file = $request->file('mainImage');
        if ($file && $file->isValid()) {
            if ($bannerImg && file_exists(public_path('uploads/conferences/' . $bannerImg))) {
                @unlink(public_path('uploads/conferences/' . $bannerImg));
            }
            $ext       = $file->getClientOriginalExtension();
            $bannerImg = time() . '.' . $ext;
            $file->move(public_path('uploads/conferences'), $bannerImg);
        }
    }

    // Folder + URL logic (same as store)
    if ((int) $validated['conferece_type'] === 1) {
        $folderName    = $this->clean($validated['folder_name']);
        $conferenceUrl = 'https://scisynopsisconferences.com/' . $folderName . '/';
    } else {
        $folderName    = $validated['folder_name'];
        $conferenceUrl = 'https://' . trim($folderName, '/') . '/';
    }

    $payload = [
        'continent_id'           => (int) $validated['continent_id'],
        'country_id'             => (int) $validated['country_id'],
        'city_id'                => (int) $validated['city_id'],
        'folder_name'            => $folderName,
        'gkey'                   => $validated['gkey'] ?? '',
        'gse'                    => $validated['gse'] ?? '',
        'conference_first_title' => $validated['conference_first_title'],
        'conference_name'        => $validated['conference_name'],
        'short_name'             => $validated['short_name'],
        'short_alias_name'       => $shortAlias,
        'conference_theme'       => $validated['conference_theme'],
        'conference_year'        => $confYear,
        'from_date'              => $fromDate,
        'to_date'                => $toDate,
        'conference_location'    => $validated['conference_location'],
        'conferece_type'         => (int) $validated['conferece_type'],
        'conference_url'         => $conferenceUrl,
        'conference_img'         => $bannerImg ?? '',
        'msubject_id'            => $msubjectId,
    ];

    DB::table('conferences')->where('id', $id)->update($payload);

    // refresh subject mapping: delete and re-insert
    DB::table('conference_categories')->where('conference_id', $id)->delete();
    foreach ($subjectIds as $sid) {
        DB::table('conference_categories')->insert([
            'conference_id' => $id,
            'subject_id'    => (int) $sid,
            'status'        => 1,
            'created_date'  => now(),
        ]);
    }

    // keep unique_id pattern
    $uploadsFolder = $folderName . '_' . $id;
    DB::table('conferences')->where('id', $id)->update([
        'unique_id' => $uploadsFolder,
    ]);
    $this->createConferenceFolders($uploadsFolder);

    $logUpdate('conferences', $id, $before);

    return response()->json([
        'status'  => true,
        'message' => 'Conference updated successfully',
    ]);
}
case 'c_past_conference_report': {

    $before = $existing;

    $validated = $request->validate([
        'conf_title' => 'required|string|max:255',
        'conf_desc'  => 'nullable|string',
    ]);

    $payload = [
        'conf_title' => $validated['conf_title'],
        'conf_desc'  => $validated['conf_desc'] ?? null,
    ];

    DB::table('c_past_conference_report')
        ->where('id', $id)
        ->update($payload);

    try {
        $this->logAudit(
            'c_past_conference_report',
            $id,
            'updated',
            (array) $before,
            $payload
        );
    } catch (\Throwable $e) {
        \Log::error("Audit log failed in v4Update [c_past_conference_report/{$id}]: " . $e->getMessage());
    }

    return response()->json([
        'status'  => true,
        'message' => 'Conference report updated successfully',
    ]);
}

           default:
                    return response()->json(['error' => 'Not supported'], 400);
            }

        } catch (\Throwable $e) {
            \Log::error("v4Update [{$table}/{$id}]: " . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
            ]);

            return response()->json([
                'status'  => false,
                'message' => $e->getMessage(),
            ], 422);
        }
    }

    /*
    |--------------------------------------------------------------------------
    | v4Delete â€“ mirror your v1Delete pattern (for v4 tables)
    |--------------------------------------------------------------------------
    */
    public function v4Delete(Request $request, $table, $id)
    {
        $table = trim((string) $table);

        if ($table === '' || !preg_match('/^[a-zA-Z0-9_]+$/', $table) || !Schema::hasTable($table)) {
            return response()->json(['error' => 'Invalid table'], 400);
        }

        $id = (int) $id;
        if ($id <= 0) {
            return response()->json(['error' => 'Invalid id'], 400);
        }

        try {
            $record = DB::table($table)->find($id);
            if (!$record) {
                return response()->json(['error' => 'Not found'], 404);
            }

            $before = $record;

            // ðŸ”¹ If v4 tables get file uploads later, put cleanup here (like v1Delete)
            // currently user_confs1 has no files â†’ plain delete


if ($table === 'web_cms') {
    $pageType = (int) ($record->page_type ?? 0);

    // board → delete image from /uploads/team
    if ($pageType === 12 && !empty($record->cms_img)) {
      $path = public_path('uploads/team/' . $record->cms_img);
      if (file_exists($path)) {
          @unlink($path);
      }
    }
}
// ... inside try, after $before = $record;

        // continents: soft delete (status=0) like deletecontinents()
        if ($table === 'continents') {
            DB::table('continents')
                ->where('id', $id)
                ->update(['status' => 0]);

            try {
                $this->logAudit('continents', $id, 'deleted', (array) $before, []);
            } catch (\Throwable $e) {
                \Log::error("Audit log failed in v4Delete [continents/{$id}]: " . $e->getMessage());
            }

            return response()->json(['status' => true]);
        }

        // countries: detach from continent (continent_id = null) like deletecountries()
        if ($table === 'countries') {
            DB::table('countries')
                ->where('id', $id)
                ->update(['continent_id' => null]);

            try {
                $this->logAudit('countries', $id, 'updated', (array) $before, ['continent_id' => null]);
            } catch (\Throwable $e) {
                \Log::error("Audit log failed in v4Delete [countries/{$id}]: " . $e->getMessage());
            }

            return response()->json(['status' => true]);
        }

        // cities: soft delete (status=0) like deletecities()
        if ($table === 'cities') {
            DB::table('cities')
                ->where('id', $id)
                ->update(['status' => 0]);

            try {
                $this->logAudit('cities', $id, 'deleted', (array) $before, ['status' => 0]);
            } catch (\Throwable $e) {
                \Log::error("Audit log failed in v4Delete [cities/{$id}]: " . $e->getMessage());
            }

            return response()->json(['status' => true]);
        }

// ... then your existing generic delete block ...

       
            DB::table($table)->where('id', $id)->delete();

            try {
                $this->logAudit(
                    $table,
                    $id,
                    'deleted',
                    (array) $before,
                    []
                );
            } catch (\Throwable $e) {
                \Log::error("Audit log failed in v4Delete [{$table}/{$id}]: " . $e->getMessage());
            }

            return response()->json(['status' => true]);

        } catch (\Throwable $e) {
            \Log::error("v4Delete [{$table} #{$id}]: " . $e->getMessage());
            return response()->json(['error' => 'Server error'], 500);
        }
    }
    
    
    public function v4BulkDelete(Request $request)
{
    $table = $request->input('table');
    $ids   = $request->input('ids', []);

    if (!$table || !Schema::hasTable($table)) {
        return response()->json([
            'status'  => false,
            'message' => 'Invalid table',
        ], 400);
    }

    if (!is_array($ids) || empty($ids)) {
        return response()->json([
            'status'  => false,
            'message' => 'No items selected',
        ], 400);
    }

    // limit which tables can use bulk delete (safety)
    if (!in_array($table, ['user_confs1'], true)) {
        return response()->json([
            'status'  => false,
            'message' => 'Bulk delete not allowed for this table',
        ], 403);
    }

    DB::table($table)->whereIn('id', $ids)->delete();

    // optional: audit log like your v1Delete
    if (method_exists($this, 'logAudit')) {
        $this->logAudit('bulk-delete', $table, null, [
            'ids'           => $ids,
            'conference_id' => $request->input('conference_id'),
            'user_id'       => $request->input('user_id'),
        ]);
    }

    return response()->json([
        'status'  => true,
        'message' => 'Successfully delete items',
    ]);
}



 public function v4GenerateReceipt($userId)
    {
        $userId = (int) $userId;
        if ($userId <= 0) {
            return response()->json(['error' => 'Invalid userId'], 400);
        }

        // Same tables as CI3 version
        $invoiceInfo   = DB::table('c_invoices')->where('user_id', $userId)->get();
        $receiptsInfo  = DB::table('c_receipts')->where('user_id', $userId)->get();
        $userInfo      = DB::table('c_reg_users')->where('id', $userId)->get();
        $paymentInfo   = DB::table('c_user_orders')->where('user_id', $userId)->get();
        $regInfo       = DB::table('c_user_registrations_dates')->where('user_id', $userId)->get();
        $addonInfo     = DB::table('c_user_addon_amounts')->where('user_id', $userId)->get();
        $userOrders    = DB::table('c_user_orders')->where('user_id', $userId)->get();
        $shippingInfo  = DB::table('c_registration_shipping')->where('regid', $userId)->get();
        $configurations = DB::table('configurations')->where('status', 1)->orderByDesc('id')->get();
        $address        = DB::table('web_cms')->where('page_type', 10)->orderByDesc('id')->get();

        $conferenceId = optional($receiptsInfo->first())->conference_id;
        $confenceInfo = DB::table('conferences')->where('id', $conferenceId)->orderByDesc('id')->get();

        $data = [
            'invoiceInfo'   => $invoiceInfo,
            'receiptsInfo'  => $receiptsInfo,
            'userInfo'      => $userInfo,
            'paymentInfo'   => $paymentInfo,
            'addonInfo'     => $addonInfo,
            'userOrders'    => $userOrders,
            'shippingInfo'  => $shippingInfo,
            'regInfo'       => $regInfo,
            'configurations'=> $configurations,
            'address'       => $address,
            'userId'        => $userId,
            'confenceInfo'  => $confenceInfo,
        ];

        // ⚠️ You need a Blade equivalent of ajax/download-receipt.php
        // For now assume you created resources/views/pdf/receipt.blade.php
        $html = view('pdf.receipt', $data)->render();

        $pdf = new Mpdf();
        $pdf->SetHTMLFooter('<div style="font-weight: bold;">{PAGENO}</div>','O');
        $pdf->SetHTMLFooter('<div style="text-align: right;">{PAGENO}</div>','E');

        $firstReceipt = $receiptsInfo->first();
        $gendate = $firstReceipt ? date('Y-m-d', strtotime($firstReceipt->created_date)) : date('Y-m-d');

        $pdf->SetHTMLHeader('Generated on: ' . $gendate);
        $pdf->SetWatermarkText('Scisynopsis Conferences');
        $pdf->showWatermarkText = true;
        $pdf->WriteHTML($html);

        $fileName = optional($paymentInfo->first())->order_id ?: date('ymdhis');
        $output   = $fileName . '.pdf';

        // Stream PDF to browser like CI3 Output('D')
        return response($pdf->Output($output, 'S'), 200, [
            'Content-Type'        => 'application/pdf',
            'Content-Disposition' => 'attachment; filename="'.$output.'"',
        ]);
    }

    /**
     * DELETE /api/v4/invoices/{id}
     * Laravel equivalent of users_invoices_delete($bannerid)
     */
    public function v4UsersInvoicesDelete($id)
    {
        $id = (int) $id;
        if ($id <= 0) {
            return response()->json(['error' => 'Invalid id'], 400);
        }

        try {
            $before = DB::table('c_invoices')->find($id);
            if (!$before) {
                return response()->json(['error' => 'Not found'], 404);
            }

            DB::table('c_invoices')->where('id', $id)->delete();

            try {
                $this->logAudit(
                    'c_invoices',
                    $id,
                    'deleted',
                    (array)$before,
                    []
                );
            } catch (\Throwable $e) {
                \Log::error("Audit log failed in v4UsersInvoicesDelete: " . $e->getMessage());
            }

            return response()->json(['status' => true, 'message' => 'Successfully Deleted']);
        } catch (\Throwable $e) {
            \Log::error("v4UsersInvoicesDelete [{$id}]: " . $e->getMessage());
            return response()->json(['error' => 'Server error'], 500);
        }
    }


    
    public function v4GalleryYears(Request $request)
{
    $parentId = (int) $request->input('parent_id', 0);

    $years = DB::table('previous_conference_gallery')
        ->select('year_id')
        ->where('parent_id', $parentId)
        ->whereNotNull('year_id')
        ->groupBy('year_id')
        ->orderBy('year_id', 'ASC')
        ->get();

    return response()->json($years);
}

// ====== v4: previous_conference_videos ======

public function v4VideosYears(Request $request)
{
    $years = DB::table('previous_conference_videos')
        ->whereNotNull('year_id')
        ->distinct()
        ->orderBy('year_id', 'DESC')
        ->pluck('year_id');

    return response()->json($years);
}
public function v4ConferencesActive()
{
    $conferences = DB::table('conferences')
        ->where('status', '>=', 1)
        ->orderByDesc('id')
        ->get([
            'id',
            'folder_name',
            'short_name',
            'conference_name',
            'conference_year',
        ])
        ->map(function ($c) {
            return [
                'id' => (int) $c->id,
                'value' => (int) $c->id,
                'label' => $c->short_name ?: $c->folder_name ?: $c->conference_name ?: "Conf #{$c->id}",
                'year' => $c->conference_year ?: null,
            ];
        });

    return response()->json($conferences);
}

// GET /api/v4/social → fetch current social links (page_type = 11)
public function v4Social()
{
    $record = DB::table('web_cms')
        ->where('page_type', 11)
        ->orderByDesc('id')
        ->first();

    // Return object (even if null) for Vue to handle
    return response()->json($record ?: (object)[]);
}

// POST /api/v4/social/save → upsert
public function v4SaveSocial(Request $request)
{
    $validated = $request->validate([
        'content_title' => 'nullable|string|max:500',
        'sub_title'     => 'nullable|string|max:500',
        'short_desc'    => 'nullable|string|max:500',
        'long_desc'     => 'nullable|string|max:1000',
        'cms_img'       => 'nullable|string|max:500',
    ]);

    $data = array_merge($validated, [
        'page_type'    => 11,
        'created_date' => now(),
    ]);

    $existing = DB::table('web_cms')->where('page_type', 11)->first();

    if ($existing) {
        DB::table('web_cms')->where('id', $existing->id)->update($data);
    } else {
        DB::table('web_cms')->insert($data);
    }

    // Optional: audit log
    try {
        $this->logAudit('web_cms', $existing->id ?? 'new', 'updated', [], $data);
    } catch (\Throwable $e) {
        \Log::warning("Audit skipped for social save: " . $e->getMessage());
    }

    return response()->json(['status' => true]);
}
// GET /v4/bank-info
public function v4BankInfo()
{
    $row = DB::table('bank_details')
        ->where('status', 1)
        ->first();

    return response()->json($row ?: (object)[]);
}

// POST /v4/bank-info/save
public function v4SaveBankInfo(Request $request)
{
    $data = $request->validate([
        'invoice_from' => 'nullable|string',
        'bank_info'    => 'nullable|string',
    ]);

    $existing = DB::table('bank_details')->where('status', 1)->first();

    if ($existing) {
        DB::table('bank_details')->where('id', $existing->id)->update($data);
    } else {
        $data['status'] = 1;
        DB::table('bank_details')->insert($data);
    }

    return response()->json(['status' => true]);
}

// GET /v4/company-info/{id}
public function v4CompanyInfo(Request $request, $id)
{
    $id = (int) $id;
    if ($id <= 0) {
        return response()->json(['error' => 'Invalid ID'], 400);
    }

    $info = DB::table('company_info')->find($id);
    return response()->json($info ?: (object)[]);
}

// POST /v4/company-info/{id}/update
public function v4UpdateCompanyInfo(Request $request, $id)
{
    $id = (int) $id;
    if ($id <= 0) {
        return response()->json(['error' => 'Invalid ID'], 400);
    }

    $validated = $request->validate([
        'company_name'    => 'required|string|max:255',
        'company_address' => 'required|string',
        'bank_name'       => 'nullable|string|max:255',
        'iban'            => 'nullable|string|max:50',
        'swift_code'      => 'nullable|string|max:20',
        'bank_address'    => 'nullable|string',
        'paypal_id'       => 'nullable|email|max:255',
    ]);

    $data = array_merge($validated, [
        'updated_at' => now(),
    ]);

    $updated = DB::table('company_info')->where('id', $id)->update($data);
    if (!$updated) {
        return response()->json(['error' => 'Record not found'], 404);
    }

    try {
        $this->logAudit('company_info', $id, 'updated', [], $data);
    } catch (\Throwable $e) {
        \Log::warning("Audit skipped: " . $e->getMessage());
    }

    return response()->json(['status' => true]);
}

// GET /v4/web-cms?page_type=5
public function v4WebCmsIndex(Request $request)
{
    $pageType = (int) $request->input('page_type', 5);
    $rows = DB::table('web_cms')
        ->where('page_type', $pageType)
        ->where('status', 1)
        ->orderBy('id', 'DESC')
        ->get();

    return response()->json($rows);
}

// POST /v4/web-cms/store
public function v4WebCmsStore(Request $request)
{
    $validated = $request->validate([
        'content_title' => 'required|string|max:500',
        'long_desc'     => 'required|string',
        'page_type'     => 'required|integer|min:1',
    ]);

    $data = [
        'content_title' => $validated['content_title'],
        'sub_title'     => Str::slug($validated['content_title']),
        'long_desc'     => $validated['long_desc'],
        'page_type'     => $validated['page_type'],
        'status'        => 1,
        'created_date'  => now(),
    ];

    $id = DB::table('web_cms')->insertGetId($data);
    return response()->json(['status' => true, 'id' => $id]);
}

// POST /v4/web-cms/{id}/update
public function v4WebCmsUpdate(Request $request, $id)
{
    $validated = $request->validate([
        'content_title' => 'required|string|max:500',
        'long_desc'     => 'required|string',
    ]);

    $data = [
        'content_title' => $validated['content_title'],
        'sub_title'     => Str::slug($validated['content_title']),
        'long_desc'     => $validated['long_desc'],
    ];

    DB::table('web_cms')->where('id', $id)->update($data);
    return response()->json(['status' => true]);
}

// DELETE /v4/web-cms/{id}
public function v4WebCmsDelete($id)
{
    DB::table('web_cms')->where('id', $id)->delete();
    return response()->json(['status' => true]);
}

public function v4RegistrationCategoriesSortOrder(Request $request)
{
    $sortOrder = (string) $request->input('sort_order', '');
    $pageType  = (int) $request->input('page_type', 1);

    if ($sortOrder === '') {
        return response()->json([
            'status'  => false,
            'message' => 'No sort_order provided',
        ], 400);
    }

    $ids = array_filter(array_map('trim', explode(',', $sortOrder)));

    foreach (array_values($ids) as $index => $rawId) {
        $id = (int) $rawId;
        if ($id <= 0) continue;

        DB::table('c_registration_categories')
            ->where('id', $id)
            ->where('page_type', $pageType)
            ->update([
                'order_position' => $index + 1,
            ]);
    }

    return response()->json([
        'status'  => true,
        'message' => 'Successfully Updated',
    ]);
}
// GET /api/v4/regcategories-sort?page_type=1
public function v4RegCategoriesSortList(Request $request)
{
    $pageType = (int) $request->input('page_type', 1);

    $rows = DB::table('c_registration_categories')
        ->select('id', 'category_name', 'order_position')
        ->where('page_type', $pageType)
        ->orderBy('order_position', 'ASC')
        ->orderBy('id', 'ASC')
        ->get();

    return response()->json($rows);
}

// POST /api/v4/regcategories-sort-save
public function v4RegCategoriesSortSave(Request $request)
{
    $validated = $request->validate([
        'page_type'          => 'required|integer',
        'sort_order'         => 'required|array',
        'sort_order.*'       => 'integer',
    ]);

    $pageType = (int) $validated['page_type'];
    $ids      = $validated['sort_order'];

    DB::beginTransaction();
    try {
        foreach ($ids as $index => $id) {
            DB::table('c_registration_categories')
                ->where('id', $id)
                ->where('page_type', $pageType)
                ->update([
                    'order_position' => $index + 1,
                ]);
        }

        DB::commit();

        return response()->json([
            'status'  => true,
            'message' => 'Successfully Updated',
        ]);
    } catch (\Throwable $e) {
        DB::rollBack();
        \Log::error("v4RegCategoriesSortSave: " . $e->getMessage(), [
            'trace' => $e->getTraceAsString(),
        ]);

        return response()->json([
            'status'  => false,
            'message' => 'Failed to Update data',
        ], 500);
    }
}
// GET /api/v4/board-sort  → list board (page_type=12) for sorting
public function v4BoardSortList()
{
    $rows = DB::table('web_cms')
        ->select('id', 'content_title', 'order_position')
        ->where('page_type', 12)              // Scientific Board
        ->orderBy('order_position', 'ASC')
        ->orderBy('id', 'ASC')
        ->get();

    return response()->json($rows);
}

// POST /api/v4/board-sort-save  → save new sequence
public function v4BoardSortSave(Request $request)
{
    $validated = $request->validate([
        'sort_order'   => 'required|array',
        'sort_order.*' => 'integer',
    ]);

    $ids = $validated['sort_order'];

    DB::beginTransaction();
    try {
        foreach ($ids as $index => $id) {
            DB::table('web_cms')
                ->where('id', $id)
                ->where('page_type', 12)
                ->update([
                    'order_position' => $index + 1,
                ]);
        }

        DB::commit();

        return response()->json([
            'status'  => true,
            'message' => 'Successfully Updated',
        ]);
    } catch (\Throwable $e) {
        DB::rollBack();
        \Log::error("v4BoardSortSave: " . $e->getMessage(), [
            'trace' => $e->getTraceAsString(),
        ]);

        return response()->json([
            'status'  => false,
            'message' => 'Failed to Update data',
        ], 500);
    }
}


// GET /api/v4/whatsapp  → fetch current WhatsApp settings
public function v4Whatsapp()
{
    // take latest active row, same idea as CI3 getAllInfo("whatsapp")
    $row = DB::table('whatsapp')
        ->where('status', 1)
        ->orderByDesc('id')
        ->first();

    // always return an object so Vue doesn’t break on null
    return response()->json($row ?: (object)[]);
}

// POST /api/v4/whatsapp/save  → insert or update single row
public function v4SaveWhatsapp(Request $request)
{
    $validated = $request->validate([
        'whatsapp_number' => 'required|string|max:50',
        'facebook_id'     => 'nullable|string|max:255',
    ]);

    $data = [
        'whatsapp_number' => $validated['whatsapp_number'],
        'facebook_id'     => $validated['facebook_id'] ?? null,
        'status'          => 1,
        'created_date'    => now(),
    ];

    // if a row exists, update it; else insert new one
    $existing = DB::table('whatsapp')->orderByDesc('id')->first();

    if ($existing) {
        DB::table('whatsapp')
            ->where('id', $existing->id)
            ->update($data);

        // optional audit
        try {
            $this->logAudit(
                'whatsapp',
                $existing->id,
                'updated',
                (array) $existing,
                $data
            );
        } catch (\Throwable $e) {
            \Log::warning("Audit skipped for WhatsApp update: " . $e->getMessage());
        }
    } else {
        $id = DB::table('whatsapp')->insertGetId($data);

        try {
            $this->logAudit(
                'whatsapp',
                $id,
                'created',
                [],
                $data
            );
        } catch (\Throwable $e) {
            \Log::warning("Audit skipped for WhatsApp create: " . $e->getMessage());
        }
    }

    return response()->json(['status' => true]);
}
// GET /api/v4/chat-script  → fetch current live chat script
public function v4ChatScript()
{
    $row = DB::table('chat_script')
        ->orderByDesc('id')
        ->first();

    // return an object (even if empty) so Vue can bind safely
    return response()->json($row ?: (object)[]);
}

// POST /api/v4/chat-script/save  → create / update live chat script
public function v4SaveChatScript(Request $request)
{
    $validated = $request->validate([
        'chat_script' => 'required|string',
    ]);

    $existing = DB::table('chat_script')
        ->orderByDesc('id')
        ->first();

    $data = [
        'chat_script'  => $validated['chat_script'],
        'status'       => 1,
        'created_date' => now(),
    ];

    if ($existing) {
        // update latest row
        DB::table('chat_script')
            ->where('id', $existing->id)
            ->update($data);

        $id = $existing->id;
    } else {
        // insert first row
        $id = DB::table('chat_script')->insertGetId($data);
    }

    // optional audit log, same pattern as other v4 helpers
    try {
        $this->logAudit(
            'chat_script',
            $id,
            'updated',
            (array) ($existing ?? []),
            $data
        );
    } catch (\Throwable $e) {
        \Log::warning("Audit skipped for chat_script save: " . $e->getMessage());
    }

    return response()->json([
        'status' => true,
    ]);
}

public function v4MessengerScript()
{
    $row = DB::table('messenger_script')
        ->orderByDesc('id')
        ->first();

    return response()->json($row ?: (object)[]);
}

public function v4SaveMessengerScript(Request $request)
{
    $data = $request->validate([
        'messenger_script' => 'required|string',
    ]);

    $existing = DB::table('messenger_script')
        ->orderByDesc('id')
        ->first();

    $payload = [
        'messenger_script' => $data['messenger_script'],
        'status'           => 1,
        'created_date'     => now(),
    ];

    if ($existing) {
        DB::table('messenger_script')
            ->where('id', $existing->id)
            ->update($payload);
    } else {
        DB::table('messenger_script')->insert($payload);
    }

    return response()->json(['status' => true]);
}

protected function createConferenceFolders(string $uploadsFolder): void
{
    $source = $_SERVER['DOCUMENT_ROOT'] ?? public_path();

    if ($source === 'E:/xampp/htdocs') {
        $base = $source . "/projects/conference-admin/uploads/";
        if (!file_exists($base . $uploadsFolder)) {
            @mkdir($base . $uploadsFolder, 0755, true);
            @mkdir($base . "conferences/" . $uploadsFolder, 0755, true);
        }
    } elseif ($source === "/home1/a9iceonj/public_html/hostgenie.in") {
        $path = $source . "/phpdev/conference-admin/uploads/";
        if (!file_exists($path . $uploadsFolder)) {
            @mkdir($path . $uploadsFolder, 0755, true);
        }
    } else {
        // default: /conference-admin/uploads or /uploads
        if (!file_exists($source . "/conference-admin/uploads/" . $uploadsFolder)) {
            @mkdir($source . "/conference-admin/uploads/" . $uploadsFolder, 0755, true);
        }
    }
}
// POST /api/v4/conferences/assign-admin
public function v4AssignConferenceAdmins(Request $request)
{
    $validated = $request->validate([
        'admin_id'       => 'required|integer',
        'conference_ids' => 'required|array|min:1',
        'conference_ids.*' => 'integer|min:1',
    ]);

    $adminId       = (int) $validated['admin_id'];
    $conferenceIds = $validated['conference_ids'];

    // Basic check – must be active admin (usertype 2 or 4 as per your needs)
    $isAdmin = DB::table('users')
        ->where('id', $adminId)
        ->whereIn('usertype', [2,4])
        ->where('status', 1)
        ->exists();

    if (!$isAdmin) {
        return response()->json([
            'status'  => false,
            'message' => 'Selected user is not an active Admin.',
        ], 422);
    }

    foreach ($conferenceIds as $confId) {
        $confId = (int) $confId;
        if ($confId <= 0) {
            continue;
        }

        // avoid duplicates
        $exists = DB::table('user_confs')
            ->where('conference_id', $confId)
            ->where('admin_id', $adminId)
            ->exists();

        if ($exists) {
            continue;
        }

        DB::table('user_confs')->insert([
            'conference_id' => $confId,
            'admin_id'      => $adminId,
            'created_date'  => now(),
        ]);
    }

    return response()->json([
        'status'  => true,
        'message' => 'Successfully Saved Data',
    ]);
}

// POST /api/v4/conferences/{id}/change-year
public function v4ConferenceYear(Request $request, $bannerid)
{
    $bannerid = (int) $bannerid;
    if ($bannerid <= 0) {
        return response()->json(['status' => false, 'message' => 'Invalid conference id'], 400);
    }

    $request->validate([
        'from_date' => 'required|date',
        'to_date'   => 'required|date',
    ]);

    $info = DB::table('conferences')->where('id', $bannerid)->first();
    if (!$info) {
        return response()->json(['status' => false, 'message' => 'Conference not found'], 404);
    }

    $oldCnfTitle   = $info->conference_name;
    $oldFolderName = $info->folder_name;
    $conferenceYear= $info->conference_year;

    // --- Update existing row folder/URL + mark as prev_conf (mirrors CI3) ---
    if ((int) $info->conferece_type === 1) {

        $baseFolderName  = $oldFolderName . '/' . $conferenceYear;
        $existingFolders = $this->getSimilarFolderNames($baseFolderName);
        $newFolderName   = $this->generateUniqueFolderName($existingFolders, $baseFolderName);
        $conferenceUrl   = 'https://scisynopsisconferences.com/' . $newFolderName . '/';

        DB::table('conferences')->where('id', $bannerid)->update([
            'folder_name'     => $newFolderName,
            'conference_url'  => $conferenceUrl,
            'prev_conf'       => 1,
            'conference_name' => $oldCnfTitle,
        ]);
    } else {

        $folderNameUrl      = $info->folder_name;
        $folderNameModified = preg_replace("/\/\d+(-\d+)?\/?$/", "/", $folderNameUrl);
        $baseFolderNameexist= $folderNameModified . '/' . $conferenceYear;

        $parentDir      = $folderNameModified;
        $baseFolderName = $conferenceYear;
        $uniqueFolder   = $this->getUniqueFolderName($parentDir, $baseFolderName);

        if ($this->folderExists($baseFolderNameexist)) {

            $url         = $info->conference_url;
            $modifiedUrl = preg_replace("/\/\d+(-\d+)?\/$/", "/", $url);
            $tempUrl     = $modifiedUrl . $uniqueFolder;

            $parsedUrl   = parse_url($tempUrl);
            $path        = $parsedUrl['path'] ?? '';
            $lastSegment = basename($path);

            $conferenceUrl = $modifiedUrl . $lastSegment . '/';

            DB::table('conferences')->where('id', $bannerid)->update([
                'folder_name'     => $uniqueFolder,
                'conference_url'  => $conferenceUrl,
                'prev_conf'       => 1,
                'conference_name' => $oldCnfTitle,
            ]);
        } else {

            $url         = $info->conference_url;
            $modifiedUrl = preg_replace("/\/\d+(-\d+)?\/$/", "/", $url);
            $conferenceUrl = $modifiedUrl . $conferenceYear . '/';

            DB::table('conferences')->where('id', $bannerid)->update([
                'folder_name'     => $baseFolderNameexist,
                'conference_url'  => $conferenceUrl,
                'prev_conf'       => 1,
                'conference_name' => $oldCnfTitle,
            ]);
        }
    }

    // --- Create new conference row for new year (same as CI3) ---
    $conferenceYearNew = date('Y', strtotime($request->input('from_date')));

    $category       = explode('.', $info->conference_img);
    $conferenceImgNew = time() . '.' . end($category);

    $prevConferenceId = ($info->prev_conference_id == 0)
        ? $bannerid
        : $info->prev_conference_id;

    $confType = ((int) $info->conferece_type === 2) ? 2 : 1;

    $newId = DB::table('conferences')->insertGetId([
        'prev_conference_id'    => $prevConferenceId,
        'continent_id'          => $info->continent_id,
        'country_id'            => $info->country_id,
        'city_id'               => $info->city_id,
        'folder_name'           => $oldFolderName,
        'conferece_type'        => $confType,
        'conference_first_title'=> $info->conference_first_title,
        'conference_theme'      => $info->conference_theme,
        'msubject_id'           => $info->msubject_id,
        'conference_name'       => $info->conference_name,
        'conference_year'       => $conferenceYearNew,
        'from_date'             => date('Y-m-d', strtotime($request->input('from_date'))),
        'to_date'               => date('Y-m-d', strtotime($request->input('to_date'))),
        'conference_location'   => $info->conference_location,
        'conference_url'        => $info->conference_url,
        'files_dump_status'     => 0,
        'conference_img'        => $conferenceImgNew,
        'created_date'          => now(),
        'gkey'                  => $info->gkey,
        'gse'                   => $info->gse,
    ]);

    if ($newId) {
        // copy subjects
        $subjects = DB::table('conference_categories')
            ->where('status', 1)
            ->where('conference_id', $bannerid)
            ->get();

        foreach ($subjects as $sub) {
            DB::table('conference_categories')->insert([
                'conference_id' => $newId,
                'subject_id'    => $sub->subject_id,
                'status'        => 1,
                'created_date'  => now(),
            ]);
        }

        $uploadsFolder = $oldFolderName . '_' . $newId;
        DB::table('conferences')->where('id', $newId)->update([
            'unique_id' => $uploadsFolder,
        ]);

        // copy image
        @copy(
            public_path('uploads/conferences/' . $info->conference_img),
            public_path('uploads/conferences/' . $conferenceImgNew)
        );

        $this->createConferenceFolders($uploadsFolder);
    }

    return response()->json([
        'status'  => true,
        'message' => 'Successfully Data Updated',
    ]);
}
// POST /api/v4/conferences/{id}/move-files
public function v4MoveFilesToConference($conference, $conferenceid = null)
{
    // Allow both signatures: /move-files/{id} OR old style /move-files/{folder}/{id}
    if ($conferenceid === null) {
        $conferenceid = (int) $conference;
        $conferenceFolder = null;
    } else {
        $conferenceFolder = $conference;
        $conferenceid     = (int) $conferenceid;
    }

    if ($conferenceid <= 0) {
        return response()->json(['status' => false, 'message' => 'Invalid conference id'], 400);
    }

    if ($conferenceFolder === '' || $conferenceFolder === '/') {
        $conf = DB::table('conferences')->where('id', $conferenceid)->first();
        if (!$conf) {
            return response()->json(['status' => false, 'message' => 'Conference not found'], 404);
        }
        $conferenceFolder = $conf->folder_name;
    }

    if ($conferenceFolder !== '' && $conferenceFolder !== '/') {
        $this->fullCopy('dump', $conferenceFolder);
    }

    DB::table('conferences')->where('id', $conferenceid)->update([
        'files_dump_status' => 1,
    ]);

    return response()->json(['status' => true, 'message' => 'Files moved successfully']);
}

// POST /api/v4/conferences/{id}/move-files-slash
public function v4MoveFilesToConferenceSlash($conferenceid)
{
    $conferenceid = (int) $conferenceid;
    if ($conferenceid <= 0) {
        return response()->json(['status' => false, 'message' => 'Invalid conference id'], 400);
    }

    $this->fullCopyNew('dump', $conferenceid);

    DB::table('conferences')->where('id', $conferenceid)->update([
        'files_dump_status' => 1,
    ]);

    return response()->json(['status' => true, 'message' => 'Files moved successfully']);
}
protected function fullCopy(string $source, string $target): void
{
    $docroot = rtrim($_SERVER['DOCUMENT_ROOT'] ?? public_path(), '/');

    if ($target === '' || $target === null) {
        return;
    }

    if (is_dir($docroot . '/' . $source)) {
        @mkdir($docroot . '/' . $target);
        $d = dir($docroot . '/' . $source);
        while (false !== ($entry = $d->read())) {
            if ($entry === '.' || $entry === '..') {
                continue;
            }
            $Entry = $docroot . '/' . $source . '/' . $entry;
            $entr  = $source . '/' . $entry;

            if (is_dir($Entry)) {
                @mkdir($docroot . '/' . $target . '/' . $entry);
                chmod($docroot . '/' . $target . '/' . $entry, 0777);
                $this->fullCopy($entr, $target . '/' . $entry);
                continue;
            }

            @copy($Entry, $docroot . '/' . $target . '/' . $entry);
            chmod($docroot . '/' . $target . '/' . $entry, 0777);
        }
        $d->close();
    } else {
        chmod($docroot . '/' . $target, 0777);
        @copy($docroot . '/' . $source, $docroot . '/' . $target);
    }
}

protected function fullCopyNew(string $source, int $conferenceId): void
{
    $conf = DB::table('conferences')->where('id', $conferenceId)->first();
    if (!$conf || !$conf->folder_name) {
        return;
    }

    $target  = $conf->folder_name;
    $docroot = rtrim($_SERVER['DOCUMENT_ROOT'] ?? public_path(), '/');

    if ($target === '' || $target === null) {
        return;
    }

    if (is_dir($docroot . '/' . $source)) {
        $d = dir($docroot . '/' . $source);
        while (false !== ($entry = $d->read())) {
            if ($entry === '.' || $entry === '..') {
                continue;
            }
            $Entry = $docroot . '/' . $source . '/' . $entry;
            $entr  = $source . '/' . $entry;

            if (is_dir($Entry)) {
                @mkdir($docroot . '/' . $target . '/' . $entry);
                chmod($docroot . '/' . $target . '/' . $entry, 0777);
                $this->fullCopy($entr, $target . '/' . $entry);
                continue;
            }

            @copy($Entry, $docroot . '/' . $target . '/' . $entry);
            chmod($docroot . '/' . $target . '/' . $entry, 0777);
        }
        $d->close();
    } else {
        chmod($docroot . '/' . $target, 0777);
        @copy($docroot . '/' . $source, $docroot . '/' . $target);
    }
}
// POST /api/v4/conferences/{id}/generate-routes
public function v4GenerateConferenceRoutes($conferenceid)
{
    $conferenceid = (int) $conferenceid;
    if ($conferenceid <= 0) {
        return response()->json(['status' => false, 'message' => 'Invalid conference id'], 400);
    }

    $conf = DB::table('conferences')->where('id', $conferenceid)->first();
    if (!$conf || !$conf->folder_name) {
        return response()->json(['status' => false, 'message' => 'Conference not found'], 404);
    }

    $this->generateRouter($conf->folder_name);

    return response()->json(['status' => true, 'message' => 'Routes file generated']);
}

protected function generateRouter(string $folder = null): void
{
    if ($folder === '' || $folder === null) {
        return;
    }

    $listings = DB::table('conferences')->where('folder_name', $folder)->get();
    if ($listings->isEmpty()) {
        return;
    }

    // CI3 logic: strip https:// and slashes
    $fname = str_replace('https://', '', $listings[0]->folder_name);
    $ffold = str_replace('/', '', $fname);

    $docroot = rtrim($_SERVER['DOCUMENT_ROOT'] ?? base_path('public'), '/');
    $apath   = $docroot . '/' . $ffold;
    $pathToFile = $apath . '/application/config/routes.php';

    @mkdir(dirname($pathToFile), 0777, true);
    @chmod($pathToFile, 0777);

    $str  = "<?php\n";
    $str .= 'defined("BASEPATH") OR exit("No direct script access allowed");' . "\n";
    $str .= '$route["default_controller"] = "home";' . "\n";
    $str .= '$route["404_override"] = "";' . "\n";
    $str .= '$route["translate_uri_dashes"] = FALSE;' . "\n";

    foreach ($listings as $conf) {
        $id = $conf->id;

        $str .= '$route["/"] = "home/index/' . $id . '";' . "\n";
        $str .= '$route["about-us"] = "conferences/aboutus/' . $id . '";' . "\n";
        $str .= '$route["committeemembers"] = "conferences/organizingcommittee/' . $id . '";' . "\n";
        $str .= '$route["keynote-speakers"] = "conferences/posterpresentation/' . $id . '/2";' . "\n";
        $str .= '$route["keynote-speakers/(:any)"] = "conferences/speakerdetails/' . $id . '/2/$1";' . "\n";
        $str .= '$route["eminent-speakers"] = "conferences/speakers/' . $id . '/6";' . "\n";
        $str .= '$route["eminent-speakers/(:any)"] = "conferences/speakerdetails/' . $id . '/6/$1";' . "\n";
        $str .= '$route["renowned-speakers"] = "conferences/posterpresentation/' . $id . '/3";' . "\n";
        $str .= '$route["performers"] = "conferences/posterpresentation/' . $id . '/3";' . "\n";
        $str .= '$route["previous-speakers"] = "conferences/previousspeakers/' . $id . '";' . "\n";
        $str .= '$route["venue"] = "conferences/venue/' . $id . '";' . "\n";

        $str .= '$route["guidelines"] = "home/guidelines";' . "\n";
        $str .= '$route["terms-conditions"] = "home/termsconditions/2";' . "\n";
        $str .= '$route["faq"] = "home/faq";' . "\n";
        $str .= '$route["sponsors-exhibitors"] = "conferences/sponsorsexhibitors/' . $id . '";' . "\n";
        $str .= '$route["poster-presentation"] = "conferences/posterpresentation/' . $id . '/4";' . "\n";
        $str .= '$route["e-poster-presentation"] = "conferences/posterpresentation/' . $id . '/5";' . "\n";
        $str .= '$route["programschedule"] = "conferences/conferenceprogram/' . $id . '";' . "\n";
        $str .= '$route["program-schedule"] = "conferences/conferenceprogram/' . $id . '";' . "\n";
        $str .= '$route["scientificprogram"] = "conferences/abstractprogram/' . $id . '";' . "\n";
        $str .= '$route["sessions"] = "conferences/callforpapers/' . $id . '";' . "\n";
        $str .= '$route["sessions/(:any)"] = "conferences/sessiondetails/' . $id . '/$1";' . "\n";
        $str .= '$route["downloadbrochure"] = "conferences/brochure/' . $id . '";' . "\n";
        $str .= '$route["download-brochure"] = "conferences/brochure/' . $id . '";' . "\n";
        $str .= '$route["thankyou"] = "conferences/thankyou/' . $id . '";' . "\n";
        $str .= '$route["registration"] = "conferences/registration/' . $id . '";' . "\n";
        $str .= '$route["registration/(:any)"] = "conferences/registration/' . $id . '/$1";' . "\n";
        $str .= '$route["discountregistration"] = "conferences/discountregistration/' . $id . '";' . "\n";
        $str .= '$route["discountregistration/(:any)"] = "conferences/discountregistration/' . $id . '/$1";' . "\n";
        $str .= '$route["contact"] = "conferences/contactus/' . $id . '";' . "\n";

        $str .= '$route["abstractsubmission"] = "conferences/abstractsubmission/' . $id . '";' . "\n";
        $str .= '$route["abstract-submission"] = "conferences/abstractsubmission/' . $id . '";' . "\n";
        $str .= '$route["group-registration"] = "conferences/groupregistration/' . $id . '";' . "\n";
        $str .= '$route["login"] = "conferences/login/' . $id . '";' . "\n";
        $str .= '$route["abstract-registration"] = "conferences/abstractregistration/' . $id . '";' . "\n";
        $str .= '$route["forgot-password"] = "conferences/forgotpassword/' . $id . '";' . "\n";
        $str .= '$route["verification"] = "conferences/verification/' . $id . '";' . "\n";
        $str .= '$route["verification-completed"] = "conferences/phoneverified/' . $id . '";' . "\n";
        $str .= '$route["conference-gallery"] = "conferences/gallery/' . $id . '";' . "\n";
        $str .= '$route["conference-gallery/(:any)"] = "conferences/gallery/' . $id . '/$1";' . "\n";
        $str .= '$route["past-events"] = "conferences/pastevents/' . $id . '";' . "\n";
        $str .= '$route["exhibitors-sponsors/(:any)"] = "conferences/exhibitors/' . $id . '/$1";' . "\n";
        $str .= '$route["media-partners"] = "conferences/mediapartners/' . $id . '";' . "\n";
        $str .= '$route["media-partners/(:any)"] = "conferences/collaboration/' . $id . '/$1";' . "\n";
        $str .= '$route["contact-us"] = "conferences/contactus/' . $id . '";' . "\n";
        $str .= '$route["scientific-committee"] = "conferences/organizingcommittee/' . $id . '";' . "\n";
        $str .= '$route["scientific-committee/(:any)"] = "conferences/speakerdetails/' . $id . '/1/$1";' . "\n";
        $str .= '$route["dashboard"] = "conferences/dashboard/' . $id . '";' . "\n";
        $str .= '$route["changepassword"] = "conferences/changepassword/' . $id . '";' . "\n";
        $str .= '$route["abstractInfo"] = "conferences/abstractInfo/' . $id . '";' . "\n";
        $str .= '$route["submitabstract"] = "conferences/submitabstract/' . $id . '";' . "\n";
        $str .= '$route["abstractDetails/(:any)"] = "conferences/abstractDetails/' . $id . '/$1";' . "\n";
        $str .= '$route["updateprofile"] = "conferences/updateprofile/' . $id . '";' . "\n";
        $str .= '$route["success"] = "home/success/' . $id . '";' . "\n";
        $str .= '$route["cancelOnline"] = "home/cancelOnline/' . $id . '";' . "\n";
        $str .= '$route["paypalSuccessOnline"] = "home/paypalSuccessOnline/' . $id . '";' . "\n";
        $str .= '$route["search"] = "home/search/' . $id . '";' . "\n";
        $str .= '$route["checkoutonline"] = "home/checkoutonline/' . $id . '";' . "\n";
        $str .= '$route["onlineregistration"] = "conferences/onlineregistration/' . $id . '";' . "\n";
        $str .= '$route["paypalSuccess/(:any)"] = "conferences/paypalSuccess/' . $id . '";' . "\n";
        $str .= '$route["cancelPaypal/(:any)"] = "conferences/cancelPaypal/' . $id . '";' . "\n";
        $str .= '$route["scientific-program"] = "conferences/abstractprogram/' . $id . '";' . "\n";
        $str .= '$route["scientific-program/(:any)"] = "conferences/programdetails/' . $id . '/$1";' . "\n";
        $str .= '$route["download-sponsorship"] = "conferences/sponsorship/' . $id . '";' . "\n";
        $str .= '$route["sponsorship-thankyou"] = "conferences/sponsorshipthankyou/' . $id . '";' . "\n";
        $str .= '$route["subscribe"] = "conferences/subscribe/' . $id . '";' . "\n";
        $str .= '$route["unsubscribe"] = "conferences/unsubscribe/' . $id . '";' . "\n";
        $str .= '$route["conference-book"] = "conferences/conferencebook/' . $id . '";' . "\n";
        $str .= '$route["online-registration"] = "conferences/onlineregistration/' . $id . '";' . "\n";
        $str .= '$route["all-speakers"] = "conferences/allspeakers/' . $id . '";' . "\n";
        $str .= '$route["presentation-submission"] = "conferences/presentationsubmission/' . $id . '";' . "\n";
    }

    file_put_contents($pathToFile, $str);
}
public function v4ConferencePublish(Request $request)
{
    $id = $request->input('conference_id');
    $status = $request->input('publish_status', 0);

    $conf = DB::table('conferences')->where('id', $id)->first();
    if (!$conf) {
        return response()->json(['status' => false, 'message' => 'Conference not found'], 404);
    }

    DB::table('conferences')
        ->where('id', $id)
        ->update(['publish_status' => (int)$status]);

    return response()->json(['status' => true, 'message' => 'Publish status updated']);
}


   public function v4ConferenceMoveInfo($id)
    {
        $conference = DB::table('conferences')->where('id', $id)->first();

        if (!$conference) {
            return response()->json([
                'status'  => false,
                'message' => 'Conference not found',
            ], 404);
        }

        // ✅ your real server root for scisynopsisconferences.com
        $basePath = rtrim('/home/scisynopsisconfe/public_html', '/');

        // /dump
        $dumpPath = $basePath . '/dump';
        $dumpFiles = [];
        if (is_dir($dumpPath)) {
            $dumpFiles = array_values(array_filter(scandir($dumpPath), function ($f) {
                return $f !== '.' && $f !== '..';
            }));
        }

        // conference folder
        $folderName   = $conference->folder_name ?: '';
        $folderExists = false;
        $folderFiles  = [];

        if ($folderName !== '') {
            $confPath = $basePath . '/' . $folderName;
            if (is_dir($confPath)) {
                $folderExists = true;
                $folderFiles  = array_values(array_filter(scandir($confPath), function ($f) {
                    return $f !== '.' && $f !== '..';
                }));
            }
        }

        return response()->json([
            'status'        => true,
            'conference'    => $conference,
            'dump_files'    => $dumpFiles,
            'folder_files'  => $folderFiles,
            'folder_exists' => $folderExists,
        ]);
    }

    /**
     * Move-files ACTION: move all files from /dump to target folder
     */
    public function v4ConferenceMoveFiles($id, Request $request)
    {
        $conference = DB::table('conferences')->where('id', $id)->first();

        if (!$conference) {
            return response()->json([
                'status'  => false,
                'message' => 'Conference not found',
            ], 404);
        }

        $targetFolder = trim($request->input('target_folder', ''));
        $dryRun       = (int) $request->input('dry_run', 0) === 1;

        if ($targetFolder === '') {
            return response()->json([
                'status'  => false,
                'message' => 'Target folder is required',
            ], 422);
        }

        // ✅ use real conferences root
        $basePath  = rtrim('/home/scisynopsisconfe/public_html', '/');
        $dumpPath  = $basePath . '/dump';
        $targetDir = $basePath . '/' . $targetFolder;

        if (!is_dir($dumpPath)) {
            return response()->json([
                'status'  => false,
                'message' => '/dump folder does not exist at ' . $dumpPath,
            ], 500);
        }

        // create target folder if needed
        if (!is_dir($targetDir) && !$dryRun) {
            if (!mkdir($targetDir, 0775, true) && !is_dir($targetDir)) {
                return response()->json([
                    'status'  => false,
                    'message' => 'Failed to create target folder: ' . $targetFolder,
                ], 500);
            }
        }

        $files = array_values(array_filter(scandir($dumpPath), function ($f) {
            return $f !== '.' && $f !== '..';
        }));

        $log = [];
        foreach ($files as $file) {
            $from = $dumpPath . '/' . $file;
            $to   = $targetDir . '/' . $file;

            if ($dryRun) {
                $log[] = "[DRY-RUN] Would move: {$from} -> {$to}";
            } else {
                if (@rename($from, $to)) {
                    $log[] = "Moved: {$from} -> {$to}";
                } else {
                    $log[] = "FAILED: {$from} -> {$to}";
                }
            }
        }

        // only update folder_name (you said no URL update)
        DB::table('conferences')->where('id', $id)->update([
            'folder_name' => $targetFolder,
        ]);

        return response()->json([
            'status'  => true,
            'message' => $dryRun ? 'Dry-run completed' : 'Files moved successfully',
            'log'     => implode("\n", $log),
        ]);
    }

}
