<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\Conferenceadmin\User as AdminUser;

class V2Controller extends Controller
{
    protected function getAdminFromToken(Request $request)
    {
        $token = $request->bearerToken()
            ?? $request->header('authorization')
            ?? $request->input('token')
            ?? $request->query('token');

        if (!$token) {
            return null;
        }

        return AdminUser::where('api_token', $token)->first();
    }

    // v2 brochureusers()  -> user_type = 1
    public function brochureusers(Request $request)
    {
        $admin = $this->getAdminFromToken($request);
        if (!$admin) {
            // new pages: no hard 401
        }

        $conferenceId = $request->query('conference_id');
        if (!$conferenceId) {
            return response()->json(['message' => 'Conference ID missing'], 422);
        }

        $users = DB::table('c_reg_users')
            ->where('conference_id', $conferenceId)
            ->where('user_type', 1)
            ->orderByDesc('id')
            ->get();

        return response()->json([
            'type'  => 1,
            'users' => $users,
        ]);
    }

    // v2 sponsorshipusers()  -> user_type = 12
    public function sponsorshipusers(Request $request)
    {
        $admin = $this->getAdminFromToken($request);
        if (!$admin) {
            // new pages: no hard 401
        }

        $conferenceId = $request->query('conference_id');
        if (!$conferenceId) {
            return response()->json(['message' => 'Conference ID missing'], 422);
        }

        $users = DB::table('c_reg_users')
            ->where('conference_id', $conferenceId)
            ->where('user_type', 12)
            ->orderByDesc('id')
            ->get();

        return response()->json([
            'type'  => 12,
            'users' => $users,
        ]);
    }

// app/Http/Controllers/Conferenceadmin/V2Controller.php

public function brochureusersDelete(Request $request, $id)
{
    // same token auth style
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $admin = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$admin) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    // only usertype 1 or 4 can delete (like old CI3)
    if (!in_array((int) ($admin->usertype ?? 0), [1, 4], true)) {
        return response()->json(['message' => 'Forbidden'], 403);
    }

    $conferenceId = $request->query('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    $user = \DB::table('c_reg_users')
        ->where('id', $id)
        ->where('conference_id', $conferenceId)
        ->where('user_type', 1) // brochure users
        ->first();

    if (!$user) {
        return response()->json(['message' => 'User not found'], 404);
    }

    \DB::table('c_reg_users')
        ->where('id', $id)
        ->where('conference_id', $conferenceId)
        ->where('user_type', 1)
        ->delete();

    return response()->json(['message' => 'User deleted']);
}


    // 🔹 NEW: sponsorshipusersDelete() -> user_type = 12
    public function sponsorshipusersDelete(Request $request, $id)
    {
        $token = $request->bearerToken()
            ?? $request->header('authorization')
            ?? $request->input('token')
            ?? $request->query('token');

        $admin = AdminUser::where('api_token', $token)->first();
        if (!$admin) {
            // new pages: no hard 401
        }

        // same rule: only 1 or 4 can delete
        if (!in_array((int) ($admin->usertype ?? 0), [1, 4], true)) {
            return response()->json(['message' => 'Forbidden'], 403);
        }

        $conferenceId = $request->query('conference_id');
        if (!$conferenceId) {
            return response()->json(['message' => 'Conference ID missing'], 422);
        }

        $user = DB::table('c_reg_users')
            ->where('id', $id)
            ->where('conference_id', $conferenceId)
            ->where('user_type', 12) // sponsorship users
            ->first();

        if (!$user) {
            return response()->json(['message' => 'User not found'], 404);
        }

        DB::table('c_reg_users')
            ->where('id', $id)
            ->where('conference_id', $conferenceId)
            ->where('user_type', 12)
            ->delete();

        return response()->json(['message' => 'User deleted']);
    }
    
     public function abstractusers(Request $request)
    {
        $admin = $this->getAdminFromToken($request);
        if (!$admin) {
            // per your new pages style, no hard 401 here
        }

        $conferenceId = $request->query('conference_id');
        if (!$conferenceId) {
            return response()->json(['message' => 'Conference ID missing'], 422);
        }

        // Join with conferences to get folder_name, conf_unique_id, prev_conf, etc.
        $rows = DB::table('c_reg_users as u')
            ->join('conferences as c', 'c.id', '=', 'u.conference_id')
            ->where('u.conference_id', $conferenceId)
            ->where('u.user_type', 2) // 2 = Abstract Users
            ->orderByDesc('u.id')
            ->select(
                'u.*',
                'c.conference_name as conferenceName',
                'c.id as conferenceId',
                'c.folder_name',
                'c.unique_id as conf_unique_id',
                'c.prev_conf'
            )
            ->get();

        // Build file_url based on your rule
        $base = 'https://scisynopsisconferences.com';

        $users = $rows->map(function ($row) use ($base) {
            $path = '';

            if ((int) $row->prev_conf === 0) {
                // prev_conf = 0
                // if conferenceId == 124 or anything else → same pattern per your rule
                $path = "/uploads/conferences/{$row->folder_name}_{$row->conferenceId}/{$row->user_file}";
            } else {
                // prev_conf != 0
                $path = "/uploads/conferences/{$row->conf_unique_id}/{$row->user_file}";
            }

            $row->file_url = $base . $path;

            return $row;
        });

        return response()->json([
            'type'   => 2,
            'users'  => $users,
        ]);
    }

    /**
     * Delete a single Abstract User (user_type = 2)
     */
    public function abstractusersDelete(Request $request, $id)
    {
        $token = $request->bearerToken()
            ?? $request->header('authorization')
            ?? $request->input('token')
            ?? $request->query('token');

        $admin = AdminUser::where('api_token', $token)->first();
        if (!$admin) {
            // new pages: no strict 401
        }

        // only usertype 1 or 4 can delete (like old CI3)
        if (!in_array((int) ($admin->usertype ?? 0), [1, 4], true)) {
            return response()->json(['message' => 'Forbidden'], 403);
        }

        $conferenceId = $request->query('conference_id');
        if (!$conferenceId) {
            return response()->json(['message' => 'Conference ID missing'], 422);
        }

        $user = DB::table('c_reg_users')
            ->where('id', $id)
            ->where('conference_id', $conferenceId)
            ->where('user_type', 2)
            ->first();

        if (!$user) {
            return response()->json(['message' => 'User not found'], 404);
        }

        DB::table('c_reg_users')
            ->where('id', $id)
            ->where('conference_id', $conferenceId)
            ->where('user_type', 2)
            ->delete();

        return response()->json(['message' => 'User deleted']);
    }

    /**
     * Delete multiple Abstract Users (Delete All Selected)
     * expects: ids[] (array) or ids (comma-separated)
     */
    public function abstractusersDeleteAll(Request $request)
    {
        $token = $request->bearerToken()
            ?? $request->header('authorization')
            ?? $request->input('token')
            ?? $request->query('token');

        $admin = AdminUser::where('api_token', $token)->first();
        if (!$admin) {
            // new pages: no strict 401
        }

        if (!in_array((int) ($admin->usertype ?? 0), [1, 4], true)) {
            return response()->json(['message' => 'Forbidden'], 403);
        }

        $conferenceId = $request->input('conference_id') ?? $request->query('conference_id');
        if (!$conferenceId) {
            return response()->json(['message' => 'Conference ID missing'], 422);
        }

        $ids = $request->input('ids', []);

        if (!is_array($ids)) {
            $ids = array_filter(array_map('trim', explode(',', (string) $ids)));
        }

        if (empty($ids)) {
            return response()->json(['message' => 'No IDs provided'], 422);
        }

        DB::table('c_reg_users')
            ->whereIn('id', $ids)
            ->where('conference_id', $conferenceId)
            ->where('user_type', 2)
            ->delete();

        return response()->json(['message' => 'Selected users deleted']);
    }
    
    
    
     public function presentationusers(Request $request)
    {
        $admin = $this->getAdminFromToken($request);
        if (!$admin) {
            // per your new pages style, no hard 401
        }

        $conferenceId = $request->query('conference_id');
        if (!$conferenceId) {
            return response()->json(['message' => 'Conference ID missing'], 422);
        }

        // Join with conferences to get folder_name, conf_unique_id, prev_conf, etc.
        $rows = DB::table('c_reg_users as u')
            ->join('conferences as c', 'c.id', '=', 'u.conference_id')
            ->where('u.conference_id', $conferenceId)
            ->where('u.user_type', 5) // 5 = Presentation Users
            ->orderByDesc('u.id')
            ->select(
                'u.*',
                'c.conference_name as conferenceName',
                'c.id as conferenceId',
                'c.folder_name',
                'c.unique_id as conf_unique_id',
                'c.prev_conf'
            )
            ->get();

        // Build file_url based on your rule
        $base = 'https://scisynopsisconferences.com';

        $users = $rows->map(function ($row) use ($base) {
            $path = '';

            if ((int) $row->prev_conf === 0) {
                // prev_conf = 0 → /uploads/conferences/{folder_name}_{conferenceId}/{user_file}
                $path = "/uploads/conferences/{$row->folder_name}_{$row->conferenceId}/{$row->user_file}";
            } else {
                // prev_conf != 0 → /uploads/conferences/{conf_unique_id}/{user_file}
                $path = "/uploads/conferences/{$row->conf_unique_id}/{$row->user_file}";
            }

            $row->file_url = $base . $path;

            return $row;
        });

        return response()->json([
            'type'  => 5,
            'users' => $users,
        ]);
    }

    /**
     * Delete a single Presentation User (user_type = 5)
     */
    public function presentationusersDelete(Request $request, $id)
    {
        $token = $request->bearerToken()
            ?? $request->header('authorization')
            ?? $request->input('token')
            ?? $request->query('token');

        $admin = AdminUser::where('api_token', $token)->first();
        if (!$admin) {
            // no strict 401 for new pages
        }

        // only usertype 1 or 4 can delete (like old CI3)
        if (!in_array((int) ($admin->usertype ?? 0), [1, 4], true)) {
            return response()->json(['message' => 'Forbidden'], 403);
        }

        $conferenceId = $request->query('conference_id');
        if (!$conferenceId) {
            return response()->json(['message' => 'Conference ID missing'], 422);
        }

        $user = DB::table('c_reg_users')
            ->where('id', $id)
            ->where('conference_id', $conferenceId)
            ->where('user_type', 5)
            ->first();

        if (!$user) {
            return response()->json(['message' => 'User not found'], 404);
        }

        DB::table('c_reg_users')
            ->where('id', $id)
            ->where('conference_id', $conferenceId)
            ->where('user_type', 5)
            ->delete();

        return response()->json(['message' => 'User deleted']);
    }

    /**
     * Delete multiple Presentation Users (Delete All Selected)
     * expects: ids[] (array) or ids (comma-separated)
     */
    public function presentationusersDeleteAll(Request $request)
    {
        $token = $request->bearerToken()
            ?? $request->header('authorization')
            ?? $request->input('token')
            ?? $request->query('token');

        $admin = AdminUser::where('api_token', $token)->first();
        if (!$admin) {
            // no strict 401
        }

        if (!in_array((int) ($admin->usertype ?? 0), [1, 4], true)) {
            return response()->json(['message' => 'Forbidden'], 403);
        }

        $conferenceId = $request->input('conference_id') ?? $request->query('conference_id');
        if (!$conferenceId) {
            return response()->json(['message' => 'Conference ID missing'], 422);
        }

        $ids = $request->input('ids', []);

        if (!is_array($ids)) {
            $ids = array_filter(array_map('trim', explode(',', (string) $ids)));
        }

        if (empty($ids)) {
            return response()->json(['message' => 'No IDs provided'], 422);
        }

        DB::table('c_reg_users')
            ->whereIn('id', $ids)
            ->where('conference_id', $conferenceId)
            ->where('user_type', 5)
            ->delete();

        return response()->json(['message' => 'Selected users deleted']);
    }
    
    
    public function registernewusers(Request $request)
{
    $admin = $this->getAdminFromToken($request);

    $conferenceId = $request->query('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    $users = DB::table('c_reg_users')
        ->where('conference_id', $conferenceId)
        ->where('user_type', 3)
        ->orderByDesc('id')
        ->get();

    $result = $users->map(function ($row) {
        $regRow = DB::table('c_user_registrations_dates')
            ->where('user_id', $row->id)
            ->orderByDesc('id')
            ->first();

        $regTotal = 0;
        if ($regRow && $regRow->radio_amount) {
            $qty = $regRow->reg_qty ?? 1;
            $regTotal = (float)$regRow->radio_amount * (float)$qty;
        }

        $addonTotal = 0;
        $addons = DB::table('c_user_addon_amounts')
            ->where('user_id', $row->id)
            ->get();

        foreach ($addons as $a) {
            $addonTotal += ((float)($a->addon_amount ?? 0)) * ((float)($a->addon_qty ?? 0));
        }

        $acpRow = DB::table('c_user_registrations_acp')
            ->where('user_id', $row->id)
            ->first();
        $acpTotal = $acpRow ? (float)$acpRow->acptotal_amount : 0.0;

        $espnRow = DB::table('c_user_registrations_espn')
            ->where('user_id', $row->id)
            ->first();
        $espnTotal = $espnRow ? (float)$espnRow->espntotal_amount : 0.0;

        $subtottal5 = $regTotal + $addonTotal + $espnTotal + $acpTotal;

        $amount = 0.0;
        if ((int)$row->user_type === 3) {
            if ((string)($row->payment_meathod ?? '') === '2') {
                $amount = $subtottal5 + ($subtottal5 * 3 / 100);
            } else {
                $amount = $subtottal5;
            }
        } else {
            $amount = $regRow ? (float)($regRow->total_amount ?? 0) : 0.0;
        }

        $paymentStatus = $regRow->payment_status ?? '';

        $row->regInfo = $regTotal;
        $row->addonInfo = $addonTotal;
        $row->acptotal_amount = $acpTotal;
        $row->espntotal_amount = $espnTotal;
        $row->amount = $amount;
        $row->payment_status = $paymentStatus;

        return $row;
    });

    return response()->json([
        'type'  => 3,
        'users' => $result,
    ]);
}

public function registernewusersDelete(Request $request, $id)
{
    $admin = $this->getAdminFromToken($request);
    if (!$admin) {}

    if ((int)($admin->usertype ?? 0) !== 1) {
        return response()->json(['message' => 'Forbidden'], 403);
    }

    $conferenceId = $request->query('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    $user = DB::table('c_reg_users')
        ->where('id', $id)
        ->where('conference_id', $conferenceId)
        ->where('user_type', 3)
        ->first();

    if (!$user) {
        return response()->json(['message' => 'User not found'], 404);
    }

    DB::table('c_reg_users')
        ->where('id', $id)
        ->where('conference_id', $conferenceId)
        ->where('user_type', 3)
        ->delete();

    return response()->json(['message' => 'User deleted']);
}
public function registeredusers(Request $request)
{
    $admin = $this->getAdminFromToken($request);

    $conferenceId = $request->query('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    $users = DB::table('c_reg_users')
        ->where('conference_id', $conferenceId)
        ->where('user_type', 3)
        ->orderByDesc('id')
        ->get();

    $result = $users->map(function ($row) {
        $regRow = DB::table('c_user_registrations_dates')
            ->where('user_id', $row->id)
            ->orderByDesc('id')
            ->first();

        $regTotal = 0;
        if ($regRow && $regRow->radio_amount) {
            $qty = $regRow->reg_qty ?? 1;
            $regTotal = (float)$regRow->radio_amount * (float)$qty;
        }

        $addonTotal = 0;
        $addons = DB::table('c_user_addon_amounts')
            ->where('user_id', $row->id)
            ->get();

        foreach ($addons as $a) {
            $addonTotal += ((float)($a->addon_amount ?? 0)) * ((float)($a->addon_qty ?? 0));
        }

        $acpRow = DB::table('c_user_registrations_acp')
            ->where('user_id', $row->id)
            ->first();
        $acpTotal = $acpRow ? (float)$acpRow->acptotal_amount : 0.0;

        $espnRow = DB::table('c_user_registrations_espn')
            ->where('user_id', $row->id)
            ->first();
        $espnTotal = $espnRow ? (float)$espnRow->espntotal_amount : 0.0;

        $subtottal5 = $regTotal + $addonTotal + $espnTotal + $acpTotal;

        $amount = 0.0;
        if ((int)$row->user_type === 3) {
            if ((string)($row->payment_meathod ?? '') === '2') {
                $amount = $subtottal5 + ($subtottal5 * 3 / 100);
            } else {
                $amount = $subtottal5;
            }
        }

        $paymentStatus = $regRow->payment_status ?? '';

        $row->regInfo = $regTotal;
        $row->addonInfo = $addonTotal;
        $row->acptotal_amount = $acpTotal;
        $row->espntotal_amount = $espnTotal;
        $row->amount = $amount;
        $row->payment_status = $paymentStatus;

        return $row;
    });

    return response()->json([
        'type'  => 3,
        'users' => $result,
    ]);
}

public function registeredusersDelete(Request $request, $id)
{
    $admin = $this->getAdminFromToken($request);
    if (!$admin) {}

    if ((int)($admin->usertype ?? 0) !== 1) {
        return response()->json(['message' => 'Forbidden'], 403);
    }

    $conferenceId = $request->query('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    $user = DB::table('c_reg_users')
        ->where('id', $id)
        ->where('conference_id', $conferenceId)
        ->where('user_type', 3)
        ->first();

    if (!$user) {
        return response()->json(['message' => 'User not found'], 404);
    }

    DB::table('c_reg_users')
        ->where('id', $id)
        ->where('conference_id', $conferenceId)
        ->where('user_type', 3)
        ->delete();

    return response()->json(['message' => 'User deleted']);
}

public function discountusers(Request $request)
{
    $admin = $this->getAdminFromToken($request);

    $conferenceId = $request->query('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    $rows = DB::table('c_reg_users as u')
        ->leftJoin('c_user_orders as o', 'o.user_id', '=', 'u.id')
        ->where('u.conference_id', $conferenceId)
        ->where('u.user_type', 4)
        ->orderByDesc('u.id')
        ->select(
            'u.*',
            'o.curency_type',
            'o.coupon_amount',
            'o.payment_status as order_payment_status'
        )
        ->get();

    $users = $rows->map(function ($row) {
        $row->curency_type = (int)($row->curency_type ?? 0);
        $row->coupon_amount = (float)($row->coupon_amount ?? 0);
        $row->payment_status = $row->order_payment_status ?: 'Pending';
        return $row;
    });

    return response()->json([
        'type'  => 4,
        'users' => $users,
    ]);
}

public function discountusersDelete(Request $request, $id)
{
    $admin = $this->getAdminFromToken($request);
    if (!$admin) {}

    if ((int)($admin->usertype ?? 0) !== 1) {
        return response()->json(['message' => 'Forbidden'], 403);
    }

    $conferenceId = $request->query('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    $user = DB::table('c_reg_users')
        ->where('id', $id)
        ->where('conference_id', $conferenceId)
        ->where('user_type', 4)
        ->first();

    if (!$user) {
        return response()->json(['message' => 'User not found'], 404);
    }

    DB::table('c_reg_users')
        ->where('id', $id)
        ->where('conference_id', $conferenceId)
        ->where('user_type', 4)
        ->delete();

    return response()->json(['message' => 'User deleted']);
}



public function onlineusers(Request $request)
{
    $admin = $this->getAdminFromToken($request);

    $conferenceId = $request->query('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    $rows = DB::table('c_reg_users as u')
        ->where('u.conference_id', $conferenceId)
        ->where('u.user_type', 6)
        ->orderByDesc('u.id')
        ->get();

    $users = $rows->map(function ($row) {
        $regRow = DB::table('c_user_registrations_dates')
            ->where('user_id', $row->id)
            ->orderByDesc('id')
            ->first();

        $row->amount = (float)($row->total_amount ?? 0);
        $row->payment_status = $regRow->payment_status ?? '';

        return $row;
    });

    return response()->json([
        'type'  => 6,
        'users' => $users,
    ]);
}

public function onlineusersDelete(Request $request, $id)
{
    $admin = $this->getAdminFromToken($request);
    if (!$admin) {}

    if ((int)($admin->usertype ?? 0) !== 1) {
        return response()->json(['message' => 'Forbidden'], 403);
    }

    $conferenceId = $request->query('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    $user = DB::table('c_reg_users')
        ->where('id', $id)
        ->where('conference_id', $conferenceId)
        ->where('user_type', 6)
        ->first();

    if (!$user) {
        return response()->json(['message' => 'User not found'], 404);
    }

    DB::table('c_reg_users')
        ->where('id', $id)
        ->where('conference_id', $conferenceId)
        ->where('user_type', 6)
        ->delete();

    return response()->json(['message' => 'User deleted']);
}


public function manualpaymentusers(Request $request)
{
    $admin = $this->getAdminFromToken($request);

    $conferenceId = $request->query('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    $rows = DB::table('c_reg_users as u')
        ->where('u.conference_id', $conferenceId)
        ->where('u.user_type', 7)
        ->orderByDesc('u.id')
        ->get();

    $users = $rows->map(function ($row) {
        $qty = $row->qty ? (int)$row->qty : 1;
        $extra = $row->total_extra_amount ? (float)$row->total_extra_amount : 0;
        $base = (float)($row->total_amount ?? 0);

        $row->amount = $base * $qty + $extra;

        $regRow = DB::table('c_user_registrations_dates')
            ->where('user_id', $row->id)
            ->orderByDesc('id')
            ->first();

        $row->payment_status = $regRow->payment_status ?? '';

        return $row;
    });

    return response()->json([
        'type'  => 7,
        'users' => $users,
    ]);
}

public function manualpaymentusersDelete(Request $request, $id)
{
    $admin = $this->getAdminFromToken($request);
    if (!$admin) {}

    if ((int)($admin->usertype ?? 0) !== 1) {
        return response()->json(['message' => 'Forbidden'], 403);
    }

    $conferenceId = $request->query('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    $user = DB::table('c_reg_users')
        ->where('id', $id)
        ->where('conference_id', $conferenceId)
        ->where('user_type', 7)
        ->first();

    if (!$user) {
        return response()->json(['message' => 'User not found'], 404);
    }

    DB::table('c_reg_users')
        ->where('id', $id)
        ->where('conference_id', $conferenceId)
        ->where('user_type', 7)
        ->delete();

    return response()->json(['message' => 'User deleted']);
}
public function groupregistrationusers(Request $request)
{
    $conference_id = $request->conference_id;

    $config = DB::table('configurations')
        ->where('status', 1)
        ->orderBy('id', 'DESC')
        ->get();

    $users = DB::table('c_reg_users AS u')
        ->where('u.user_type', 8)
        ->where('u.conference_id', $conference_id)
        ->orderBy('u.id', 'DESC')
        ->get();

    $result = [];

    foreach ($users as $u) {

        // Payment record
        $payment = DB::table('c_user_group_registrations_dates')
            ->where('user_id', $u->id)
            ->get();

        // Conference Info
        $conf = DB::table('conferences')
            ->where('id', $u->conference_id)
            ->first();

        $result[] = [
            "id"             => $u->id,
            "unique_id"      => $u->unique_id,
            "designation"    => $u->designation,
            "firstname"      => $u->firstname,
            "lastname"       => $u->lastname,
            "usr_email"      => $u->usr_email,
            "usr_alternate_email" => $u->usr_alternate_email,
            "user_phone"     => $u->user_phone,
            "usr_whatsapp"   => $u->usr_whatsapp,

            "city"           => $u->city,
            "country"        => $u->country,
            "country_code"   => $u->country_code,
            "company"        => $u->company,
            "category"       => $u->category,
            "absCategory"    => $u->absCategory,

            "total_amount"   => $u->total_amount,
            "currency"       => $u->currency,
            "payment_method" => $u->payment_method,
            "payment_status" => $u->payment_status,

            "created_date"   => $u->created_date,

            "conference_name" => @$conf->conference_name,
            "conferenceId"    => @$conf->id,
            "folder_name"     => @$conf->folder_name,
            "conf_unique_id"  => @$conf->unique_id,
            "prev_conf"       => @$conf->prev_conf,

            "payment"         => $payment
        ];
    }

    return response()->json([
        "users"          => $result,
        "configurations" => $config
    ]);
}
public function groupregistrationusersDelete($id)
{
    DB::table('c_reg_users')->where('id', $id)->delete();
    DB::table('c_user_group_registrations_dates')->where('user_id', $id)->delete();

    return response()->json([
        "status" => "success",
        "message" => "Group Registered User deleted successfully"
    ]);
}


}
