<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Stripe\Stripe;
use Stripe\PaymentIntent;
use Illuminate\Support\Facades\Log;

class GroupRegistrationApiController extends Controller
{
    public function indexV5($id, $currency = 3)
    {
        $currency = in_array((int)$currency, [1,2,3]) ? (int)$currency : 3;

        $dates  = DB::table('c_registrations_dates')->where(['status'=>1,'conference_id'=>$id])->get();
        $amount = DB::table('c_group_registrations_amount')->where(['status'=>1,'conference_id'=>$id,'currency_id'=>$currency])->orderBy('id')->get();
        $onsite = DB::table('c_onsite_reg_status')->where('conference_id',$id)->first();
        $design = DB::table('designations')->where('status',1)->orderBy('id')->get();
        $countries = DB::table('countries')->where('status',1)->orderBy('country_name')->get();
        $captcha = rand(100000,999999);
    $categories = DB::table('categories')
                ->where('status', 1)
                ->orderBy('id', 'ASC')
                ->get();
        return response()->json([
            'success'=>true,
            'data'=>[
                'conferenceId'=>$id,
                'currency'=>$currency,
                'dates'=>$dates,
                'registrationAmount'=>$amount,
                'onsiteStatus'=>$onsite,
                'designations'=>$design,
                'countries'=>$countries,
                'newcaptcha'=>$captcha,
                  'categories'=>$categories,
            ]
        ]);
    }

    public function getPhoneCodeV5($name)
    {
        $c = DB::table('countries')->where(['status'=>1,'country_name'=>$name])->first();
        if(!$c || empty($c->country_code)) return response()->json(['success'=>false,'message'=>'Country code not found'],404);
        return response()->json(['success'=>true,'phonecode'=>'+'.$c->country_code]);
    }

  public function submitV5(Request $r)
{
    $orderId  = 'GRP-'.strtoupper(uniqid());
    $currency = in_array((int)($r->currsel ?? 3), [1,2,3]) ? (int)$r->currsel : 3;

    // Accept both payload shapes:
    $sel      = is_array($r->selection ?? null) ? $r->selection : [];
    $qty      = (int)($sel['qty']       ?? $r->selected_qty ?? $r->qty ?? 0);
    $planId   = (int)($sel['plan_id']   ?? $r->selected_id  ?? $r->plan_id ?? 0);
    $totalAmt = (float)($r->totalAmount ?? $r->total_Amount ?? $r->total_amount ?? 0.0);

    if ($qty <= 0 || $planId <= 0) {
        return response()->json(['success'=>false,'message'=>'Invalid selection or quantity'], 422);
    }

    DB::beginTransaction();
    try {
        // group lead
        $countryCode = DB::table('countries')->where('country_name',$r->country)->value('country_code') ?? '';

        $userId = DB::table('c_reg_users')->insertGetId([
            'conference_id'  => $r->conferenceId,
            'designation'    => $r->designation,
            'firstname'      => $r->firstname,
            'lastname'       => $r->lastname,
            'usr_email'      => $r->usr_email,
            'country'        => $r->country,
            'country_code'   => $countryCode,
            'user_phone'     => $r->user_phone,
            'usr_whatsapp'   => $r->whatsapp_number,
            'company'        => $r->company,
            'category'       => $r->category,
            'address'        => $r->address,
            'user_type'      => 8, // group
            'currency'       => $currency,
            'payment_method' => $r->payment_meathod,
            'total_amount'   => $totalAmt,
            'created_date'   => now(),
        ]);

        // billing/shipping
        DB::table('c_registration_shipping')->insert([
            'regid'         => $userId,
            'conference_id' => $r->conferenceId,
            'shipDesign'    => $r->shipdesignation,
            'shipFirstname' => $r->shipfirstname,
            'shipLastname'  => $r->shiplastname,
            'shipCompany'   => $r->shipcompany,
            'shipCountry'   => $r->shipcountry,
            'shipEmail'     => $r->shipusr_email,
            'shipPhone'     => $r->shipphone,
            'shipWhatsapp'  => $r->shipwhatsapp_number,
            'shipAddress'   => $r->shipaddress,
            'createDate'    => now(),
        ]);

        // compute unit based on active window
        $now   = date('Y-m-d');
        $row   = DB::table('c_group_registrations_amount')->where([
                    'conference_id'=>$r->conferenceId,'currency_id'=>$currency,'id'=>$planId,'status'=>1
                ])->first();
        $dates = DB::table('c_registrations_dates')->where('conference_id',$r->conferenceId)->first();

        $unit = 0.0;
        if ($row && $dates) {
            if (!empty($dates->early_date)        && strtotime($dates->early_date)        >= strtotime($now)) $unit = (float)$row->early_bird_amount;
            else if (!empty($dates->standard_date) && strtotime($dates->standard_date)     >= strtotime($now)) $unit = (float)$row->first_call_amount;
            else if (!empty($dates->onsite_date)   && strtotime($dates->onsite_date)       >= strtotime($now)) $unit = (float)$row->final_call_amount;
            else if (!empty($dates->onsite_reg_date) && strtotime($dates->onsite_reg_date) >= strtotime($now)) $unit = (float)$row->onsite_registration;
        }
        $regTotal = round($qty * $unit, 2);

        DB::table('c_user_group_registrations_dates')->insert([
            'order_id'        => $orderId,
            'user_id'         => $userId,
            'earlyType'       => null,
            'registration_id' => $planId,
            'reg_qty'         => $qty,
            'reg_total_amount'=> $regTotal,
            'total_amount'    => $totalAmt,
            'curency_type'    => $currency,
            'payment_meathod' => $r->payment_meathod,
            'reg_date'        => now(),
            'created_date'    => now(),
        ]);

        DB::commit();

        return response()->json([
            'success'=>true,
            'order_id'=>$orderId,
            'total'=>$totalAmt,
            'currency'=>$this->code($currency),
            'symbol'=>$this->sym($currency),
        ]);
    } catch (\Throwable $e) {
        DB::rollBack();
        \Log::error('group submit v5 failed: '.$e->getMessage());
        return response()->json(['success'=>false,'message'=>'Submit failed'],500);
    }
}


    public function createPaymentIntentV5(Request $r)
    {
        $r->validate([
            'order_id'=>'required|string',
            'amount'=>'required|integer|min:1',
            'currency'=>'required|string|in:usd,eur,gbp',
            'user.name'=>'required|string',
            'user.email'=>'required|email',
            'conference_id'=>'required|integer',
        ]);

        Stripe::setApiKey(env('STRIPE_SECRET_KEY'));
        $conf = DB::table('conferences')->where('id',$r->conference_id)->first();
        if(!$conf) return response()->json(['error'=>'Conference not found'],404);

        $confName = trim(($conf->conference_first_title ?? '').' '.($conf->conference_name ?? ''));
        $intent = PaymentIntent::create([
            'amount'=>$r->amount,
            'currency'=>$r->currency,
            'description'=>"Group Registration: {$confName}",
            'metadata'=>[
                'order_id'=>$r->order_id,
                'user_name'=>$r->user['name'],
                'user_email'=>$r->user['email'],
                'conference_id'=>$r->conference_id,
                'conference_name'=>$confName,
            ],
            'payment_method_types'=>['card'],
        ]);
        return response()->json(['clientSecret'=>$intent->client_secret]);
    }

public function stripeReturnV5(Request $request)
{
    \Stripe\Stripe::setApiKey(env('STRIPE_SECRET_KEY'));
    $piId = $request->query('payment_intent');
    if(!$piId) {
        $frontend = rtrim(config('app.frontend_url', env('FRONTEND_URL','/')), '/');
        return redirect($frontend.'/group-payment-failed-v5');
    }

    try {
        $pi = \Stripe\PaymentIntent::retrieve($piId);
        $orderId = $pi->metadata['order_id'] ?? null;
        if(!$orderId) {
            $frontend = rtrim(config('app.frontend_url', env('FRONTEND_URL','/')), '/');
            return redirect($frontend.'/group-payment-failed-v5');
        }

        $newStatus = $pi->status === 'succeeded' ? 'Completed' : 'Pending';

        DB::table('c_user_group_registrations_dates')
          ->where('order_id',$orderId)
          ->update([
            'payment_status'    => $newStatus,
            'payment_meathod'   => 1,        // Stripe
            'payment_intent_id' => $pi->id,
          ]);

        $row = DB::table('c_user_group_registrations_dates')->where('order_id',$orderId)->first();
        if ($row) {
            DB::table('c_reg_users')->where('id',$row->user_id)->update([
                'payment_status'=>$newStatus,
                'transaction_id'=>$orderId,
            ]);
        }

        $cookie = cookie('scs_group_order', $orderId, 5, null, null, false, false, false, 'Lax');
        $frontend = rtrim(config('app.frontend_url', env('FRONTEND_URL','/')), '/');
        return redirect($frontend.'/group-payment-success-v5')->withCookie($cookie);

    } catch (\Exception $e) {
        \Log::error('stripe group v5 return failed: '.$e->getMessage());
        $frontend = rtrim(config('app.frontend_url', env('FRONTEND_URL','/')), '/');
        return redirect($frontend.'/group-payment-failed-v5');
    }
}


    public function getPaymentSuccessDataV5($order_id)
    {
        $d = DB::table('c_reg_users as u')
            ->join('c_user_group_registrations_dates as r','u.id','=','r.user_id')
            ->where('r.order_id',$order_id)
            ->select(
                'u.firstname','u.lastname','u.usr_email','u.user_phone','u.company','u.country',
                'u.invoiceid',
                'r.order_id','r.total_amount','r.curency_type as currency_id','r.payment_meathod as payment_method',
                'r.payment_status','u.transaction_id','r.created_date'
            )->first();

        if(!$d) return response()->json(['success'=>false,'message'=>'Order not found'],404);

        return response()->json([
            'success'=>true,
            'data'=>[
                'order_id'=>$d->order_id,
                'firstname'=>$d->firstname,
                'lastname'=>$d->lastname,
                'usr_email'=>$d->usr_email,
                'user_phone'=>$d->user_phone,
                'company'=>$d->company,
                'country'=>$d->country,
                'total_amount'=>$d->total_amount,
                'currency'=>$this->code((int)$d->currency_id),
                'payment_method'=>(int)$d->payment_method,
                'payment_status'=>$d->payment_status,
                'transaction_id'=>$d->transaction_id,
                'invoiceid'=>$d->invoiceid,
                'created_date'=>$d->created_date,
            ]
        ]);
    }

    public function summaryV5($order_id)
    {
        // minimal: just reuse success payload
        return $this->getPaymentSuccessDataV5($order_id);
    }

    private function code($id){ return [1=>'GBP',2=>'USD',3=>'EUR'][$id] ?? 'EUR'; }
    private function sym($id){ return [1=>'£',2=>'$',3=>'€'][$id] ?? '€'; }
    
    


public function paypalIpnV5(Request $request)
{
    // 0) Read raw body (needed for PayPal verification)
    $raw = file_get_contents('php://input');
    Log::info('PayPal IPN hit', $request->all());

    // 1) Verify with PayPal
    $sandbox = filter_var(env('PAYPAL_SANDBOX', true), FILTER_VALIDATE_BOOLEAN);
    $verifyUrl = $sandbox
        ? 'https://ipnpb.sandbox.paypal.com/cgi-bin/webscr'
        : 'https://ipnpb.paypal.com/cgi-bin/webscr';

    $ch = curl_init($verifyUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST           => true,
        CURLOPT_POSTFIELDS     => 'cmd=_notify-validate&'.$raw,
        CURLOPT_HTTPHEADER     => ['Content-Type: application/x-www-form-urlencoded', 'Connection: Close'],
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_SSL_VERIFYHOST => 2,
        CURLOPT_TIMEOUT        => 30,
    ]);
    $resp = curl_exec($ch);
    $cerr = curl_error($ch);
    curl_close($ch);

    if ($resp !== 'VERIFIED') {
        Log::warning('PayPal IPN not VERIFIED', ['resp' => $resp, 'err' => $cerr]);
        return response('INVALID', 400);
    }

    // 2) Parse payload (from raw body to match verification)
    parse_str($raw, $p);
    if (empty($p)) $p = $request->all();

    // 3) Receiver email guard
    $recv = $p['receiver_email'] ?? $p['business'] ?? '';
    $expect = $sandbox ? env('PAYPAL_BUSINESS_SANDBOX', '') : env('PAYPAL_BUSINESS_LIVE', '');
    if ($expect && strcasecmp(trim($recv), trim($expect)) !== 0) {
        Log::warning('IPN receiver_email mismatch', compact('recv','expect'));
        return response('BAD_RECEIVER', 400);
    }

    // 4) Extract essentials
    $status  = $p['payment_status'] ?? '';
    $txnId   = $p['txn_id'] ?? '';
    $orderId = $p['custom']
            ?? $p['invoice']
            ?? $p['item_number1']
            ?? $p['item_number_1']
            ?? $p['item_number']
            ?? null;

    Log::info('IPN parsed', ['status' => $status, 'orderId' => $orderId, 'txn' => $txnId]);
    if (!$orderId) return response('NO_ORDER', 400);

    $reg = DB::table('c_user_group_registrations_dates')->where('order_id', $orderId)->first();
    if (!$reg) {
        Log::warning('IPN order not found', compact('orderId'));
        return response('NO_MATCH', 404);
    }

    $newStatus = $status === 'Completed' ? 'Completed' : 'Pending';

    // 5) Persist (only known columns; set txn if present)
    DB::beginTransaction();
    try {
        $update = ['payment_status' => $newStatus, 'payment_meathod' => 2];
        if (!empty($txnId)) $update['transaction_id'] = $txnId;

        DB::table('c_user_group_registrations_dates')
          ->where('order_id', $orderId)
          ->update($update);

        DB::table('c_reg_users')
          ->where('id', $reg->user_id)
          ->update([
              'payment_status' => $newStatus,
              'transaction_id' => $txnId ?: $orderId,
          ]);

        DB::commit();
    } catch (\Throwable $e) {
        DB::rollBack();
        Log::error('IPN DB update failed', ['orderId' => $orderId, 'err' => $e->getMessage()]);
        return response('DB_ERR', 500);
    }

    Log::info('IPN updated', [
        'orderId' => $orderId,
        'status'  => DB::table('c_user_group_registrations_dates')->where('order_id',$orderId)->value('payment_status')
    ]);

    return response('OK', 200);
}



}
