<?php

namespace App\Http\Controllers\Api;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\File;
use App\Http\Controllers\Controller;

class AbstractSubmissionApiController extends Controller
{
    public function show($id)
    {
        $conferenceInfo = DB::table('conferences')->find($id);
        if (!$conferenceInfo) {
            return response()->json(['error' => 'Conference not found'], 404);
        }

        $meta = DB::table('c_meta_data')
            ->where(['conference_id' => $id, 'page_type' => 14])
            ->first();

        return response()->json([
            'success' => true,
            'conferenceInfo' => $conferenceInfo,
            'headerData' => $this->getHeaderData($id),
            'footerData' => $this->getFooterData($id),
            'sampleabstract' => DB::table('c_brouchers')
                ->where(['status' => 1, 'page_type' => 2, 'conference_id' => $id])
                ->orderBy('id', 'ASC')
                ->first(),
            'sessionTracks' => DB::table('c_session_tracks')
                ->where(['status' => 1, 'conference_id' => $id])
                ->orderBy('orderpos', 'ASC')
                ->get(),
            'abstractCategories' => DB::table('abstract_categories')
                ->where('status', 1)
                ->orderBy('id', 'ASC')
                ->get(),
            'countriesList' => DB::table('countries')
                ->where('status', 1)
                ->orderBy('country_name', 'ASC')
                ->get(),
            'meta' => [
                'title' => $meta->meta_title ?? $meta->title ?? '',
                'description' => $meta->meta_description ?? '',
                'keywords' => $meta->meta_keywords ?? ''
            ],
            'favicon' => $conferenceInfo->fav_icon ?? ''
        ]);
    }

    private function getHeaderData($id)
    {
        return [
            'conferenceInfo' => DB::table('conferences')->find($id),
            'headercontent' => DB::table('c_header_content')->where('conference_id', $id)->first(),
            'headerNavigation' => DB::table('c_page_navigations')
                ->where(['conference_id' => $id, 'page_type' => 1, 'status' => 1, 'parent_id' => 0])
                ->orderBy('orderpos')->get(),
            'socialLinks' => DB::table('c_social')->where(['conference_id' => $id, 'status' => 1])->get(),
        ];
    }

    private function getFooterData($id)
    {
        return [
            'quickLinks' => DB::table('c_page_navigations')
                ->where(['conference_id' => $id, 'page_type' => 2, 'status' => 1])
                ->orderBy('orderpos')->get(),
            'usefulLinks' => DB::table('c_page_navigations')
                ->where(['conference_id' => $id, 'page_type' => 3, 'status' => 1])
                ->orderBy('orderpos')->get(),
            'socialLinks' => DB::table('c_social')->where(['conference_id' => $id, 'status' => 1])->get(),
            'mainaddress' => DB::table('web_cms')->where(['page_type' => 10, 'status' => 1])->first(),
            'contactInfo' => DB::table('c_address')->where(['conference_id' => $id, 'status' => 1])->first(),
            'whatsappNumber' => DB::table('c_address')->where('conference_id', $id)->value('whatsapp_number'),
        ];
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'conference_id' => 'required|integer',
            'designation' => 'required|string|max:100',
            'abstractcategory' => 'required|string|max:255',
            'abstract_title' => 'required|string|max:255',
            'firstname' => 'required|string|max:100',
            'company' => 'required|string|max:255',
            'country' => 'required|string|max:100',
            'usr_email' => 'required|email',
            'user_phone' => 'required|string|max:50',
            'abstract_message' => 'nullable|string',
            'session_id' => 'required|string|max:255',
            'user_file' => 'nullable|file|max:10240|mimes:pdf,doc,docx',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        if ((int)$request->captcha_answer !== (int)$request->captcha_expected) {
            return response()->json(['error' => 'Invalid captcha'], 422);
        }

        $conferenceId = $request->conference_id;
        $conference = DB::table('conferences')->where('id', $conferenceId)->first();
        if (!$conference) {
            return response()->json(['error' => 'Invalid Conference'], 404);
        }

        $docfile = '';
        if ($request->hasFile('user_file')) {
            $file = $request->file('user_file');
            $docfile = time() . '.' . $file->getClientOriginalExtension();
            $uploadPath = public_path("uploads/conferences/{$conference->folder_name}_{$conferenceId}");
            if (!File::exists($uploadPath)) {
                File::makeDirectory($uploadPath, 0777, true);
            }
            $file->move($uploadPath, $docfile);
        }

        $params = [
            'unique_id' => now()->format('smdiYH'),
            'conference_id' => $conferenceId,
            'designation' => $request->designation,
            'absCategory' => $request->abstractcategory,
            'abstract_title' => $request->abstract_title,
            'firstname' => $request->firstname,
            'lastname' => $request->lastname,
            'country' => $request->country,
            'company' => $request->company,
            'usr_email' => strtolower($request->usr_email),
            'user_phone' => $request->user_phone,
            'usr_whatsapp' => $request->whatsapp_no,
            'sessions_id' => $request->session_id,
            'user_type' => 2,
            'user_file' => $docfile,
            'queries' => $request->abstract_message,
            'created_date' => now(),
        ];
        DB::table('c_reg_users')->insert($params);

        return response()->json(['message' => 'Thank you submitting Abstract. We will get back to you soon']);
    }
    
    public function getPhonecode($countryName)
{
    $countryCode = DB::table('countries')
        ->where('country_name', $countryName)
        ->value('country_code');

    return response()->json($countryCode ?: '');
}
}