<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Conference;
use App\Models\Cspeaker;
use App\Models\PreviousSpeaker;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class SpeakersPageApiController extends Controller
{
    public function show(Request $request, $id): JsonResponse
    {
        $speaker_type = $request->query('speaker_type', '6');

        // Map speaker_type to page_type and other values
        $config = $this->getSpeakerConfig($speaker_type);

        $conference = Conference::with([
            'banners' => fn($q) => $q->where('page_type', $config['page_type'])->where('status', 1)->orderByDesc('id'),
            'SpeakersPagemetaData' => fn($q) => $q->where('page_type', $config['page_type'])
        ])->find($id);

        if (!$conference) {
            return response()->json(['error' => 'Conference not found'], 404);
        }

        $metaRow = $conference->SpeakersPagemetaData()->first();

        // Preload current speakers with countries
        $speakers = Cspeaker::with('country')
            ->where([
                'conference_id' => $id,
                'speaker_type' => $speaker_type
            ])
            ->orderBy('orderpos', 'ASC')
            ->get();

        $cmsInfo = $speakers->map(function ($s) {
            return [
                'id' => $s->id,
                'alias_name' => $s->alias_name,
                'speaker_img' => $s->speaker_img_url, // auto-appended via accessor
                'speaker_name' => $s->speaker_name,
                'speaker_title' => $s->speaker_title,
                'affliation' => $s->affliation,
                'speaker_type' => $s->speaker_type,
                'countryInfo' => $s->country ? [['country_name' => $s->country->country_name]] : []
            ];
        })->values();

        // Preload previous speakers
        $previousSpeakers = PreviousSpeaker::with(['previousConference', 'speakers.country'])
            ->where([
                'conference_id' => $id,
                'speaker_type' => $speaker_type
            ])
            ->orderBy('orderpos', 'ASC')
            ->get()
            ->map(function ($row) {
                $prevConf = $row->previousConference;
                if (!$prevConf) return null;

                $speakersInfo = $row->speakers->map(function ($s) {
                    return [
                        'id' => $s->id,
                        'alias_name' => $s->alias_name,
                        'speaker_img' => $s->speaker_img_url, // auto-appended via accessor
                        'speaker_name' => $s->speaker_name,
                        'affliation' => $s->affliation,
                        'speaker_title' => $s->speaker_title,
                        'conference_id' => $s->conference_id,
                        'countryInfo' => $s->country ? [['country_name' => $s->country->country_name]] : [],
                        'universityInfo' => [],
                        'hospitalInfo' => []
                    ];
                })->values();

                return [
                    'id' => $row->id,
                    'conference_id' => $row->conference_id,
                    'previous_conference_id' => $row->previous_conference_id,
                    'orderpos' => $row->orderpos,
                    'status' => $row->status,
                    'prevConfInfo' => [
                        [
                            'id' => $prevConf->id,
                            'conference_name' => $prevConf->conference_name,
                            'unique_id' => $prevConf->unique_id,
                            'folder_name' => $prevConf->folder_name
                        ]
                    ],
                    'speakersInfo' => $speakersInfo
                ];
            })->filter();

        return response()->json([
            'success' => true,
            'cntlrname' => $config['cntlrname'],
            'clmName' => $config['clmName'],
            'clmName1' => $config['clmName1'],
            'clmName2' => $config['clwName2'],
            'speaker_type' => $speaker_type,
            'cmsInfo' => $cmsInfo,
            'previousSpeakers' => $previousSpeakers,
            'banners' => $conference->banners,
            'meta' => [
                'title' => $metaRow?->meta_title ?? $config['defaultMeta']['title'],
                'description' => $metaRow?->meta_description ?? $config['defaultMeta']['description'],
                'keywords' => $metaRow?->meta_keywords ?? $config['defaultMeta']['keywords']
            ],
            'conferenceInfo' => $conference ? [$conference] : []
        ]);
    }

    private function getSpeakerConfig($speaker_type)
    {
        return match ($speaker_type) {
            '2' => [
                'page_type' => 7,
                'clmName' => 'Keynote Speakers',
                'clmName1' => 'Keynote',
                'clwName2' => 'Speakers',
                'cntlrname' => 'keynote-speakers',
                'defaultMeta' => [
                    'title' => 'Keynote Speakers – Conference',
                    'description' => 'Our distinguished keynote speakers.',
                    'keywords' => 'keynote speakers, conference, experts'
                ]
            ],
            default => [
                'page_type' => 24,
                'clmName' => 'EminentSpeakers',
                'clmName1' => 'Eminent',
                'clwName2' => 'Speakers',
                'cntlrname' => 'eminent-speakers',
                'defaultMeta' => [
                    'title' => 'EminentSpeakers – Conference',
                    'description' => 'Our eminent speakers and industry leaders.',
                    'keywords' => 'eminent speakers, conference, leaders'
                ]
            ]
        };
    }
}