<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Conference;
use App\Models\ProgramSpeaker;
use Illuminate\Http\JsonResponse;

class ScientificProgramApiController extends Controller
{
    public function show($id): JsonResponse
    {
        $conference = Conference::with(['ScientificProgrammetaData'])->find($id);

        if (!$conference) {
            return response()->json([
                'success' => false,
                'message' => 'Conference not found'
            ], 404);
        }

        $scheduleDays = $this->generateScheduleDays($conference->from_date, $conference->to_date);

        $programSpeakers = [];
        foreach ($scheduleDays as $index => $date) {
            $dayNum = $index + 1;

            $speakers = ProgramSpeaker::with(['country', 'programContent'])
                ->where([
                    'conference_id' => $id,
                    'sch_day' => $dayNum,
                    'status' => 1
                ])
                ->orderBy('orderpos', 'ASC')
                ->get();

            $speakersList = $speakers->map(function ($s) {
                return [
                    'id' => $s->id,
                    'sTitle' => $s->sTitle,
                    'subTitle' => $s->subTitle,
                    'alias_name' => $s->alias_name,
                    'spk_start_time' => $s->spk_start_time,
                    'spk_end_time' => $s->spk_end_time,
                    'speaker_name' => $s->speaker_name,
                    'orderpos' => $s->orderpos,
                    'affiliation' => $s->affiliation,
                    'speaker_img' => $s->photo_url, // auto-appended via accessor
                    'country_name' => $s->country?->country_name ?? '',
                    'programContent' => $s->programContent->toArray()
                ];
            });

            $programSpeakers[] = [
                'day' => "Day-" . $dayNum,
                'schDate' => $date,
                'schId' => $index,
                'speakersList' => $speakersList
            ];
        }

        $meta = $conference->ScientificProgrammetaData;

        return response()->json([
            'success' => true,
            'scientificProgramData' => [
                'programSpeakers' => $programSpeakers
            ],
            'meta' => [
                'title' => $meta?->meta_title ?? 'Scientific Program – Conference',
                'description' => $meta?->meta_description ?? 'View the full scientific program and session details.',
                'keywords' => $meta?->meta_keywords ?? 'scientific program, abstract, sessions, conference'
            ]
        ]);
    }

    private function generateScheduleDays($fromDate, $toDate)
    {
        if (!$fromDate || !$toDate) {
            return [];
        }

        $scheduleDays = [];
        $current = new \DateTime($fromDate);
        $end = new \DateTime($toDate);
        $end->modify('+1 day');

        while ($current < $end) {
            $scheduleDays[] = $current->format('Y-m-d');
            $current->modify('+1 day');
        }

        return $scheduleDays;
    }
}