<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Stripe\Stripe;
use Stripe\PaymentIntent;
use App\Models\{
    Conference,
    Designation,
    Country,
    Category,
    Registration,
    RegistrationShipping,
    UserRegistrationDate,
    UserAddonAmount
};

class RegistrationApiControllerV10 extends Controller
{
    /**
     * GET /api/registrationv10/{id?}/{currency?}
     * Returns the data needed to render the registration form.
     */
    public function index($id = null, $currency = 3): JsonResponse
    {
        try {
            // Handle bad/missing id (NaN, undefined, null, empty)
            if (!$id || !is_numeric($id)) {
                $defaultConference = Conference::latest('id')->first();
                if (!$defaultConference) {
                    return response()->json(['success' => false, 'message' => 'No conference found'], 404);
                }
                $id = $defaultConference->id;
            }

            $currency = in_array((int)$currency, [1, 2, 3]) ? (int)$currency : 3;

            $conference = Conference::with([
                'registrationDates',
                'onsiteRegistrationStatus',
                'settings',
                'registrationContent',
                'RegistrationmetaData',
            ])->find($id);

            if (!$conference) {
                return response()->json(['success' => false, 'message' => 'Conference not found'], 404);
            }

            $designations = Designation::where('status', 1)->orderBy('id')->get();
            $countries    = Country::where('status', 1)->orderBy('country_name')->get();
            $categories   = Category::where('status', 1)->get();
            $captcha      = rand(100000, 999999);

            $meta = $conference->RegistrationmetaData;

            return response()->json([
                'success' => true,
                'data' => [
                    'conferenceInfo'      => $conference,
                    'designations'        => $designations,
                    'countries'           => $countries,
                    'categories'          => $categories,
                    'registrationContent' => $conference->registrationContent,
                    'newcaptcha'          => $captcha,
                    'symb'                => $this->getCurrencySymbol($currency),
                    'meta'                => [
                        'title'       => $meta?->meta_title ?? '',
                        'description' => $meta?->meta_description ?? '',
                        'keywords'    => $meta?->meta_keywords ?? '',
                    ],
                    'favicon'             => $conference->fav_icon ?? '',
                ],
            ]);
        } catch (\Throwable $e) {
            Log::error('RegistrationV10 index error: ' . $e->getMessage());
            return response()->json(['success' => false, 'message' => 'Failed to load data'], 500);
        }
    }

    /**
     * POST /api/registrationv10/submit
     * Creates a registration row + order row.
     */
    public function submit(Request $req): JsonResponse
    {
        DB::beginTransaction();
        try {
            $order_id = 'SCS-' . strtoupper(uniqid());
            $confId = $req->conferenceId ?? $req->onconference_id ?? $req->conference_id;
            $conf = Conference::find($confId);

            if (!$conf) {
                return response()->json(['success' => false, 'message' => 'Conference not found'], 404);
            }

            // Basic validation (front-end validates too)
            foreach (['designation','firstname','lastname','usr_email','country','user_phone','company','category','currency','payment_meathod'] as $f) {
                if (!$req->filled($f)) {
                    return response()->json(['success' => false, 'message' => "Missing field: $f"], 422);
                }
            }

            $registration = Registration::create([
                'conference_id' => $conf->id,
                'designation'   => $req->designation,
                'firstname'     => $req->firstname,
                'lastname'      => $req->lastname,
                'usr_email'     => $req->usr_email,
                'country'       => $req->country,
                'country_code'  => $this->getCountryCode($req->country),
                'user_phone'    => $req->user_phone,
                'usr_whatsapp'  => $req->whatsapp_number,
                'company'       => $req->company,
                'category'      => $req->category,
                'address'       => $req->address,
                'payment_method'=> $req->payment_meathod,
                'total_amount'  => $req->totalAmount,
                'currency'      => $req->currency,
                'created_date'  => now(),
                'payment_status'=> 'Pending',
            ]);

            RegistrationShipping::create([
                'conference_id'   => $conf->id,
                'user_id'         => $registration->id,
                'shipDesign'      => $req->shipdesignation,
                'shipFirstname'   => $req->shipfirstname,
                'shipLastname'    => $req->shiplastname,
                'shipCompany'     => $req->shipcompany,
                'shipCountry'     => $req->shipcountry,
                'shipEmail'       => $req->shipusr_email,
                'shipPhone'       => $req->shipphone,
                'shipWhatsapp'    => $req->shipwhatsapp_number,
                'shipAddress'     => $req->shipaddress,
                'createDate'      => now(),
                'conference_name' => trim(($conf->conference_first_title ?? '').' '.($conf->conference_name ?? '')),
            ]);

            $regDate = UserRegistrationDate::create([
                'user_id'          => $registration->id,
                'order_id'         => $order_id,
                'registration_id'  => $req->selectedCatId,
                'radio_type'       => $req->selectedItemId,
                'reg_qty'          => $req->selectedQty,
                'reg_total_amount' => ($req->selectedAmount ?? 0) * ($req->selectedQty ?? 0),
                'total_amount'     => $req->totalAmount,
                'curency_type'     => $req->currency,
                'payment_meathod'  => $req->payment_meathod,
                'payment_status'   => 'Pending',
                'reg_date'         => now(),
                'created_date'     => now(),
            ]);

            foreach ($req->add_ons ?? [] as $addOn) {
                UserAddonAmount::create([
                    'order_id'           => $order_id,
                    'user_id'            => $registration->id,
                    'reg_id'             => $regDate->id,
                    'addon_catid'        => $addOn['cat_id'],
                    'addon_subcatid'     => $addOn['item_id'],
                    'addon_qty'          => $addOn['qty'],
                    'addon_amount'       => $addOn['amount'],
                    'addon_total_amount' => $addOn['qty'] * $addOn['amount'],
                    'currency_type'      => $req->currency,
                    'reg_date'           => now(),
                    'created_date'       => now(),
                ]);
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'registration_id' => $registration->id,
                'order_id'        => $order_id,
                'total'           => $req->totalAmount,
                'currency'        => $this->getCurrencyCode($req->currency),
                'symbol'          => $this->getCurrencySymbol($req->currency),
            ]);
        } catch (\Throwable $e) {
            DB::rollBack();
            Log::error('v10 submit error: ' . $e->getMessage());
            return response()->json(['success' => false, 'message' => 'Registration failed'], 500);
        }
    }

    /**
     * POST /api/registrationv10/create-payment-intent
     * Creates Stripe PaymentIntent and returns clientSecret.
     */
    public function createPaymentIntent(Request $request): JsonResponse
    {
        $request->validate([
            'order_id'       => 'required|string',
            'amount'         => 'required|integer|min:1',
            'currency'       => 'required|string|in:usd,eur,gbp',
            'user.name'      => 'required|string',
            'user.email'     => 'required|email',
            'conference_id'  => 'required|integer',
        ]);

        Stripe::setApiKey(env('STRIPE_SECRET_KEY'));

        $conf = Conference::find($request->conference_id);
        if (!$conf) {
            return response()->json(['error' => 'Conference not found'], 404);
        }

        try {
            $desc = "Registration: " . trim(($conf->conference_first_title ?? '') . ' ' . ($conf->conference_name ?? ''));
            $intent = PaymentIntent::create([
                'amount'   => $request->amount,   // integer cents
                'currency' => $request->currency, // usd/eur/gbp
                'description' => $desc,
                'metadata' => [
                    'order_id'       => $request->order_id,
                    'conference_id'  => $request->conference_id,
                    'user_name'      => $request->user['name'],
                    'user_email'     => $request->user['email'],
                    'frontend_url'   => $request->frontend_url ?? config('app.frontend_url', env('FRONTEND_URL')),
                ],
                'payment_method_types' => ['card'],
            ]);

            return response()->json(['clientSecret' => $intent->client_secret]);
        } catch (\Throwable $e) {
            Log::error('Stripe v10 PI error: ' . $e->getMessage());
            return response()->json(['error' => 'Payment initialization failed'], 500);
        }
    }

    /**
     * GET /api/registrationv10/payment/stripe/return
     * Stripe returns here after user pays.
     */
    public function stripeReturn(Request $request)
    {
        $piId = $request->query('payment_intent');
        $frontend = config('app.frontend_url', env('FRONTEND_URL'));

        if (!$piId) {
            return redirect(rtrim($frontend, '/') . '/stripe-payment-failed-v10');
        }

        Stripe::setApiKey(env('STRIPE_SECRET_KEY'));

        try {
            $pi = PaymentIntent::retrieve($piId);
            $orderId = $pi->metadata['order_id'] ?? null;
            $frontendUrl = $pi->metadata['frontend_url'] ?? $frontend;

            if (!$orderId) {
                return redirect(rtrim($frontendUrl, '/') . '/stripe-payment-failed-v10');
            }

            $newStatus = $pi->status === 'succeeded' ? 'paid' : 'cancelled';

            UserRegistrationDate::where('order_id', $orderId)->update([
                'payment_status'    => $newStatus,
                'payment_intent_id' => $pi->id,
            ]);

            $ur = UserRegistrationDate::where('order_id', $orderId)->first();
            if ($ur) {
                Registration::where('id', $ur->user_id)->update([
                    'payment_status' => $newStatus,
                    'transaction_id' => $orderId,
                ]);
            }

            return redirect(rtrim($frontendUrl, '/') . '/payment-success-v10?order_id=' . urlencode($orderId));
        } catch (\Throwable $e) {
            Log::error('Stripe return failed: ' . $e->getMessage());
            return redirect(rtrim($frontend, '/') . '/stripe-payment-failed-v10');
        }
    }

    /**
     * GET /api/registrationv10/payment-success/{order_id}
     * JSON payload for the success page.
     */
    public function getPaymentSuccessData($order_id): JsonResponse
    {
        $row = UserRegistrationDate::with('registration')->where('order_id', $order_id)->first();
        if (!$row) return response()->json(['success' => false, 'message' => 'Order not found'], 404);

        $u = $row->registration;

        return response()->json([
            'success' => true,
            'data' => [
                'firstname'      => $u->firstname,
                'lastname'       => $u->lastname,
                'usr_email'      => $u->usr_email,
                'user_phone'     => $u->user_phone,
                'company'        => $u->company,
                'country'        => $u->country,
                'total_amount'   => $row->total_amount,
                'currency'       => $this->getCurrencyCode($row->curency_type),
                'payment_method' => (int)$row->payment_meathod,
                'payment_status' => $row->payment_status,
                'transaction_id' => $u->transaction_id,
                'invoiceid'      => $u->invoiceid ?? '',
                'created_date'   => $row->created_date,
                'order_id'       => $row->order_id,
            ],
        ]);
    }

    /**
     * GET /api/registrationv10/get/phone-code/{name}
     */
    public function getPhoneCode($name): JsonResponse
    {
        $country = Country::where('country_name', $name)->where('status', 1)->first();
        if ($country) {
            $code = $country->phonecode ?? $country->phone_code ?? $country->country_code ?? '';
            if ($code !== '') {
                return response()->json(['success' => true, 'phonecode' => '+' . $code]);
            }
        }
        return response()->json(['success' => false, 'message' => 'Country code not found'], 404);
    }

    // Helpers
    private function getCurrencySymbol($id)
    {
        return match ((int)$id) {
            1 => '£',
            2 => '$',
            3 => '€',
            default => '€',
        };
    }
    private function getCurrencyCode($id)
    {
        return match ((int)$id) {
            1 => 'GBP',
            2 => 'USD',
            3 => 'EUR',
            default => 'EUR',
        };
    }
    private function getCountryCode($countryName)
    {
        if (!$countryName) return '';
        $country = Country::where('country_name', $countryName)->first();
        return $country ? ($country->phonecode ?? $country->phone_code ?? $country->country_code ?? '') : '';
    }
}
