<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Conference;
use App\Models\Designation;
use App\Models\Country;
use App\Models\Category;
use App\Models\RegistrationDate;
use App\Models\RegistrationCategory;
use App\Models\RegistrationAmount;
use App\Models\Accommodation;
use App\Models\OnsiteRegistrationStatus;
use App\Models\Setting;
use App\Models\RegistrationContent;
use App\Models\Registration;
use App\Models\RegistrationShipping;
use App\Models\UserRegistrationDate;
use App\Models\UserAddonAmount;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Stripe\Stripe;
use Stripe\PaymentIntent;
use Illuminate\Support\Carbon;

class RegistrationApiController extends Controller
{
   public function index($id, $currency = 3): JsonResponse
{
    $currency = in_array($currency, [1, 2, 3]) ? $currency : 3;

    $conference = Conference::with([
        'registrationDates',
        'onsiteRegistrationStatus',
        'settings',
        'registrationContent',
        'RegistrationmetaData'
    ])->find($id);

    if (!$conference) {
        return response()->json(['success' => false, 'message' => 'Conference not found'], 404);
    }
    $meta = $conference->RegistrationmetaData;

    $designations = Designation::where('status', 1)->get();
    $countries = Country::where('status', 1)->orderBy('country_name', 'ASC')->get();
    $categories = Category::where('status', 1)->get();

    // ✅ Use Eloquent — NOT old model
    $regCategories = RegistrationCategory::where('status', 1)
        ->where('page_type', 1)
        ->orderBy('id', 'ASC')
        ->get();

    $regCatArray = $regCategories->map(function ($cat) use ($currency, $id) {
        $amounts = RegistrationAmount::where([
            'status' => 1,
            'currency_id' => $currency,
            'conference_id' => $id,
            'category_id' => $cat->id
        ])->get()->map(function ($item) {
            return (object) [
                'id' => $item->id,
                'registration_name' => $item->registration_name,
                'early_bird_amount' => $item->early_bird_amount,
                'first_call_amount' => $item->first_call_amount,
                'final_call_amount' => $item->final_call_amount,
            ];
        });

        return [
            'id' => $cat->id,
            'category_name' => $cat->category_name,
            'registrationAmount' => $amounts
        ];
    });

    // Same for add-ons
    $addOnCategories = RegistrationCategory::where('status', 1)
        ->where('page_type', 2)
        ->orderBy('id', 'ASC')
        ->get();

    $addOnArray = $addOnCategories->map(function ($cat) use ($currency, $id) {
        $amounts = Accommodation::where([
            'status' => 1,
            'currency_id' => $currency,
            'conference_id' => $id,
            'category_id' => $cat->id
        ])->get()->map(function ($item) {
            return (object) [
                'id' => $item->id,
                'accommodation_name' => $item->accommodation_name,
                'accommodation_amount' => $item->accommodation_amount,
            ];
        });

        return [
            'id' => $cat->id,
            'category_name' => $cat->category_name,
            'addOnAmount' => $amounts
        ];
    });

    // Safe dates
    $dates = (object) [
        'early_date' => $conference->registrationDates?->early_date,
        'standard_date' => $conference->registrationDates?->standard_date,
        'onsite_date' => $conference->registrationDates?->onsite_date,
    ];

    $captcha = rand(100000, 999999);

    return response()->json([
        'success' => true,
        'data' => [
            'conferenceInfo' => $conference,
            'designations' => $designations,
            'countries' => $countries,
            'categories' => $categories,
            'dates' => $dates,
            'regCategories' => $regCatArray,
            'addOnCategories' => $addOnArray,
            'onsiteStatus' => $conference->onsiteRegistrationStatus,
            'settings' => $conference->settings,
            'newcaptcha' => $captcha,
            'symb' => $this->getCurrencySymbol($currency),
            'registrationContent' => $conference->registrationContent,
            'meta' => [
                'title'       => $meta?->meta_title ?? $meta?->title ?? '',
                'description' => $meta?->meta_description ?? '',
                'keywords'    => $meta?->meta_keywords ?? ''
            ],
            'favicon' => $conference->fav_icon ?? ''
        ]
    ]);
}
    private function getCountryCode($countryName)
    {
        if (!$countryName) return '';

        $country = Country::where('country_name', $countryName)->first();
        return $country ? ($country->phonecode ?? $country->phone_code ?? $country->country_code ?? $country->code ?? '') : '';
    }

    private function getCurrencySymbol($id)
    {
        return match((int)$id) {
            1 => '£', // GBP
            2 => '$', // USD
            3 => '€', // EUR
            default => '€'
        };
    }

    public function submit(Request $request): JsonResponse
    {
        \DB::beginTransaction();
        try {
            $order_id = rand(9999,9999999);
        $conference = Conference::find($request->conferenceId);
        if (!$conference) {
            return response()->json(['error' => 'Conference not found'], 404);
        }

        $conferenceName = trim("{$conference->conference_first_title} {$conference->conference_name}");
            $registration = Registration::create([
                'conference_id' => $request->conferenceId,
                'designation' => $request->designation,
                'firstname' => $request->firstname,
                'lastname' => $request->lastname,
                'usr_email' => $request->usr_email,
                'country' => $request->country,
                'country_code' => $this->getCountryCode($request->country),
                'user_phone' => $request->user_phone,
                'usr_whatsapp' => $request->whatsapp_number,
                'company' => $request->company,
                'category' => $request->category,
                'address' => $request->address,
                'payment_method' => $request->payment_meathod,
                'total_amount' => $request->totalAmount,
                'currency' => $request->currsel,
                'created_date' => now(),
                     'invoiceid'           => '',
            'abstract_title'      => '',
            'transaction_id'      => '',
            'extra_info'          => '',
            'total_extra_amount'  => '',
            'coupon_value'        => '',
            'aditional_details'   => '',
            
            ]);

            $registration->shipping()->create([
                'conference_id' => $request->conferenceId,
                'shipDesign' => $request->shipdesignation,
                'shipFirstname' => $request->shipfirstname,
                'shipLastname' => $request->shiplastname,
                'shipCompany' => $request->shipcompany,
                'shipCountry' => $request->shipcountry,
                'shipEmail' => $request->shipusr_email,
                'shipPhone' => $request->shipphone,
                'shipWhatsapp' => $request->shipwhatsapp_number,
                'shipAddress' => $request->shipaddress,
                'createDate' => now(),
                            'conference_name'   => $conferenceName,

            ]);

            $userRegistrationDate = UserRegistrationDate::create([
                'user_id' => $registration->id,
                'order_id' => $order_id,
                'earlyType' => null,
                'registration_id' => $request->selectedCatId,
                'radio_type' => $request->selectedItemId,
                'reg_qty' => $request->selectedQty,
                'reg_total_amount' => $request->selectedAmount * $request->selectedQty,
                'total_amount' => $request->totalAmount,
                'curency_type' => $request->currsel,
                'payment_meathod' => $request->payment_meathod,
                'reg_date' => now(),
                'created_date' => now(),
            ]);

            foreach ($request->add_ons ?? [] as $addOn) {
                UserAddonAmount::create([
                    'order_id' => $order_id,
                    'user_id' => $registration->id,
                    'reg_id' => $userRegistrationDate->id,
                    'addon_catid' => $addOn['cat_id'],
                    'addon_subcatid' => $addOn['item_id'],
                    'addon_qty' => $addOn['qty'],
                    'addon_amount' => $addOn['amount'],
                    'addon_total_amount' => $addOn['qty'] * $addOn['amount'],
                    'currency_type' => $request->currsel,
                    'reg_date' => now(),
                    'created_date' => now(),
                ]);
            }

            \DB::commit();

            return response()->json([
                'success' => true,
                'registration_id' => $registration->id,
                'order_id' => $order_id,
                'total' => $request->totalAmount,
                'currency' => $this->getCurrencyCode($request->currsel),
                'symbol' => $this->getCurrencySymbol($request->currsel),
            ]);
        } catch (\Exception $e) {
            \DB::rollBack();
            \Log::error('Registration Submit Error: ' . $e->getMessage());
            return response()->json(['success' => false, 'message' => 'Registration failed. Please try again.'], 500);
        }
    }

    private function getCurrencyCode($id)
    {
        return match((int)$id) {
            1 => 'GBP',
            2 => 'USD',
            3 => 'EUR',
            default => 'EUR'
        };
    }

    public function createPaymentIntent(Request $request): JsonResponse
    {
        $request->validate([
            'order_id' => 'required|string',
            'amount' => 'required|integer|min:1',
            'currency' => 'required|string|in:usd,eur,gbp',
            'user.name' => 'required|string',
            'user.email' => 'required|email',
            'conference_id' => 'required|integer',
            'transaction_id' => 'nullable|string|max:255',
        ]);

        Stripe::setApiKey(env('STRIPE_SECRET_KEY'));

        $conference = Conference::find($request->conference_id);
        if (!$conference) {
            return response()->json(['error' => 'Conference not found'], 404);
        }

        $conferenceName = trim("{$conference->conference_first_title} {$conference->conference_name}");
        $description = "Registration: {$conferenceName}";
        if ($request->transaction_id) {
            $description .= " (Item ID: {$request->transaction_id})";
        }

        try {
            $intent = PaymentIntent::create([
                'amount' => $request->amount,
                'currency' => $request->currency,
                'description' => $description,
                'metadata' => [
                    'order_id' => $request->order_id,
                    'user_name' => $request->user['name'],
                    'user_email' => $request->user['email'],
                    'conference_id' => $request->conference_id,
                    'conference_name' => $conferenceName,
                    'transaction_id' => $request->transaction_id ?? '',
                        'frontend_url'   => $request->frontend_url ?? config('app.frontend_url', env('FRONTEND_URL')),

                ],
                'payment_method_types' => ['card'],
            ]);

            return response()->json(['clientSecret' => $intent->client_secret]);
        } catch (\Exception $e) {
            \Log::error('Stripe PaymentIntent failed: ' . $e->getMessage());
            return response()->json(['error' => 'Payment setup failed'], 500);
        }
    }

    public function updatePaymentSuccess(Request $request, $order_id): JsonResponse
    {
        $request->validate([
            'payment_intent' => 'required|string',
            'status' => 'required|in:paid,cancelled'
        ]);

        $updated = UserRegistrationDate::where('order_id', $order_id)->update([
            'payment_status' => $request->status,
            'payment_intent_id' => $request->payment_intent,
        ]);

        $userRegistrationDate = UserRegistrationDate::where('order_id', $order_id)->first();
        if ($userRegistrationDate) {
            Registration::where('id', $userRegistrationDate->user_id)->update([
                'payment_status' => $request->status,
                'transaction_id' => $order_id,
            ]);
        }

        if (!$updated) {
            return response()->json(['success' => false, 'message' => 'Order not found'], 404);
        }

        return response()->json(['success' => true]);
    }

    public function getPaymentSuccessDataV4($order_id): JsonResponse
    {
        $userRegistrationDate = UserRegistrationDate::with('registration')->where('order_id', $order_id)->first();

        if (!$userRegistrationDate) {
            return response()->json(['success' => false, 'message' => 'Order not found'], 404);
        }

        $user = $userRegistrationDate->registration;

        return response()->json([
            'success' => true,
            'data' => [
                'firstname' => $user->firstname,
                'lastname' => $user->lastname,
                'usr_email' => $user->usr_email,
                'user_phone' => $user->user_phone,
                'company' => $user->company,
                'country' => $user->country,
                'total_amount' => $userRegistrationDate->total_amount,
                'currency' => $this->getCurrencyCode($userRegistrationDate->curency_type),
                'payment_method' => (int) $userRegistrationDate->payment_meathod,
                'payment_status' => $userRegistrationDate->payment_status,
                'transaction_id' => $user->transaction_id,
                'invoiceid' => $user->invoiceid ?? '',
                'created_date' => $userRegistrationDate->created_date,
                'order_id' => $userRegistrationDate->order_id,
            ]
        ]);
    }

    public function summary($orderId): JsonResponse
    {
        return $this->getPaymentSuccessDataV4($orderId);
    }

    public function init(): JsonResponse
    {
        try {
            $designations = Designation::where('status', 1)->orderBy('id', 'ASC')->get();
            $categories = Category::where('status', 1)->orderBy('id', 'ASC')->get();
            $countries = Country::where('status', 1)->orderBy('country_name', 'ASC')->get();

            return response()->json([
                'success' => true,
                'designations' => $designations,
                'categories' => $categories,
                'countries' => $countries,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load registration data',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function getPhoneCode($name): JsonResponse
    {
        $country = Country::where('country_name', $name)->where('status', 1)->first();

        if ($country && $country->country_code) {
            return response()->json([
                'success' => true,
                'phonecode' => '+' . $country->country_code,
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => 'Country code not found'
        ], 404);
    }

    // V4 Aliases
    public function indexV4($id, $currency = 3): JsonResponse { return $this->index($id, $currency); }
    public function submitV4(Request $request): JsonResponse { return $this->submit($request); }
    public function createPaymentIntentV4(Request $request): JsonResponse { return $this->createPaymentIntent($request); }
    public function updatePaymentSuccessV4(Request $request, $order_id): JsonResponse { return $this->updatePaymentSuccess($request, $order_id); }
    public function summaryV4($order_id): JsonResponse { return $this->summary($order_id); }
    public function initV4(): JsonResponse { return $this->init(); }
    public function getPhoneCodeV4($name): JsonResponse { return $this->getPhoneCode($name); }
    public function getPaymentSuccessData($order_id): JsonResponse { return $this->getPaymentSuccessDataV4($order_id); }

public function stripeReturnV4(Request $request)
{
    $piId = $request->query('payment_intent');
    if (!$piId) {
        return redirect(config('app.frontend_url', env('FRONTEND_URL')) . '/stripe-payment-failed');
    }

    Stripe::setApiKey(env('STRIPE_SECRET_KEY'));

    try {
        $pi = PaymentIntent::retrieve($piId);
        $orderId = $pi->metadata['order_id'] ?? null;
        $frontendUrl = $pi->metadata['frontend_url'] ?? config('app.frontend_url', env('FRONTEND_URL'));
        \Log::error('Stripe return verify failed: ' . $frontendUrl);

        if (!$orderId) {
            return redirect($frontendUrl . '/stripe-payment-failed');
        }

        $newStatus = $pi->status === 'succeeded' ? 'paid' : 'cancelled';

        $updated = UserRegistrationDate::where('order_id', $orderId)->update([
            'payment_status' => $newStatus,
            'payment_intent_id' => $pi->id,
        ]);

        if ($updated) {
            $userRow = UserRegistrationDate::where('order_id', $orderId)->first();
            if ($userRow) {
                Registration::where('id', $userRow->user_id)->update([
                    'payment_status' => $newStatus,
                    'transaction_id' => $orderId,
                ]);
            }
        }

        $cookie = cookie('scs_order', $orderId, 5, null, null, false, false, false, 'Lax');

        return redirect(rtrim($frontendUrl, '/') . '/stripe-payment-success-v4')
            ->withCookie($cookie);

    } catch (\Exception $e) {
        \Log::error('Stripe return verify failed: ' . $e->getMessage());
        $frontendUrl = config('app.frontend_url', env('FRONTEND_URL'));
        return redirect(rtrim($frontendUrl, '/') . '/stripe-payment-failed');
    }
}

    public function online($id, $currency = 3): JsonResponse
    {
        $currency = in_array((int)$currency, [1,2,3]) ? (int)$currency : 3;

        $conference = Conference::with([
            'registrationDates',
            'onsiteRegistrationStatus',
             'OnlinemetaData'
        ])->find($id);

        if (!$conference) {
            return response()->json(['success' => false, 'message' => 'Conference not found'], 404);
        }

    $meta = $conference->OnlinemetaData;

        return response()->json([
            'success' => true,
            'data' => [
                'conferenceId' => $id,
                'currency' => $currency,
               
                'meta' => [
                'title'       => $meta?->meta_title ?? $meta?->title ?? '',
                'description' => $meta?->meta_description ?? '',
                'keywords'    => $meta?->meta_keywords ?? ''
            ],
            'favicon' => $conference->fav_icon ?? ''
            ]
        ]);
    }
    
    public function submitv7(Request $request): JsonResponse
    {
        \DB::beginTransaction();
        try {
            $order_id = rand(9999,9999999);
        $conference = Conference::find($request->conferenceId);
        if (!$conference) {
            return response()->json(['error' => 'Conference not found'], 404);
        }

        $conferenceName = trim("{$conference->conference_first_title} {$conference->conference_name}");
            $registration = Registration::create([
                'conference_id' => $request->conferenceId,
                'designation' => $request->designation,
                'firstname' => $request->firstname,
                'lastname' => $request->lastname,
                'usr_email' => $request->usr_email,
                'country' => $request->country,
                'country_code' => $this->getCountryCode($request->country),
                'user_phone' => $request->user_phone,
                'usr_whatsapp' => $request->whatsapp_number,
                'company' => $request->company,
                'category' => $request->category,
                'address' => $request->address,
                'payment_method' => $request->payment_meathod,
                'total_amount' => $request->totalAmount,
                'currency' => $request->currsel,
                'created_date' => now(),
                     'invoiceid'           => '',
            'abstract_title'      => '',
            'transaction_id'      => '',
            'extra_info'          => '',
            'total_extra_amount'  => '',
            'coupon_value'        => '',
            'aditional_details'   => '',
            
            ]);

            $registration->shipping()->create([
                'conference_id' => $request->conferenceId,
                'shipDesign' => $request->shipdesignation,
                'shipFirstname' => $request->shipfirstname,
                'shipLastname' => $request->shiplastname,
                'shipCompany' => $request->shipcompany,
                'shipCountry' => $request->shipcountry,
                'shipEmail' => $request->shipusr_email,
                'shipPhone' => $request->shipphone,
                'shipWhatsapp' => $request->shipwhatsapp_number,
                'shipAddress' => $request->shipaddress,
                'createDate' => now(),
                            'conference_name'   => $conferenceName,

            ]);

            $userRegistrationDate = UserRegistrationDate::create([
                'user_id' => $registration->id,
                'order_id' => $order_id,
                'earlyType' => null,
                'registration_id' => $request->selectedCatId,
                'radio_type' => $request->selectedItemId,
                'reg_qty' => $request->selectedQty,
                'reg_total_amount' => $request->selectedAmount * $request->selectedQty,
                'total_amount' => $request->totalAmount,
                'curency_type' => $request->currsel,
                'payment_meathod' => $request->payment_meathod,
                'reg_date' => now(),
                'created_date' => now(),
            ]);

            foreach ($request->add_ons ?? [] as $addOn) {
                UserAddonAmount::create([
                    'order_id' => $order_id,
                    'user_id' => $registration->id,
                    'reg_id' => $userRegistrationDate->id,
                    'addon_catid' => $addOn['cat_id'],
                    'addon_subcatid' => $addOn['item_id'],
                    'addon_qty' => $addOn['qty'],
                    'addon_amount' => $addOn['amount'],
                    'addon_total_amount' => $addOn['qty'] * $addOn['amount'],
                    'currency_type' => $request->currsel,
                    'reg_date' => now(),
                    'created_date' => now(),
                ]);
            }

            \DB::commit();

            return response()->json([
                'success' => true,
                'registration_id' => $registration->id,
                'order_id' => $order_id,
                'total' => $request->totalAmount,
                'currency' => $this->getCurrencyCode($request->currsel),
                'symbol' => $this->getCurrencySymbol($request->currsel),
            ]);
        } catch (\Exception $e) {
            \DB::rollBack();
            \Log::error('Registration Submit Error: ' . $e->getMessage());
            return response()->json(['success' => false, 'message' => 'Registration failed. Please try again.'], 500);
        }
    }

public function createPaymentIntentV7(Request $request): JsonResponse {
    return $this->createPaymentIntent($request);
}



public function handleStripePaymentChatGPT(Request $request): JsonResponse
{
    try {
        $conference = Conference::find($request->conference_id);
        if (!$conference) {
            return response()->json(['success' => false, 'message' => 'Conference not found.'], 404);
        }

        $conferenceUrl = rtrim($conference->conference_url ?? config('app.frontend_url', env('FRONTEND_URL')), '/');
        Stripe::setApiKey(env('STRIPE_SECRET_KEY'));

        $checkout = \Stripe\Checkout\Session::create([
            'payment_method_types' => ['card'],
            'line_items' => [[
                'price_data' => [
                    'currency' => strtolower($this->getCurrencyCode($request->currsel ?? 2)),
                    'product_data' => ['name' => 'Conference Registration'],
                    'unit_amount' => intval($request->amount * 100),
                ],
                'quantity' => 1,
            ]],
            'mode' => 'payment',
            'success_url' => "{$conferenceUrl}/chatgpt-stripe-success?order_id={$request->order_id}",
            'cancel_url'  => "{$conferenceUrl}/chatgpt-stripe-cancel?order_id={$request->order_id}",
            'metadata' => [
                'order_id' => $request->order_id,
                'conference_id' => $request->conference_id,
                'conference_name' => $conference->conference_name,
            ],
        ]);

        return response()->json([
            'success' => true,
            'url' => $checkout->url,
            'session_id' => $checkout->id,
        ]);
    } catch (\Exception $e) {
        \Log::error('Stripe checkout create failed: ' . $e->getMessage());
        return response()->json(['success' => false, 'message' => 'Stripe initialization failed.'], 500);
    }
}




public function handlePaypalPaymentChatGPT(Request $request): JsonResponse
{
    try {
        $conference = Conference::find($request->conference_id);
        if (!$conference) {
            return response()->json(['success' => false, 'message' => 'Conference not found.'], 404);
        }

        $sandbox = env('PAYMENT_SANDBOX_V7', true);
        $paypalBusiness = $sandbox
            ? env('PAYPAL_BUSINESS_SANDBOX_V7')
            : env('PAYPAL_BUSINESS_LIVE_V7');

        $paypalUrl = $sandbox
            ? 'https://www.sandbox.paypal.com/cgi-bin/webscr'
            : 'https://www.paypal.com/cgi-bin/webscr';

        $conferenceUrl = rtrim($conference->conference_url ?? config('app.frontend_url', env('FRONTEND_URL')), '/');
        $currencyCode = $this->getCurrencyCode($request->currsel ?? 2);

        $query = http_build_query([
            'cmd'           => '_xclick',
            'business'      => $paypalBusiness,
            'item_name'     => 'Conference Registration',
            'amount'        => number_format($request->amount, 2, '.', ''),
            'currency_code' => $currencyCode,
            'quantity'      => 1,
            'return'        => "{$conferenceUrl}/chatgpt-paypal-success?order_id={$request->order_id}",
            'cancel_return' => "{$conferenceUrl}/chatgpt-paypal-cancel?order_id={$request->order_id}",
            'notify_url'    => url('/api/paypal/ipn-handler'),
            'custom'        => $request->order_id,
        ]);

        return response()->json([
            'success' => true,
            'redirect_url' => "{$paypalUrl}?{$query}"
        ]);
    } catch (\Exception $e) {
        \Log::error('PayPal redirect failed: ' . $e->getMessage());
        return response()->json(['success' => false, 'message' => 'PayPal initialization failed.'], 500);
    }
}

}