<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Designation;
use App\Models\Category;
use App\Models\Country;
use App\Models\Conference;
use App\Models\Registration;
use App\Models\RegistrationShipping;
use App\Models\UserRegistrationDate;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class OnlineRegistrationController extends Controller
{
    /**
     * Init data for Vue (designations, categories, countries, conference info, etc.)
     */
    public function init($conferenceId = null): JsonResponse
    {
        if (!$conferenceId) {
            return response()->json(['success' => false, 'message' => 'Conference required'], 400);
        }

        $conference = Conference::find($conferenceId);
        if (!$conference) {
            return response()->json(['success' => false, 'message' => 'Conference not found'], 404);
        }

        $designations = Designation::where('status', 1)->orderBy('id', 'ASC')->get();
        $categories = Category::where('status', 1)->orderBy('id', 'ASC')->get();
        $countries = Country::where('status', 1)->orderBy('country_name', 'ASC')->get();

        return response()->json([
            'success' => true,
            'conferenceInfo' => $conference,
            'designations' => $designations,
            'categories' => $categories,
            'countries' => $countries
        ]);
    }

    /**
     * Checkout (similar to checkoutonline in CI)
     */
    public function checkout(Request $request): JsonResponse
    {
        try {
            $created_date = now();
            $orderid = rand(9999, 9999999);
            $transaction_id = "SCS" . rand(999, 99999) . date("ymdHis");

            // Create registration
            $registration = Registration::create([
                'conference_id' => $request->onconference_id,
                'conference_name' => $request->conference_name,
                'designation' => $request->designation,
                'firstname' => $request->firstname,
                'lastname' => $request->lastname,
                'category' => $request->category,
                'company' => $request->company,
                'country' => $request->country,
                'usr_email' => $request->usr_email,
                'user_phone' => $request->user_phone,
                'usr_whatsapp' => $request->shipwhatsapp_number,
                'currency' => $request->currency,
                'total_amount' => $request->total_amount,
                'payment_meathod' => $request->payment_meathod,
                'created_date' => $created_date,
                'invoiceid' => date("ymdHis"),
                'transaction_id' => $transaction_id,
            ]);

            // Create shipping record
            $registration->shipping()->create([
                'cat_type' => $request->cat_type,
                'conference_name' => $request->conference_name,
                'shipDesign' => $request->shipdesignation,
                'shipFirstname' => $request->shipfirstname,
                'shipLastname' => $request->shiplastname,
                'shipCompany' => $request->shipcompany,
                'shipCountry' => $request->shipcountry,
                'shipEmail' => $request->shipusr_email,
                'shipPhone' => $request->shipphone,
                'shipWhatsapp' => $request->shipwhatsapp_number,
                'shipAddress' => $request->shipaddress,
                'createDate' => $created_date,
            ]);

            // Create registration date record
            UserRegistrationDate::create([
                'order_id' => $orderid,
                'registration_name' => $request->conference_name,
                'user_id' => $registration->id,
                'curency_type' => $request->currency,
                'total_amount' => $request->total_amount,
                'payment_meathod' => $request->payment_meathod,
                'reg_date' => $created_date,
                'created_date' => $created_date,
            ]);

            // PayPal
            if ($request->payment_meathod == 2) {
                $paypalAmt = ($request->total_amount * 3) / 100;
                $finalAmount = $request->total_amount + $paypalAmt;

                return response()->json([
                    'success' => true,
                    'payment' => 'paypal',
                    'finalAmount' => $finalAmount,
                    'paypal_url' => url("/api/paypal/success/{$orderid}"),
                    'order_id' => $orderid
                ]);
            } else {
                // Stripe
                return response()->json([
                    'success' => true,
                    'payment' => 'stripe',
                    'finalAmount' => $request->total_amount,
                    'stripe_url' => url("/api/stripe/success/{$orderid}"),
                    'order_id' => $orderid
                ]);
            }
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    /**
     * PayPal summary
     */
public function paypalSummary($orderId): JsonResponse
{
    $userRegistrationDate = UserRegistrationDate::with('registration')->where('order_id', $orderId)->first();

    if (!$userRegistrationDate) {
        return response()->json(['success' => false, 'message' => 'Order not found'], 404);
    }

    $user = $userRegistrationDate->registration;

    return response()->json([
        'success' => true,
        'data' => array_merge(
            $userRegistrationDate->toArray(),
            $user->toArray()
        )
    ]);
}

    /**
     * Stripe summary
     */
    public function stripeSummary($orderId): JsonResponse
    {
        return $this->paypalSummary($orderId); // Reuse logic
    }
}