<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use App\Models\Conferenceadmin\User;
use DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Storage;
use App\Helpers\AuditLogger; // 👈 ADD

class ConferenceAdminController extends Controller
{
    use AuditLogger; // 👈 ADD

    /**
     * LOGIN (SHA1 based)
     */
    public function login(Request $request)
    {
        $email = $request->input('email');
        $password = sha1($request->input('password')); // ✅ same as CodeIgniter logic

        $user = \App\Models\Conferenceadmin\User::where('email', $email)
            ->where('password', $password)
            ->whereIn('usertype', [1, 2, 4])
            ->where('status', 1)
            ->first();

        if (!$user) {
            return response()->json(['message' => 'Invalid Username / Password'], 401);
        }

        // 🔹 OLD DATA for audit (before token change)
        $old = ['api_token' => $user->api_token];

        // Generate a simple token (you can improve later)
        $token = bin2hex(random_bytes(32));

        $user->api_token = $token;
        $user->save();

        // 🔹 NEW DATA for audit
        $new = ['api_token' => $user->api_token];

        // 🔥 Audit log (user table name from model)
      //  $this->logAudit($user->getTable(), $user->id, 'updated', $old, $new);

        $user->role_text = match ((string)($user->usertype ?? '2')) {
            '1' => 'Super Admin',
            '4' => 'Sub Super Admin',
            default => 'Admin',
        };

        return response()->json([
            'token' => $token,
            'user'  => $user
        ]);
    }

    public function conferences()
    {
        $conferences = \DB::table('conferences')
            ->select('id', 'conference_name', 'short_name', 'conference_year','conference_url','unique_id','folder_name')
            ->where('status', 1)
            ->where('short_name', '!=', '') 
            ->orderBy('short_name')
            ->get();

        return response()->json($conferences);
    }

    /**
     * LOGOUT
     */
    public function logout(Request $request)
    {
        $user = $request->user;

        if ($user) {
            // 🔹 OLD DATA
            $old = ['api_token' => $user->api_token];

            $user->api_token = null;
            $user->save();

            // 🔹 NEW DATA
            $new = ['api_token' => $user->api_token];

            // 🔥 Audit
            $this->logAudit($user->getTable(), $user->id, 'updated', $old, $new);
        }

        return response()->json(['message' => 'Logged out successfully']);
    }

    /**
     * HEADER API
     */
    public function header(Request $request)
    {
        // ✅ Extract token from header
        $token = $request->bearerToken()
            ?? $request->header('authorization')
            ?? $request->input('token')
            ?? $request->query('token');

        $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();

        if (!$user) {
            return response()->json(['message' => 'Unauthorized'], 401);
        }

        // Assign role text safely
        $user->role_text = match ((string)($user->usertype ?? '2')) {
            '1' => 'Super Admin',
            '4' => 'Sub Super Admin',
            default => 'Admin',
        };

        $conferenceName = 'Scisynopsis Conferences';

        $menu = [
            // 0. Dashboard
            [ 'label' => 'Dashboard', 'icon' => 'fa fa-home', 'to' => '/conferenceadmin/dashboard' ],

            // 1–10: Single items in order
            [ 'label' => 'Logo', 'icon' => 'fa fa-image', 'to' => '/conferenceadmin/logo' ],
            [ 'label' => 'Conference Info', 'icon' => 'fa fa-info-circle', 'to' => '/conferenceadmin/conference-info' ],

            // ✅ Past Conference – ఇప్పుడు v1 కాదు, నువ్వు పెట్టుకున్న SPA రూట్
            [ 'label' => 'Past Conference', 'icon' => 'fa fa-history', 'to' => '/conferenceadmin/past-conference' ],

            [ 'label' => 'Field Wise', 'icon' => 'fa fa-list', 'to' => '/conferenceadmin/fieldwise' ],

            // ✅ Banners → /conferenceadmin/v1/hban
            [ 'label' => 'Banners', 'icon' => 'fa fa-picture-o', 'to' => '/conferenceadmin/v1/hban' ],

            // ✅ Testimonials → /conferenceadmin/v1/testimonials
            [ 'label' => 'Testimonials', 'icon' => 'fa fa-image', 'to' => '/conferenceadmin/v1/testimonials' ],

            // ✅ What Speaker Says → /conferenceadmin/v1/testimonialspeakers
            [ 'label' => 'What Speaker Says', 'icon' => 'fa fa-image', 'to' => '/conferenceadmin/v1/testimonialspeakers' ],

            // ✅ Scientific Tracks → /conferenceadmin/v1/tracks
            [ 'label' => 'Scientific Tracks', 'icon' => 'fa fa-list', 'to' => '/conferenceadmin/v1/tracks' ],

            // ✅ Page Settings → /conferenceadmin/v1/pagesettings
            [ 'label' => 'Page Settings', 'icon' => 'fa fa-image', 'to' => '/conferenceadmin/v1/pagesettings' ],

            [ 'label' => 'Header &Footer Navigation', 'icon' => 'fa fa-image', 'to' => '/conferenceadmin/pagenavigations/1' ],

            // 11. Home Past Gallery
            [
                'label' => 'Home Past Gallery',
                'icon'  => 'fa fa-home',
                'children' => [
                    [ 'label' => 'Previous Gallery',      'to' => '/conferenceadmin/pastgallery' ],
                    [ 'label' => 'All Previous Gallery',  'to' => '/conferenceadmin/pastallgallery' ],
                ],
            ],

            // 12. Home Page Content
            [
                'label' => 'Home Page Content',
                'icon'  => 'fa fa-home',
                'children' => [
                    // ✅ Top Panel
                    [ 'label' => 'Top Panel',                'to' => '/conferenceadmin/v1/hedrpnl_1' ],
                    // ✅ Side Panel
                    [ 'label' => 'Side Panel',              'to' => '/conferenceadmin/v1/hedrpnl_2' ],
                    // ✅ Home Page Popup
                    [ 'label' => 'Home Page Popup',         'to' => '/conferenceadmin/v1/homepopup' ],
                    // ✅ Previous Edition
                    [ 'label' => 'Previous Edition',        'to' => '/conferenceadmin/v1/ephban' ],
                    // ✅ Quick Links
                    [ 'label' => 'Quick Links',             'to' => '/conferenceadmin/v1/qlAct' ],
                    // ✅ Early Bird Content
                    [ 'label' => 'Early Bird Content',      'to' => '/conferenceadmin/v1/earlyAct' ],
                    // ✅ Global Target Audience
                    [ 'label' => 'Global Target Audience',  'to' => '/conferenceadmin/v1/globalAct' ],
                    // ✅ Skeleton Schedule
                    [ 'label' => 'Skeleton Schedule',       'to' => '/conferenceadmin/v1/skeleton' ],
                    // ✅ Post-Conference Workshops
                    [ 'label' => 'Post-Conference Workshops','to' => '/conferenceadmin/v1/pcw' ],
                    // ✅ Supported Journals
                    [ 'label' => 'Supported Journals',      'to' => '/conferenceadmin/v1/supAct' ],
                    // ✅ Social Media Timeline
                    [ 'label' => 'Social Media Timeline',   'to' => '/conferenceadmin/v1/spages' ],
                    // ✅ Important Dates
                    [ 'label' => 'Important Dates',         'to' => '/conferenceadmin/v1/impcont' ],
                    // ✅ About Conference
                    [ 'label' => 'About Conference',        'to' => '/conferenceadmin/v1/aboutus' ],
                    // ✅ Welcome Message
                    [ 'label' => 'Welcome Message',         'to' => '/conferenceadmin/v1/welcome' ],
                    // ✅ Sponsors (logos)
                    [ 'label' => 'Sponsors',                'to' => '/conferenceadmin/v1/sponsors?page_type=1' ],
                    // ✅ Past Editions Group Photo (SPA, not v1)
                    [ 'label' => 'Past Editions Group Photo', 'to' => '/conferenceadmin/past-editions' ],
                ],
            ],

            // 13–14. Speakers
            [ 'label' => 'Speakers',      'icon' => 'fa fa-users', 'to' => '/conferenceadmin/v1/spkcont?type=1' ],
            [ 'label' => 'New Speakers',  'icon' => 'fa fa-users', 'to' => '/conferenceadmin/v1/new_spkcont?type=1' ],

            // 15. Past Events
            [
                'label' => 'Past Events',
                'icon'  => 'fa fa-home',
                'children' => [
                    [ 'label' => 'Conference Book', 'to' => '/conferenceadmin/v1/conferencepdf' ],
                    [ 'label' => 'Past Events',     'to' => '/conferenceadmin/v1/pastevents' ],
                ],
            ],

            // 16. Others
            [
                'label' => 'Others',
                'icon'  => 'fa fa-bars',
                'children' => [
                    [ 'label' => 'Header Content',       'to' => '/conferenceadmin/headercontent' ],
                    [ 'label' => 'Brochure',             'to' => '/conferenceadmin/v1/doccontent_1' ],
                    [ 'label' => 'Sponsorship',          'to' => '/conferenceadmin/v1/doccontent_12' ],
                    [ 'label' => 'Sample Abstract',      'to' => '/conferenceadmin/v1/doccontent_2' ],
                    [ 'label' => 'Conference Program',   'to' => '/conferenceadmin/v1/doccontent_3' ],
                    [ 'label' => 'Conference Book',      'to' => '/conferenceadmin/v1/doccontent_4' ],
                    [ 'label' => 'Contact Info',         'to' => '/conferenceadmin/address' ],
                    [ 'label' => 'Social Links',         'to' => '/conferenceadmin/social' ],
                    [ 'label' => 'Chat Script',          'to' => '/conferenceadmin/tawk' ],
                    [ 'label' => 'Google Tracking Code', 'to' => '/conferenceadmin/homscr' ],
                    [ 'label' => 'Facebook Messenger',   'to' => '/conferenceadmin/messengerScript' ],
                ],
            ],

            // 17–18. Program Speakers
            [ 'label' => 'Program Speakers',       'icon' => 'fa fa-users', 'to' => '/conferenceadmin/programSpeakers' ],
            [ 'label' => 'Program Speakers sort',  'icon' => 'fa fa-list',  'to' => '/conferenceadmin/c_program_spkrs_sort' ],

            // 19. Abstract Pages Content
            [
                'label' => 'Abstract Pages Content',
                'icon'  => 'fa fa-home',
                'children' => [
                    [ 'label' => 'Abstract Content', 'to' => '/conferenceadmin/v1/abstract_1' ],
                    [ 'label' => 'Abstract Banner',  'to' => '/conferenceadmin/v1/doccontent_6?page_type=6' ],
                ],
            ],

            // 20. Presentation Content
            [
                'label' => 'Presentation Content',
                'icon'  => 'fa fa-home',
                'children' => [
                    [ 'label' => 'Presentation Content', 'to' => '/conferenceadmin/v1/abstract_5' ],
                ],
            ],

            // 21. Registration
            [
                'label' => 'Registration',
                'icon'  => 'fa fa-cogs',
                'children' => [
                    [ 'label' => 'Registration Dates',        'to' => '/conferenceadmin/v1/rdatAct' ],
                    [ 'label' => 'Registration Amounts',      'to' => '/conferenceadmin/v1/ramAct?currency=2' ],
                    [ 'label' => 'New Registration Amounts',  'to' => '/conferenceadmin/v1/ramAct1?currency=2' ],
                    [ 'label' => 'Add Ons',                   'to' => '/conferenceadmin/v1/accAct?currency=2' ],
                    [ 'label' => 'Add Ons V1',                'to' => '/conferenceadmin/v1/accommodation_v1?currency=2' ],
                    [ 'label' => 'Registration Info',         'to' => '/conferenceadmin/v1/regcontinfo' ],
                    [ 'label' => 'Registration Content',      'to' => '/conferenceadmin/v1/regcont' ],
                    [ 'label' => 'Coupons',                   'to' => '/conferenceadmin/v1/cpnAct' ],
                ],
            ],

            // 22. Group Registration
            [
                'label' => 'Group Registration',
                'icon'  => 'fa fa-cogs',
                'children' => [
                    [ 'label' => 'Group Registration Amounts', 'to' => '/conferenceadmin/v1/grpamt' ],
                    [ 'label' => 'Group Registration Content', 'to' => '/conferenceadmin/v1/grpc' ],
                ],
            ],

            // 23. Venue
            [
                'label' => 'Venue',
                'icon'  => 'fa fa-map-marker',
                'children' => [
                    [ 'label' => 'Venue',              'to' => '/conferenceadmin/v1/venueMenu' ],
                    [ 'label' => 'Venue Images',       'to' => '/conferenceadmin/v1/venueImages' ],
                    [ 'label' => 'FAQ\'s',             'to' => '/conferenceadmin/v1/venuePage' ],
                    [ 'label' => 'City Attractions',   'to' => '/conferenceadmin/v1/gallery' ],
                    [ 'label' => 'Accommodation Venue','to' => '/conferenceadmin/v1/venue1' ],
                ],
            ],

            // 24. Sponsors & Exhibitors
            [
                'label' => 'Sponsors & Exhibitors',
                'icon'  => 'fa fa-users',
                'children' => [
                    [ 'label' => 'Content',           'to' => '/conferenceadmin/v1/sponsersContent' ],
                    [ 'label' => 'Sponsors Content',  'to' => '/conferenceadmin/v1/exhibitorsContent' ],
                ],
            ],

            // 25. Users
            [
                'label' => 'Users',
                'icon'  => 'fa fa-users',
                'children' => [
                    [ 'label' => 'Brochure Downloaded Users',     'to' => '/conferenceadmin/brochure-users-v2' ],
                    [ 'label' => 'Sponsorship Downloaded Users',  'to' => '/conferenceadmin/sponsorship-users-v2' ],
                    [ 'label' => 'Abstract Submission Users',     'to' => '/conferenceadmin/abstract-users-v2' ],
                    [ 'label' => 'Presentation Submission Users', 'to' => '/conferenceadmin/presentation-users-v2' ],
                    [ 'label' => 'Registered Users',              'to' => '/conferenceadmin/registered-users-v2' ],
                    [ 'label' => 'New Registered Users',          'to' => '/conferenceadmin/register-new-users-v2' ],
                    [ 'label' => 'Discount Users',                'to' => '/conferenceadmin/discount-users-v2' ],
                    [ 'label' => 'Manual Payments',               'to' => '/conferenceadmin/manual-payment-users-v2' ],
                    [ 'label' => 'Online Payments',               'to' => '/conferenceadmin/online-users-v2' ],
                    [ 'label' => 'Group Registration Users',      'to' => '/conferenceadmin/group-registration-users-v2' ],
                    [ 'label' => 'Subscribers/Unsubscribers',     'to' => '/conferenceadmin/v1/subscribers' ],
                ],
            ],

            // 26–27. Footer + Meta
            [ 'label' => 'Footer Section', 'icon' => 'fa fa-bars', 'to' => '/conferenceadmin/footerSection' ],
            [ 'label' => 'Meta Data',      'icon' => 'fa fa-book', 'to' => '/conferenceadmin/cmeta' ],

            // 28. Conference Report
            [
                'label' => 'Conference Report',
                'icon'  => 'fa fa-sitemap',
                'children' => [
                    [ 'label' => 'Conference Report', 'to' => '/conferenceadmin/v1/conferenceReport' ],
                    [ 'label' => 'Conference Gallery','to' => '/conferenceadmin/v1/prevGallery' ],
                    [ 'label' => 'Conference Videos', 'to' => '/conferenceadmin/v1/prevVideos' ],
                ],
            ],

            // 29–30. Enquiries
            [ 'label' => 'Ask Questions', 'icon' => 'fa fa-users',    'to' => '/conferenceadmin/enquiries/2' ],
            [ 'label' => 'Enquiries',     'icon' => 'fa fa-envelope', 'to' => '/conferenceadmin/enquiries/1' ],
                        [ 'label' => 'Audit Logs',     'icon' => 'fa fa-history', 'to' => '/conferenceadmin/v1/audit_logs' ],


        ];

        return response()->json([
            'user'            => $user,
            'conferenceName'  => $conferenceName,
            'menu'            => $menu,
        ]);
    }

    /**
     * FOOTER API
     */
    public function footer()
    {
        return response()->json([
            'name' => 'Scisynopsis Conferences',
            'year' => date('Y'),
            'copyright' => '© ' . date('Y') . ' Scisynopsis Conferences',
        ]);
    }

    /**
     * DASHBOARD API
     */
    public function dashboard(Request $request)
    {
        return response()->json([
            'welcome_message'     => 'Welcome To Scisynopsis Conferences Admin Panel',
            'conference_name'     => 'Scisynopsis 2025',
            'total_speakers'      => 128,
            'total_registrations' => 342,
            'user'                => $request->user,
        ]);
    }

    /**
     * GET /api/logo
     * Returns current conference logo data
     */
    public function logo(Request $request)
    {
        $token = $request->bearerToken()
            ?? $request->header('authorization')
            ?? $request->input('token')
            ?? $request->query('token');

        $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
        if (!$user) {
            return response()->json(['message' => 'Unauthorized'], 401);
        }

        // ✅ Use conference_id from request instead of session
        $conferenceId = $request->input('conference_id');
        if (!$conferenceId) {
            return response()->json(['message' => 'Conference ID missing'], 422);
        }

        $conference = \DB::table('conferences')->where('id', $conferenceId)->first();
        if (!$conference) {
            return response()->json(['message' => 'Conference not found'], 404);
        }

        $baseUrl = "https://scisynopsisconferences.com/uploads/conferences/{$conference->unique_id}";

        return response()->json([
            'conference_logo'       => $conference->conference_logo ? "{$baseUrl}/{$conference->conference_logo}" : null,
            'conference_inner_logo' => $conference->conference_inner_logo ? "{$baseUrl}/{$conference->conference_inner_logo}" : null,
            'footer_logo'           => $conference->footer_logo ? "{$baseUrl}/{$conference->footer_logo}" : null,
            'fav_icon'              => $conference->fav_icon ? "{$baseUrl}/{$conference->fav_icon}" : null,
            'folderName'            => $conference->folder_name ?? 'default',
        ]);
    }

    /**
     * POST /api/logo/save
     * Save uploaded logos
     */
    public function saveLogo(Request $request)
    {
        $token = $request->bearerToken()
            ?? $request->header('authorization')
            ?? $request->input('token')
            ?? $request->query('token');

        $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
        if (!$user) {
            return response()->json(['message' => 'Unauthorized'], 401);
        }

        $conferenceId = $request->input('conference_id');
        if (!$conferenceId) {
            return response()->json(['message' => 'Conference ID missing'], 422);
        }

        $folderName = \DB::table('conferences')->where('id', $conferenceId)->value('unique_id');
        $uploadPath = "/home/scisynopsisconfe/public_html/uploads/conferences/{$folderName}";

        if (!file_exists($uploadPath)) {
            mkdir($uploadPath, 0777, true);
        }

        $update = [];
        $fields = [
            'conference_logo'       => 'logo_',
            'conference_inner_logo' => 'inner_',
            'footer_logo'           => 'flogo_',
            'fav_icon'              => 'fav_'
        ];

        foreach ($fields as $field => $prefix) {
            if ($request->hasFile($field)) {
                $file = $request->file($field);
                \Log::info("📸 Uploading {$field}", [
                    'originalName' => $file->getClientOriginalName(),
                    'valid'        => $file->isValid(),
                ]);

                if ($file->isValid()) {
                    $filename = $prefix . time() . '.' . $file->getClientOriginalExtension();
                    $file->move($uploadPath, $filename);
                    $update[$field] = $filename;
                    \Log::info("✅ File moved", ['path' => "{$uploadPath}/{$filename}"]);
                } else {
                    \Log::error("❌ Invalid file: {$field}");
                }
            }
        }

        if (!empty($update)) {
            // 🔹 OLD row before update
            $old = \DB::table('conferences')->where('id', $conferenceId)->first();

            \DB::table('conferences')->where('id', $conferenceId)->update($update);

            // 🔹 NEW row after update
            $new = \DB::table('conferences')->where('id', $conferenceId)->first();

            // 🔥 Audit
            if ($old && $new) {
                $this->logAudit('conferences', $conferenceId, 'updated', (array) $old, (array) $new);
            }
        }

        return response()->json([
            'message'        => 'Logos updated successfully',
            'updated'        => $update,
            'conference_id'  => $conferenceId,
            'upload_path'    => $uploadPath // 👀 Optional: include in API response for debug
        ]);
    }

    public function getConferenceInfo(Request $request)
    {
        $token = $request->bearerToken()
            ?? $request->header('authorization')
            ?? $request->input('token')
            ?? $request->query('token');

        $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
        if (!$user) {
            return response()->json(['message' => 'Unauthorized'], 401);
        }

        $conferenceId = $request->query('conference_id');
        if (!$conferenceId) {
            return response()->json(['message' => 'Conference ID missing'], 422);
        }

        $conf = \DB::table('conferences')->where('id', $conferenceId)->first();
        if (!$conf) {
            return response()->json(['message' => 'Conference not found'], 404);
        }

        // Helper to make absolute
        $abs     = fn($p) => $p ? "https://scisynopsisconferences.com/uploads/homeBanners/{$p}" : null;
        $absBook = fn($p) => $p ? "https://scisynopsisconferences.com/uploads/book/{$p}" : null;

        return response()->json([
            'id'                     => $conf->id,
            'conference_first_title' => $conf->conference_first_title,
            'conference_name'        => $conf->conference_name,
            'short_name'             => $conf->short_name,
            'short_alias_name'       => $conf->short_alias_name,
            'conference_theme'       => $conf->conference_theme,
            'from_date'              => $conf->from_date,  // YYYY-mm-dd
            'to_date'                => $conf->to_date,    // YYYY-mm-dd
            'conference_location'    => $conf->conference_location,
            'pink_color'             => $conf->pink_color,
            'blue_color'             => $conf->blue_color,
            'gkey'                   => $conf->gkey,
            'gse'                    => $conf->gse,
            'protocol'               => $conf->protocol,
            'smtp_host'              => $conf->smtp_host,
            'smtp_port'              => $conf->smtp_port,
            'smtp_user'              => $conf->smtp_user,
            'smtp_pass'              => $conf->smtp_pass,
            'smtp_crypto'            => $conf->smtp_crypto,
            'suname'                 => $conf->suname,
            'banner_title'           => $conf->banner_title,
            'home_type'              => $conf->home_type,
            'reg_type'               => $conf->reg_type,

            // files (absolute)
            'home_banner'      => $abs($conf->home_banner),
            'banner_img'       => $abs($conf->banner_img),
            'conf_inf_img'     => $abs($conf->conf_inf_img),
            'sess_img'         => $abs($conf->sess_img),
            'speakers_opinion' => $abs($conf->speakers_opinion),
            'conf_book'        => $absBook($conf->conf_book),
        ]);
    }

    public function saveConferenceInfo(Request $request)
    {
        $token = $request->bearerToken()
            ?? $request->header('authorization')
            ?? $request->input('token')
            ?? $request->query('token');

        $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
        if (!$user) {
            return response()->json(['message' => 'Unauthorized'], 401);
        }

        $conferenceId = $request->input('conference_id');
        if (!$conferenceId) {
            return response()->json(['message' => 'Conference ID missing'], 422);
        }

        // Basic validation (extend as needed)
        $request->validate([
            'conference_name'       => 'nullable|string|max:255',
            'short_name'            => 'nullable|string|max:255',
            'from_date'             => 'nullable|date',
            'to_date'               => 'nullable|date',
            'homeBannerImage'       => 'nullable|file|mimes:jpg,jpeg,png,webp,gif',
            'homeBannerImageLogo'   => 'nullable|file|mimes:jpg,jpeg,png,webp,gif',
            'homeBannerImageLogo1'  => 'nullable|file|mimes:jpg,jpeg,png,webp,gif',
            'homeBannerImageLogo2'  => 'nullable|file|mimes:jpg,jpeg,png,webp,gif',
            'speakers_opinion'      => 'nullable|file|mimes:jpg,jpeg,png,webp,gif',
            'confBook'              => 'nullable|file|mimes:pdf,doc,docx',
        ]);

        // Build params
        $params = [
            'conference_first_title' => $request->input('conference_first_title'),
            'conference_name'        => $request->input('conference_name'),
            'short_name'             => $request->input('short_name'),
            'short_alias_name'       => \Str::slug((string)$request->input('short_name') ?: ''),
            'conference_theme'       => $request->input('conference_theme'),
            'from_date'              => $request->input('from_date'),
            'to_date'                => $request->input('to_date'),
            'conference_location'    => $request->input('conference_location'),
            'pink_color'             => $request->input('pink_color'),
            'blue_color'             => $request->input('blue_color'),
            'gkey'                   => $request->input('gkey'),
            'gse'                    => $request->input('gse'),
            'protocol'               => $request->input('protocol'),
            'smtp_host'              => $request->input('smtp_host'),
            'smtp_port'              => $request->input('smtp_port'),
            'smtp_user'              => $request->input('smtp_user'),
            'smtp_pass'              => $request->input('smtp_pass'),
            'smtp_crypto'            => $request->input('smtp_crypto'),
            'suname'                 => $request->input('suname'),
            'banner_title'           => $request->input('banner_title'),
            'home_type'              => $request->input('home_type'),
            'reg_type'               => $request->input('reg_type'),
        ];

        // Upload physical paths
        $bannersPath = "/home/scisynopsisconfe/public_html/uploads/homeBanners";
        $booksPath   = "/home/scisynopsisconfe/public_html/uploads/book";

        if (!file_exists($bannersPath)) { @mkdir($bannersPath, 0777, true); }
        if (!file_exists($booksPath))   { @mkdir($booksPath, 0777, true); }

        // Map: input name => db column
        $map = [
            'homeBannerImage'      => 'home_banner',
            'homeBannerImageLogo'  => 'banner_img',
            'homeBannerImageLogo1' => 'conf_inf_img',
            'homeBannerImageLogo2' => 'sess_img',
            'speakers_opinion'     => 'speakers_opinion',
        ];

        foreach ($map as $input => $column) {
            if ($request->hasFile($input)) {
                $file = $request->file($input);
                $filename = time().'_'.$column.'.'.$file->getClientOriginalExtension();
                $file->move($bannersPath, $filename);
                $params[$column] = $filename;
            }
        }

        if ($request->hasFile('confBook')) {
            $file = $request->file('confBook');
            $filename = $file->getClientOriginalName(); // keep original name like CI
            $file->move($booksPath, $filename);
            $params['conf_book'] = $filename;
        }

        // Clear flags (mirror CI3 behavior)
        if ($request->boolean('homeBannerImageYes'))      { $params['home_banner']  = ''; }
        if ($request->boolean('homeBannerImageLogoYes'))  { $params['banner_img']   = ''; }
        if ($request->boolean('homeBannerImageLogoYes1')) { $params['conf_inf_img'] = ''; }
        if ($request->boolean('homeBannerImageLogoYes2')) { $params['sess_img']     = ''; }

        // 🔹 OLD row before update
        $old = \DB::table('conferences')->where('id', $conferenceId)->first();

        \DB::table('conferences')->where('id', $conferenceId)->update($params);

        // 🔹 NEW row after update
        $new = \DB::table('conferences')->where('id', $conferenceId)->first();

        // 🔥 Audit
        if ($old && $new) {
            $this->logAudit('conferences', $conferenceId, 'updated', (array) $old, (array) $new);
        }

        return response()->json([
            'message'        => 'Conference info updated successfully',
            'conference_id'  => $conferenceId,
        ]);
    }

    public function getHeaderContent(Request $request)
    {
        $token = $request->bearerToken()
            ?? $request->header('authorization')
            ?? $request->input('token')
            ?? $request->query('token');

        $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
        if (!$user) {
            return response()->json(['message' => 'Unauthorized'], 401);
        }

        $conferenceId = $request->query('conference_id');
        if (!$conferenceId) {
            return response()->json(['message' => 'Conference ID missing'], 422);
        }

        $row = DB::table('c_header_content')
            ->where('conference_id', $conferenceId)
            ->orderBy('id', 'asc')
            ->first();

        if (!$row) {
            // No row yet – return empty defaults so Vue can still bind
            return response()->json([
                'id'                   => null,
                'conference_id'        => (int) $conferenceId,
                'header_content'       => '',
                'registration_content' => '',
            ]);
        }

        return response()->json([
            'id'                   => $row->id,
            'conference_id'        => $row->conference_id,
            'header_content'       => $row->header_content,
            'registration_content' => $row->registration_content,
        ]);
    }

    public function saveHeaderContent(Request $request)
    {
        $token = $request->bearerToken()
            ?? $request->header('authorization')
            ?? $request->input('token')
            ?? $request->query('token');

        $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
        if (!$user) {
            return response()->json(['message' => 'Unauthorized'], 401);
        }

        $conferenceId = $request->input('conference_id');
        if (!$conferenceId) {
            return response()->json(['message' => 'Conference ID missing'], 422);
        }

        $request->validate([
            'header_content'       => 'nullable|string',
            'registration_content' => 'nullable|string',
        ]);

        $params = [
            'conference_id'        => $conferenceId,
            'header_content'       => $request->input('header_content'),
            'registration_content' => $request->input('registration_content'),
        ];

        // Like CI3: use hidden bannersid to decide insert vs update
        $id = $request->input('bannersid');

        if ($id) {
            $exists = DB::table('c_header_content')
                ->where('conference_id', $conferenceId)
                ->where('id', $id)
                ->exists();

            if ($exists) {
                // 🔹 OLD
                $oldHeader = DB::table('c_header_content')
                    ->where('id', $id)
                    ->first();

                DB::table('c_header_content')
                    ->where('id', $id)
                    ->update($params);

                $recordId = $id;

                // 🔹 NEW
                $newHeader = DB::table('c_header_content')
                    ->where('id', $recordId)
                    ->first();

                // 🔥 Audit
                if ($oldHeader && $newHeader) {
                    $this->logAudit('c_header_content', $recordId, 'updated', (array) $oldHeader, (array) $newHeader);
                }
            } else {
                $recordId = DB::table('c_header_content')->insertGetId($params);

                // 🔥 Audit (created)
                $this->logAudit('c_header_content', $recordId, 'created', null, array_merge($params, ['id' => $recordId]));
            }
        } else {
            // If no id passed, upsert by conference_id (keep at most one row per conference)
            $existing = DB::table('c_header_content')
                ->where('conference_id', $conferenceId)
                ->first();

            if ($existing) {
                // 🔹 OLD
                $oldHeader = $existing;

                DB::table('c_header_content')
                    ->where('id', $existing->id)
                    ->update($params);

                $recordId = $existing->id;

                // 🔹 NEW
                $newHeader = DB::table('c_header_content')
                    ->where('id', $recordId)
                    ->first();

                // 🔥 Audit
                if ($oldHeader && $newHeader) {
                    $this->logAudit('c_header_content', $recordId, 'updated', (array) $oldHeader, (array) $newHeader);
                }
            } else {
                $recordId = DB::table('c_header_content')->insertGetId($params);

                // 🔥 Audit (created)
                $this->logAudit('c_header_content', $recordId, 'created', null, array_merge($params, ['id' => $recordId]));
            }
        }

        return response()->json([
            'message'       => 'Header content saved successfully',
            'id'            => $recordId,
            'conference_id' => $conferenceId,
        ]);
    }

    public function deleteHeaderContent(Request $request, $id)
    {
        $token = $request->bearerToken()
            ?? $request->header('authorization')
            ?? $request->input('token')
            ?? $request->query('token');

        $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
        if (!$user) {
            return response()->json(['message' => 'Unauthorized'], 401);
        }

        $conferenceId = $request->query('conference_id');
        if (!$conferenceId) {
            return response()->json(['message' => 'Conference ID missing'], 422);
        }

        // 🔹 OLD row before delete
        $old = DB::table('c_header_content')
            ->where('conference_id', $conferenceId)
            ->where('id', $id)
            ->first();

        $deleted = DB::table('c_header_content')
            ->where('conference_id', $conferenceId)
            ->where('id', $id)
            ->delete();

        if (!$deleted) {
            return response()->json(['message' => 'Header content not found'], 404);
        }

        // 🔥 Audit (deleted)
        if ($old) {
            $this->logAudit('c_header_content', (int) $id, 'deleted', (array) $old, null);
        }

        return response()->json([
            'message'       => 'Header content deleted successfully',
            'id'            => (int) $id,
            'conference_id' => (int) $conferenceId,
        ]);
    }

    /* ============================
       PAST CONFERENCE MANAGEMENT
       ============================ */

    public function getPastConferences(Request $request)
    {
        $token = $request->bearerToken()
            ?? $request->header('authorization')
            ?? $request->input('token')
            ?? $request->query('token');

        $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
        if (!$user) {
            return response()->json(['message' => 'Unauthorized'], 401);
        }

        $conferenceId = $request->query('conference_id');
        if (!$conferenceId) {
            return response()->json(['message' => 'Conference ID missing'], 422);
        }

        $rows = \DB::table('c_past_conferences')
            ->where('conference_id', $conferenceId)
            ->orderBy('id', 'DESC')
            ->get();

        $data = [];
        foreach ($rows as $row) {
            $conf = \DB::table('conferences')->where('id', $row->past_conference_id)->first();
            if ($conf) {
                $data[] = [
                    'id'                 => $row->id,
                    'past_conference_id' => $row->past_conference_id,
                    'conference'         => [
                        'id'               => $conf->id,
                        'short_name'       => $conf->short_name,
                        'conference_name'  => $conf->conference_name,
                    ],
                ];
            }
        }

        return response()->json($data);
    }

    public function storePastConference(Request $request)
    {
        $token = $request->bearerToken()
            ?? $request->header('authorization')
            ?? $request->input('token')
            ?? $request->query('token');

        $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
        if (!$user) {
            return response()->json(['message' => 'Unauthorized'], 401);
        }

        $request->validate([
            'conference_id'      => 'required|integer',
            'past_conference_id' => 'required|integer',
        ]);

        $exists = \DB::table('c_past_conferences')
            ->where('conference_id', $request->conference_id)
            ->where('past_conference_id', $request->past_conference_id)
            ->exists();

        if ($exists) {
            return response()->json(['message' => 'Already exists'], 409);
        }

        \DB::table('c_past_conferences')->insert([
            'conference_id'      => $request->conference_id,
            'past_conference_id' => $request->past_conference_id,
            'status'             => 1,
            'created_date'       => now(),
        ]);

        // 🔥 Audit for created row (find last inserted for this pair)
        $row = \DB::table('c_past_conferences')
            ->where('conference_id', $request->conference_id)
            ->where('past_conference_id', $request->past_conference_id)
            ->orderByDesc('id')
            ->first();

        if ($row) {
            $this->logAudit('c_past_conferences', $row->id, 'created', null, (array) $row);
        }

        return response()->json(['message' => 'Successfully Data Updated']);
    }



// ... previous methods of ConferenceAdminController ...

public function getPastConferenceById(Request $request, $id)
{
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    $row = \DB::table('c_past_conferences')->where('id', $id)->first();
    if (!$row) {
        return response()->json(['message' => 'Not Found'], 404);
    }

    return response()->json($row);
}


public function updatePastConference(Request $request)
{
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    $request->validate([
        'id' => 'required|integer',
        'past_conference_id' => 'required|integer',
    ]);

    // 🔹 OLD row before update
    $old = \DB::table('c_past_conferences')
        ->where('id', $request->id)
        ->first();

    $updated = \DB::table('c_past_conferences')
        ->where('id', $request->id)
        ->update(['past_conference_id' => $request->past_conference_id]);

    if ($updated) {
        // 🔹 NEW row after update
        $new = \DB::table('c_past_conferences')
            ->where('id', $request->id)
            ->first();

        if ($old && $new) {
            $this->logAudit('c_past_conferences', $request->id, 'updated', (array) $old, (array) $new);
        }

        return response()->json(['message' => 'Successfully Data Updated']);
    } else {
        return response()->json(['message' => 'Failed to Update data'], 500);
    }
}


public function deletePastConference(Request $request, $id)
{
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    // 🔹 OLD row before delete
    $old = \DB::table('c_past_conferences')->where('id', $id)->first();

    $deleted = \DB::table('c_past_conferences')->where('id', $id)->delete();

    if ($deleted) {
        if ($old) {
            $this->logAudit('c_past_conferences', $id, 'deleted', (array) $old, null);
        }
        return response()->json(['message' => 'Successfully Deleted']);
    } else {
        return response()->json(['message' => 'Failed to Delete data'], 500);
    }
}

/* ============================
   FIELDWISE SETUP MANAGEMENT
   ============================ */

public function getFieldwiseStatus(Request $request)
{
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    $conferenceId = $request->query('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    $row = \DB::table('c_fieldwise')->where('conference_id', $conferenceId)->first();

    if ($row) {
        return response()->json([
            'id' => $row->id,
            'conference_id' => $row->conference_id,
            'fieldwise_status' => (int) $row->fieldwise_status,
        ]);
    } else {
        // If not found, return default disabled state
        return response()->json([
            'fieldwise_status' => 0,
        ]);
    }
}


public function saveFieldwiseStatus(Request $request)
{
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    $request->validate([
        'conference_id' => 'required|integer',
        'fieldwise_status' => 'required|in:0,1',
    ]);

    $conferenceId = $request->input('conference_id');
    $status = $request->input('fieldwise_status');

    $existing = \DB::table('c_fieldwise')->where('conference_id', $conferenceId)->first();

    if ($existing) {
        // 🔹 OLD
        $old = $existing;

        \DB::table('c_fieldwise')
            ->where('id', $existing->id)
            ->update(['fieldwise_status' => $status]);

        // 🔹 NEW
        $new = \DB::table('c_fieldwise')->where('id', $existing->id)->first();

        if ($new) {
            $this->logAudit('c_fieldwise', $existing->id, 'updated', (array) $old, (array) $new);
        }

        return response()->json(['message' => 'Successfully Updated Fieldwise Status']);
    } else {
        \DB::table('c_fieldwise')->insert([
            'conference_id' => $conferenceId,
            'fieldwise_status' => $status,
        ]);

        // 🔹 Fetch last inserted row for this conference for audit
        $new = \DB::table('c_fieldwise')
            ->where('conference_id', $conferenceId)
            ->orderByDesc('id')
            ->first();

        if ($new) {
            $this->logAudit('c_fieldwise', $new->id, 'created', null, (array) $new);
        }

        return response()->json(['message' => 'Successfully created status record']);
    }
}

/* ============================
   BANNERS MANAGEMENT
   ============================ */

public function getBanners(Request $request)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $conferenceId = $request->query('conference_id');
    $folderName = \DB::table('conferences')->where('id', $conferenceId)->value('unique_id');

    $rows = \DB::table('c_banners')->where('conference_id', $conferenceId)->get();
    foreach ($rows as $r) {
        $r->full_path = "https://scisynopsisconferences.com/uploads/conferences/{$folderName}/{$r->banner_img}";
    }
    return response()->json($rows);
}

public function storeBanner(Request $request)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $request->validate([
        'conference_id' => 'required|integer',
        'mainImage' => 'required|image|mimes:jpg,jpeg,png,webp,gif|max:4096'
    ]);

    $conferenceId = $request->conference_id;
    $folderName = \DB::table('conferences')->where('id', $conferenceId)->value('unique_id');
    $uploadPath = "/home/scisynopsisconfe/public_html/uploads/conferences/{$folderName}";
    if (!file_exists($uploadPath)) mkdir($uploadPath, 0777, true);

    $file = $request->file('mainImage');
    $filename = time() . '.' . $file->getClientOriginalExtension();
    $file->move($uploadPath, $filename);

    \DB::table('c_banners')->insert([
        'conference_id' => $conferenceId,
        'banner_img' => $filename,
        'page_type' => 1,
        'created_date' => now(),
    ]);

    // 🔥 Audit (created)
    $row = \DB::table('c_banners')
        ->where('conference_id', $conferenceId)
        ->where('banner_img', $filename)
        ->orderByDesc('id')
        ->first();

    if ($row) {
        $this->logAudit('c_banners', $row->id, 'created', null, (array) $row);
    }

    return response()->json(['message' => 'Banner uploaded successfully']);
}

public function getBannerById(Request $request, $id)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $row = \DB::table('c_banners')->where('id', $id)->first();
    if (!$row) return response()->json(['message' => 'Not found'], 404);

    $conferenceId = $request->query('conference_id');
    $folderName = \DB::table('conferences')->where('id', $conferenceId)->value('unique_id');
    $row->full_path = "https://scisynopsisconferences.com/uploads/conferences/{$folderName}/{$row->banner_img}";

    return response()->json($row);
}

public function updateBanner(Request $request, $id)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $conferenceId = $request->conference_id;
    $folderName = \DB::table('conferences')->where('id', $conferenceId)->value('unique_id');
    $uploadPath = "/home/scisynopsisconfe/public_html/uploads/conferences/{$folderName}";

    $row = \DB::table('c_banners')->where('id', $id)->first();
    if (!$row) return response()->json(['message' => 'Not found'], 404);

    // 🔹 OLD row
    $old = $row;

    if ($request->hasFile('mainImage')) {
        if (!file_exists($uploadPath)) mkdir($uploadPath, 0777, true);
        $file = $request->file('mainImage');
        $filename = time() . '.' . $file->getClientOriginalExtension();
        $file->move($uploadPath, $filename);

        $oldPath = "{$uploadPath}/{$row->banner_img}";
        if (file_exists($oldPath)) @unlink($oldPath);

        \DB::table('c_banners')->where('id', $id)->update(['banner_img' => $filename]);

        // 🔹 NEW row
        $new = \DB::table('c_banners')->where('id', $id)->first();
        if ($new) {
            $this->logAudit('c_banners', $id, 'updated', (array) $old, (array) $new);
        }
    }

    return response()->json(['message' => 'Banner updated successfully']);
}

public function deleteBanner(Request $request, $id)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $row = \DB::table('c_banners')->where('id', $id)->first();
    if (!$row) return response()->json(['message' => 'Not found'], 404);

    $conferenceId = $request->query('conference_id');
    $folderName = \DB::table('conferences')->where('id', $conferenceId)->value('unique_id');
    $path = "/home/scisynopsisconfe/public_html/uploads/conferences/{$folderName}/{$row->banner_img}";
    if (file_exists($path)) @unlink($path);

    \DB::table('c_banners')->where('id', $id)->delete();

    // 🔥 Audit (deleted)
    $this->logAudit('c_banners', $id, 'deleted', (array) $row, null);

    return response()->json(['message' => 'Banner deleted successfully']);
}

// =======================
// 📘 TESTIMONIALS MODULE
// =======================
public function getTestimonials(Request $request)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $conferenceId = $request->query('conference_id');
    $folderName = DB::table('conferences')->where('id', $conferenceId)->value('unique_id');

    $rows = DB::table('testimonials')->where('conference_id', $conferenceId)->get();
    foreach ($rows as $r) {
        $r->full_path = "https://scisynopsisconferences.com/uploads/testimonials/{$r->testimonial_image}";
    }

    return response()->json($rows);
}

// ✅ Get one testimonial by ID
public function getTestimonialById(Request $request, $id)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $row = DB::table('testimonials')->where('id', $id)->first();
    if (!$row) return response()->json(['message' => 'Not found'], 404);

    $conferenceId = $request->query('conference_id');
    $folderName = DB::table('conferences')->where('id', $conferenceId)->value('unique_id');
    $row->full_path = "https://scisynopsisconferences.com/uploads/testimonials/{$row->testimonial_image}";

    return response()->json($row);
}

// ✅ Create new testimonial
public function storeTestimonial(Request $request)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $request->validate([
        'conference_id' => 'required|integer',
        'testimonial_title' => 'required|string|max:255',
        'sub_title' => 'required|string|max:255',
        'testimonial_desc' => 'required|string',
        'mainImage' => 'required|image|mimes:jpg,jpeg,png,webp,gif|max:4096'
    ]);

    $conferenceId = $request->conference_id;
    $folderName = DB::table('conferences')->where('id', $conferenceId)->value('unique_id');

    $uploadPath = "/home/scisynopsisconfe/public_html/uploads/testimonials";
    if (!file_exists($uploadPath)) mkdir($uploadPath, 0777, true);

    $file = $request->file('mainImage');
    $filename = time() . '.' . $file->getClientOriginalExtension();
    $file->move($uploadPath, $filename);

    DB::table('testimonials')->insert([
        'conference_id' => $conferenceId,
        'testimonial_title' => $request->testimonial_title,
        'sub_title' => $request->sub_title,
        'testimonial_desc' => $request->testimonial_desc,
        'testimonial_image' => $filename,
    ]);

    // 🔥 Audit (created)
    $row = DB::table('testimonials')
        ->where('conference_id', $conferenceId)
        ->where('testimonial_image', $filename)
        ->orderByDesc('id')
        ->first();

    if ($row) {
        $this->logAudit('testimonials', $row->id, 'created', null, (array) $row);
    }

    return response()->json(['message' => 'Testimonial added successfully']);
}

// ✅ Update testimonial
public function updateTestimonial(Request $request, $id)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $row = DB::table('testimonials')->where('id', $id)->first();
    if (!$row) return response()->json(['message' => 'Not found'], 404);

    $conferenceId = $request->conference_id;
    $folderName = DB::table('conferences')->where('id', $conferenceId)->value('unique_id');
    $uploadPath = "/home/scisynopsisconfe/public_html/uploads/testimonials";

    // 🔹 OLD
    $old = $row;

    $updateData = [
        'testimonial_title' => $request->testimonial_title,
        'sub_title' => $request->sub_title,
        'testimonial_desc' => $request->testimonial_desc,
    ];

    if ($request->hasFile('mainImage')) {
        if (!file_exists($uploadPath)) mkdir($uploadPath, 0777, true);

        $file = $request->file('mainImage');
        $filename = time() . '.' . $file->getClientOriginalExtension();
        $file->move($uploadPath, $filename);

        // delete old image
        $oldPath = "{$uploadPath}/{$row->testimonial_image}";
        if (file_exists($oldPath)) @unlink($oldPath);

        $updateData['testimonial_image'] = $filename;
    }

    DB::table('testimonials')->where('id', $id)->update($updateData);

    // 🔹 NEW
    $new = DB::table('testimonials')->where('id', $id)->first();
    if ($new) {
        $this->logAudit('testimonials', $id, 'updated', (array) $old, (array) $new);
    }

    return response()->json(['message' => 'Testimonial updated successfully']);
}

// ✅ Delete testimonial
public function deleteTestimonial(Request $request, $id)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $row = DB::table('testimonials')->where('id', $id)->first();
    if (!$row) return response()->json(['message' => 'Not found'], 404);

    $conferenceId = $request->query('conference_id');
    $folderName = DB::table('conferences')->where('id', $conferenceId)->value('unique_id');
    $path = "/home/scisynopsisconfe/public_html/uploads/testimonials/{$row->testimonial_image}";
    if (file_exists($path)) @unlink($path);

    DB::table('testimonials')->where('id', $id)->delete();

    // 🔥 Audit (deleted)
    $this->logAudit('testimonials', $id, 'deleted', (array) $row, null);

    return response()->json(['message' => 'Testimonial deleted successfully']);
}


// ===============================
// ✅ Testimonials Speaker Says (Vue API)
// ===============================
public function getTestimonialsSpeakers(Request $request)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $conferenceId = $request->query('conference_id');
    $rows = \DB::table('testimonialspeakersays')
        ->where('conference_id', $conferenceId)
        ->orderBy('id', 'asc')
        ->get();

    return response()->json($rows);
}

public function storeTestimonialSpeaker(Request $request)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $request->validate([
        'conference_id' => 'required|integer',
        'testimonial_title' => 'required|string|max:255',
        'testimonial_desc' => 'nullable|string',
    ]);

    \DB::table('testimonialspeakersays')->insert([
        'conference_id' => $request->conference_id,
        'testimonial_title' => $request->testimonial_title,
        'testimonial_desc' => $request->testimonial_desc,
        'createDate' => now(),
    ]);

    // 🔥 Audit (created)
    $row = \DB::table('testimonialspeakersays')
        ->where('conference_id', $request->conference_id)
        ->where('testimonial_title', $request->testimonial_title)
        ->orderByDesc('id')
        ->first();

    if ($row) {
        $this->logAudit('testimonialspeakersays', $row->id, 'created', null, (array) $row);
    }

    return response()->json(['message' => 'Speaker testimonial created successfully']);
}

public function showTestimonialSpeaker(Request $request, $id)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $row = \DB::table('testimonialspeakersays')->where('id', $id)->first();
    if (!$row) return response()->json(['message' => 'Not found'], 404);

    return response()->json($row);
}

public function updateTestimonialSpeaker(Request $request, $id)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $request->validate([
        'conference_id' => 'required|integer',
        'testimonial_title' => 'required|string|max:255',
        'testimonial_desc' => 'nullable|string',
    ]);

    // 🔹 OLD
    $old = \DB::table('testimonialspeakersays')->where('id', $id)->first();

    \DB::table('testimonialspeakersays')
        ->where('id', $id)
        ->update([
            'testimonial_title' => $request->testimonial_title,
            'testimonial_desc' => $request->testimonial_desc,
        ]);

    // 🔹 NEW
    $new = \DB::table('testimonialspeakersays')->where('id', $id)->first();
    if ($old && $new) {
        $this->logAudit('testimonialspeakersays', $id, 'updated', (array) $old, (array) $new);
    }

    return response()->json(['message' => 'Speaker testimonial updated successfully']);
}

public function deleteTestimonialSpeaker(Request $request, $id)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    // 🔹 OLD
    $old = \DB::table('testimonialspeakersays')->where('id', $id)->first();

    \DB::table('testimonialspeakersays')->where('id', $id)->delete();

    if ($old) {
        $this->logAudit('testimonialspeakersays', $id, 'deleted', (array) $old, null);
    }

    return response()->json(['message' => 'Speaker testimonial deleted successfully']);
}

/* ============================
   SCIENTIFIC TRACKS MANAGEMENT
   ============================ */

public function getTracks(Request $request)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $conferenceId = $request->query('conference_id');
    $parentId = $request->query('parent_id', 0);
    $folderName = DB::table('conferences')->where('id', $conferenceId)->value('unique_id');

    $tracks = \DB::table('c_session_tracks')
        ->where('conference_id', $conferenceId)
        ->where('parent_id', $parentId)
        ->orderBy('orderpos', 'ASC')
        ->get();

    foreach ($tracks as $r) {
        $r->full_path = "https://scisynopsisconferences.com/uploads/conferences/{$folderName}/{$r->track_icon}";
    }
    return response()->json($tracks);
}

public function getTrackById(Request $request, $id)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $track = \DB::table('c_session_tracks')->where('id', $id)->first();
    if (!$track) return response()->json(['message' => 'Not found'], 404);

    $conferenceId = $request->query('conference_id');
    $folderName = \DB::table('conferences')->where('id', $conferenceId)->value('unique_id');
    $track->full_path = "https://scisynopsisconferences.com/uploads/conferences/{$folderName}/{$track->track_icon}";
    return response()->json($track);
}

public function storeTrack(Request $request)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $request->validate([
        'conference_id' => 'required|integer',
        'track_title'   => 'required|string|max:255',
        'track_desc'    => 'nullable|string',
        'mainImage'     => 'nullable|image|mimes:jpg,jpeg,png,webp,gif|max:4096',
    ]);

    $conferenceId = $request->conference_id;
    $parentId = $request->input('parentId', 0);

    $folderName = \DB::table('conferences')->where('id', $conferenceId)->value('unique_id');
    $uploadPath = "/home/scisynopsisconfe/public_html/uploads/conferences/{$folderName}";
    if (!file_exists($uploadPath)) mkdir($uploadPath, 0777, true);

    $iconFile = null;
    if ($request->hasFile('mainImage')) {
        $file = $request->file('mainImage');
        $iconFile = 'track_' . time() . '.' . $file->getClientOriginalExtension();
        $file->move($uploadPath, $iconFile);
    }

    $orderpos = (\DB::table('c_session_tracks')->where('conference_id', $conferenceId)->max('orderpos') ?? 0) + 1;

    \DB::table('c_session_tracks')->insert([
        'conference_id' => $conferenceId,
        'track_title'   => $request->track_title,
        'track_desc'    => $request->track_desc,
        'track_icon'    => $iconFile,
        'orderpos'      => $orderpos,
        'parent_id'     => $parentId,
    ]);

    // 🔥 Audit (created)
    $row = \DB::table('c_session_tracks')
        ->where('conference_id', $conferenceId)
        ->where('track_title', $request->track_title)
        ->orderByDesc('id')
        ->first();

    if ($row) {
        $this->logAudit('c_session_tracks', $row->id, 'created', null, (array) $row);
    }

    return response()->json(['message' => 'Track created successfully']);
}

public function updateTrack(Request $request, $id)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $request->validate([
        'conference_id' => 'required|integer',
        'track_title'   => 'required|string|max:255',
        'track_desc'    => 'nullable|string',
        'mainImage'     => 'nullable|image|mimes:jpg,jpeg,png,webp,gif|max:4096',
    ]);

    $conferenceId = $request->conference_id;
    $folderName = \DB::table('conferences')->where('id', $conferenceId)->value('unique_id');
    $uploadPath = "/home/scisynopsisconfe/public_html/uploads/conferences/{$folderName}";

    $row = \DB::table('c_session_tracks')->where('id', $id)->first();
    if (!$row) return response()->json(['message' => 'Not found'], 404);

    // 🔹 OLD
    $old = $row;

    $iconFile = $row->track_icon;
    if ($request->hasFile('mainImage')) {
        if (!file_exists($uploadPath)) mkdir($uploadPath, 0777, true);
        $file = $request->file('mainImage');
        $iconFile = 'track_' . time() . '.' . $file->getClientOriginalExtension();
        $file->move($uploadPath, $iconFile);

        $oldPath = "{$uploadPath}/{$row->track_icon}";
        if (file_exists($oldPath)) @unlink($oldPath);
    }

    \DB::table('c_session_tracks')->where('id', $id)->update([
        'track_title' => $request->track_title,
        'track_desc'  => $request->track_desc,
        'track_icon'  => $iconFile,
    ]);

    // 🔹 NEW
    $new = \DB::table('c_session_tracks')->where('id', $id)->first();
    if ($new) {
        $this->logAudit('c_session_tracks', $id, 'updated', (array) $old, (array) $new);
    }

    return response()->json(['message' => 'Track updated successfully']);
}

public function deleteTrack(Request $request, $id)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $row = \DB::table('c_session_tracks')->where('id', $id)->first();
    if (!$row) return response()->json(['message' => 'Not found'], 404);

    $conferenceId = $request->query('conference_id');
    $folderName = \DB::table('conferences')->where('id', $conferenceId)->value('unique_id');
    $path = "/home/scisynopsisconfe/public_html/uploads/conferences/{$folderName}/{$row->track_icon}";
    if (file_exists($path)) @unlink($path);

    \DB::table('c_session_tracks')->where('id', $id)->delete();

    // 🔥 Audit (deleted)
    $this->logAudit('c_session_tracks', $id, 'deleted', (array) $row, null);

    return response()->json(['message' => 'Track deleted successfully']);
}

public function sortTracks(Request $request)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $request->validate([
        'conference_id' => 'required|integer',
        'sort_order'    => 'required|string',
    ]);

    $ids = explode(',', $request->sort_order);
    foreach ($ids as $i => $id) {
        $id = (int) $id;
        if (!$id) continue;

        // 🔹 OLD
        $old = \DB::table('c_session_tracks')->where('id', $id)->first();

        \DB::table('c_session_tracks')->where('id', $id)->update(['orderpos' => $i + 1]);

        // 🔹 NEW
        $new = \DB::table('c_session_tracks')->where('id', $id)->first();

        if ($old && $new) {
            $this->logAudit('c_session_tracks', $id, 'updated', (array) $old, (array) $new);
        }
    }

    return response()->json(['message' => 'Track order updated successfully']);
}

/* ============================
   PAGE SETTINGS MANAGEMENT
   ============================ */
public function getPageSettings(Request $request)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $conferenceId = $request->query('conference_id');
    if (!$conferenceId) return response()->json(['message' => 'Conference ID missing'], 422);

    $rows = \DB::table('c_pages')
        ->where('conference_id', $conferenceId)
        ->orderBy('id', 'DESC')
        ->get();

    foreach ($rows as $r) {
        $r->page_name = match((int)$r->page_type) {
            1 => 'Abstract Program',
            2 => 'Organizing Committee',
            default => 'Unknown',
        };
    }

    return response()->json($rows);
}

public function storePageSetting(Request $request)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $request->validate([
        'conference_id' => 'required|integer',
        'page_type' => 'required|integer|in:1,2'
    ]);

    $exists = \DB::table('c_pages')
        ->where('conference_id', $request->conference_id)
        ->where('page_type', $request->page_type)
        ->exists();

    if ($exists) {
        return response()->json(['message' => 'Page already exists'], 409);
    }

    \DB::table('c_pages')->insert([
        'conference_id' => $request->conference_id,
        'page_type' => $request->page_type,
        'created_date' => now(),
    ]);

    // 🔥 Audit (created)
    $row = \DB::table('c_pages')
        ->where('conference_id', $request->conference_id)
        ->where('page_type', $request->page_type)
        ->orderByDesc('id')
        ->first();

    if ($row) {
        $this->logAudit('c_pages', $row->id, 'created', null, (array) $row);
    }

    return response()->json(['message' => 'Page setting created successfully']);
}

public function deletePageSetting(Request $request, $id)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    // 🔹 OLD
    $old = \DB::table('c_pages')->where('id', $id)->first();

    $deleted = \DB::table('c_pages')->where('id', $id)->delete();

    if ($deleted) {
        if ($old) {
            $this->logAudit('c_pages', $id, 'deleted', (array) $old, null);
        }
        return response()->json(['message' => 'Successfully Deleted']);
    }

    return response()->json(['message' => 'Failed to delete'], 500);
}

public function listPageNavigations(Request $request)
{
    $conferenceId = $request->get('conference_id');
    $page_type = $request->get('page_type');
    $rows = DB::table('c_page_navigations')
        ->where('conference_id', $conferenceId)
        ->where('page_type', $page_type)
        ->where('parent_id', 0)
        ->orderBy('orderpos', 'ASC')
        ->get();

    return response()->json($rows);
}

public function createPageNavigation(Request $request)
{
    $conferenceId = $request->get('conference_id');
    $page_type = $request->input('page_type', 1);
    $targetchk = $request->input('targetchk', 2);
    $page_url_name = $request->input('page_url_name');
    $page_name = $request->input('page_name');

    $count = DB::table('c_page_navigations')
        ->where('conference_id', $conferenceId)
        ->where('page_type', $page_type)
        ->where('parent_id', 0)
        ->count();

    $orderpos = $count + 1;

    $inserted = DB::table('c_page_navigations')->insert([
        'conference_id' => $conferenceId,
        'page_name'       => $page_name,
        'page_url_name'   => $page_url_name,
        'targetchk'       => $targetchk,
        'page_type'       => $page_type,
        'parent_id'       => 0,
        'orderpos'        => $orderpos,
        'is_url'          => 1,
        'created_date'    => now(),
    ]);

    if ($inserted) {
        // 🔥 Audit (created)
        $row = DB::table('c_page_navigations')
            ->where('conference_id', $conferenceId)
            ->where('page_type', $page_type)
            ->where('parent_id', 0)
            ->where('page_name', $page_name)
            ->orderByDesc('id')
            ->first();

        if ($row) {
            $this->logAudit('c_page_navigations', $row->id, 'created', null, (array) $row);
        }

        return response()->json(['message' => 'Successfully Saved Data'], 201);
    }
    return response()->json(['message' => 'Failed to save the data'], 500);
}

public function getPageNavigation($id)
{
    $row = DB::table('c_page_navigations')->where('id', $id)->first();
    if (!$row) {
        return response()->json(['message' => 'Not Found'], 404);
    }
    return response()->json($row);
}

public function updatePageNavigation(Request $request, $id)
{
    $targetchk = $request->input('targetchk', 2);

    // 🔹 OLD
    $old = DB::table('c_page_navigations')->where('id', $id)->first();

    $updated = DB::table('c_page_navigations')
        ->where('id', $id)
        ->update([
            'page_name'       => $request->input('page_name'),
            'page_url_name'   => $request->input('page_url_name'),
            'targetchk'       => $targetchk,
            'page_type'       => $request->input('page_type'),
            'is_url'          => 1,
            'created_date'    => now(),
        ]);

    if ($updated) {
        // 🔹 NEW
        $new = DB::table('c_page_navigations')->where('id', $id)->first();
        if ($old && $new) {
            $this->logAudit('c_page_navigations', $id, 'updated', (array) $old, (array) $new);
        }

        return response()->json(['message' => 'Successfully Saved Data']);
    }
    return response()->json(['message' => 'Failed to save the data'], 500);
}

public function deletePageNavigation($id)
{
    // 🔹 OLD
    $old = DB::table('c_page_navigations')->where('id', $id)->first();

    $deleted = DB::table('c_page_navigations')->where('id', $id)->delete();

    if ($deleted) {
        if ($old) {
            $this->logAudit('c_page_navigations', $id, 'deleted', (array) $old, null);
        }
        return response()->json(['message' => 'Successfully Deleted']);
    }
    return response()->json(['message' => 'Failed to Delete data'], 500);
}

public function sortPageNavigations(Request $request)
{
    $ids = $request->input('sort_order'); // e.g. "5,3,2"
    $conferenceId = $request->input('conference_id');
    $page_type = $request->input('page_type', 1);

    if (!is_array($ids)) {
        $ids = explode(',', $ids);
    }

    foreach ($ids as $index => $id) {
        // 🔹 OLD
        $old = DB::table('c_page_navigations')
            ->where('id', $id)
            ->where('conference_id', $conferenceId)
            ->where('page_type', $page_type)
            ->first();

        DB::table('c_page_navigations')
            ->where('id', $id)
            ->where('conference_id', $conferenceId)
            ->where('page_type', $page_type)
            ->update(['orderpos' => ($index + 1)]);

        // 🔹 NEW
        $new = DB::table('c_page_navigations')->where('id', $id)->first();
        if ($old && $new) {
            $this->logAudit('c_page_navigations', $id, 'updated', (array) $old, (array) $new);
        }
    }

    return response()->json(['message' => 'Successfully Changed The Order']);
}

/* ============================
   SUB NAVIGATIONS (Child)
   ============================ */

public function listSubNavigations(Request $request, $pageType, $pageId)
{
    $conferenceId = $request->query('conference_id', 1);

    $rows = DB::table('c_page_navigations')
        ->where('conference_id', $conferenceId)
        ->where('page_type', $pageType)
        ->where('parent_id', $pageId)
        ->orderBy('orderpos', 'ASC')
        ->get()
        ->map(function ($row) use ($pageType, $pageId) {
            $folder = DB::table('conferences')
                ->where('id', $row->conference_id)
                ->value('folder_name');

            $row->page_url = match ($row->page_url_type) {
                1 => url($row->page_url_name),
                2 => url("conferences/{$folder}/{$row->page_url_name}"),
                3 => $row->page_url_name,
                default => '',
            };

            $row->actions = '
                <a href="/conferenceadmin/subnavigations/edit/' . $pageType . '/' . $pageId . '/' . $row->id . '">
                    <i class="fa fa-pencil"></i>
                </a> |
                <a href="/conferenceadmin/subnavigations/delete/' . $row->id . '/' . $pageType . '/' . $pageId . '">
                    <i class="fa fa-trash"></i>
                </a>';

            return $row;
        });

    return response()->json(['data' => $rows]);
}

public function createSubNavigation(Request $request)
{
    $conferenceId = $request->get('conference_id');
    $page_type = $request->input('page_type', 1);
    $pageId   = $request->input('page_id');
    $targetchk = $request->input('targetchk', 2);
    $page_name = $request->input('page_name');
    $page_url_name = $request->input('page_url_name');

    $count = DB::table('c_page_navigations')
        ->where('conference_id', $conferenceId)
        ->where('page_type', $page_type)
        ->where('parent_id', $pageId)
        ->count();

    $orderpos = $count + 1;

    $inserted = DB::table('c_page_navigations')->insert([
        'conference_id'   => $conferenceId,
        'parent_id'       => $pageId,
        'page_name'       => $page_name,
        'page_url_name'   => $page_url_name,
        'targetchk'       => $targetchk,
        'page_type'       => $page_type,
        'orderpos'        => $orderpos,
        'is_url'          => 1,
        'created_date'    => now(),
    ]);

    if ($inserted) {
        // 🔥 Audit (created)
        $row = DB::table('c_page_navigations')
            ->where('conference_id', $conferenceId)
            ->where('page_type', $page_type)
            ->where('parent_id', $pageId)
            ->where('page_name', $page_name)
            ->orderByDesc('id')
            ->first();

        if ($row) {
            $this->logAudit('c_page_navigations', $row->id, 'created', null, (array) $row);
        }

        return response()->json(['message' => 'Successfully Saved Data'], 201);
    }
    return response()->json(['message' => 'Failed to save the data'], 500);
}

public function getSubNavigation($id)
{
    $row = DB::table('c_page_navigations')->where('id', $id)->first();
    if (!$row) {
        return response()->json(['message' => 'Not Found'], 404);
    }
    return response()->json($row);
}

public function updateSubNavigation(Request $request, $id)
{
    $targetchk = $request->input('targetchk', 2);

    // 🔹 OLD
    $old = DB::table('c_page_navigations')->where('id', $id)->first();

    $updated = DB::table('c_page_navigations')
        ->where('id', $id)
        ->update([
            'parent_id'       => $request->input('page_id'),
            'page_name'       => $request->input('page_name'),
            'page_url_name'   => $request->input('page_url_name'),
            'targetchk'       => $targetchk,
            'page_type'       => $request->input('page_type'),
            'is_url'          => 1,
            'created_date'    => now(),
        ]);

    if ($updated) {
        // 🔹 NEW
        $new = DB::table('c_page_navigations')->where('id', $id)->first();
        if ($old && $new) {
            $this->logAudit('c_page_navigations', $id, 'updated', (array) $old, (array) $new);
        }

        return response()->json(['message' => 'Successfully Saved Data']);
    }
    return response()->json(['message' => 'Failed to save the data'], 500);
}

public function deleteSubNavigation($id)
{
    // 🔹 OLD
    $old = DB::table('c_page_navigations')->where('id', $id)->first();

    $deleted = DB::table('c_page_navigations')->where('id', $id)->delete();

    if ($deleted) {
        if ($old) {
            $this->logAudit('c_page_navigations', $id, 'deleted', (array) $old, null);
        }
        return response()->json(['message' => 'Successfully Deleted']);
    }
    return response()->json(['message' => 'Failed to Delete data'], 500);
}

public function sortSubNavigations(Request $request)
{
    $ids = $request->input('sort_order'); // e.g. "7,4,12"
    $conferenceId = $request->input('conference_id');
    $page_type = $request->input('page_type');
    $pageId = $request->input('page_id');

    if (!is_array($ids)) {
        $ids = explode(',', $ids);
    }

    foreach ($ids as $index => $id) {
        // 🔹 OLD
        $old = DB::table('c_page_navigations')
            ->where('id', $id)
            ->where('conference_id', $conferenceId)
            ->where('page_type', $page_type)
            ->where('parent_id', $pageId)
            ->first();

        DB::table('c_page_navigations')
            ->where('id', $id)
            ->where('conference_id', $conferenceId)
            ->where('page_type', $page_type)
            ->where('parent_id', $pageId)
            ->update(['orderpos' => ($index + 1)]);

        // 🔹 NEW
        $new = DB::table('c_page_navigations')->where('id', $id)->first();
        if ($old && $new) {
            $this->logAudit('c_page_navigations', $id, 'updated', (array) $old, (array) $new);
        }
    }

    return response()->json(['message' => 'Successfully Changed The Order']);
}

public function listPageNavigations2(Request $request, $page_type = null)
{
    $conferenceId = $request->get('conference_id', 1);
    $page_type = $page_type ?? 1;

    $rows = DB::table('c_page_navigations')
        ->where('conference_id', $conferenceId)
        ->where('page_type', $page_type)
        ->where('parent_id', 0)
        ->orderBy('orderpos', 'ASC')
        ->get()
        ->map(function ($row) use ($page_type) {
            $folder = DB::table('conferences')
                ->where('id', $row->conference_id)
                ->value('folder_name');

            $row->page_url = match ($row->page_url_type) {
                1 => url($row->page_url_name),
                2 => url("conferences/{$folder}/{$row->page_url_name}"),
                3 => $row->page_url_name,
                default => '',
            };

            $row->subpages = $row->page_url_name
                ? 'N/A'
                : '<a href="/conferenceadmin/subnavigations/' . $page_type . '/' . $row->id . '">View Sub Page Navigation</a>';

            $row->actions = '
                <a href="/conferenceadmin/pagenavigations/edit/' . $page_type . '/' . $row->id . '">
                    <i class="fa fa-pencil"></i>
                </a> |
                <a href="/conferenceadmin/pagenavigations/delete/' . $row->id . '/' . $page_type . '">
                    <i class="fa fa-trash"></i>
                </a>';

            return $row;
        });

    return response()->json(['data' => $rows]);
}

// =============================
// 📬 ENQUIRIES MANAGEMENT
// =============================

public function getEnquiries(Request $request)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $conferenceId = $request->query('conference_id');
    $type = $request->query('type');

    if (!$conferenceId || !$type) {
        return response()->json(['message' => 'Missing parameters'], 400);
    }

    $enquiries = \DB::table('enquiries')
        ->where('conference_id', $conferenceId)
        ->where('enq_type', $type)
        ->orderBy('id', 'DESC')
        ->get();

    return response()->json($enquiries);
}

public function deleteEnquiry($id, Request $request)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    // 🔹 OLD
    $old = \DB::table('enquiries')->where('id', $id)->first();

    $deleted = \DB::table('enquiries')->where('id', $id)->delete();

    if ($deleted) {
        if ($old) {
            $this->logAudit('enquiries', $id, 'deleted', (array) $old, null);
        }
        return response()->json(['message' => 'Successfully Deleted']);
    }
    return response()->json(['message' => 'Failed to Delete data'], 500);
}

public function deleteAllEnquiries(Request $request)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $ids = $request->input('ids');
    $conferenceId = $request->input('conference_id');

    if (empty($ids) || !$conferenceId) {
        return response()->json(['message' => 'Missing parameters'], 400);
    }

    // 🔹 OLD rows
    $rows = \DB::table('enquiries')
        ->where('conference_id', $conferenceId)
        ->whereIn('id', $ids)
        ->get();

    $deleted = \DB::table('enquiries')
        ->where('conference_id', $conferenceId)
        ->whereIn('id', $ids)
        ->delete();

    if ($deleted) {
        foreach ($rows as $row) {
            $this->logAudit('enquiries', $row->id, 'deleted', (array) $row, null);
        }
        return response()->json(['message' => 'Selected enquiries deleted successfully']);
    }

    return response()->json(['message' => 'Failed to delete selected enquiries'], 500);
}


/* ================================================================
 |   C-META MANAGEMENT (Laravel API for Vue)
 * ================================================================ */
public function getCmeta(Request $request)
{
    $conferenceId = $request->query('conference_id');

    if (!$conferenceId) {
        return response()->json([], 200);
    }

    $data = DB::table('c_meta_data')
        ->where('conference_id', $conferenceId)
        ->get();

    return response()->json($data);
}

public function getCmetaById($id)
{
    $data = DB::table('c_meta_data')
        ->where('id', $id)
        ->first();

    return response()->json($data);
}

public function saveCmeta(Request $request)
{
    $validated = $request->validate([
        'conference_id' => 'required',
        'page_type' => 'required',
        'meta_title' => 'required|string',
        'meta_desc' => 'required|string',
        'meta_keywords' => 'required|string',
    ]);

    DB::table('c_meta_data')->insert($validated);

    // 🔥 Audit (created)
    $row = DB::table('c_meta_data')
        ->where('conference_id', $validated['conference_id'])
        ->where('page_type', $validated['page_type'])
        ->orderByDesc('id')
        ->first();

    if ($row) {
        $this->logAudit('c_meta_data', $row->id, 'created', null, (array) $row);
    }

    return response()->json(['success' => true, 'message' => 'Meta created successfully']);
}

public function updateCmetaById(Request $request, $id)
{
    $validated = $request->validate([
        'meta_title' => 'required|string',
        'meta_desc' => 'required|string',
        'meta_keywords' => 'required|string',
    ]);

    // 🔹 OLD
    $old = DB::table('c_meta_data')->where('id', $id)->first();

    DB::table('c_meta_data')->where('id', $id)->update($validated);

    // 🔹 NEW
    $new = DB::table('c_meta_data')->where('id', $id)->first();
    if ($old && $new) {
        $this->logAudit('c_meta_data', $id, 'updated', (array) $old, (array) $new);
    }

    return response()->json(['success' => true, 'message' => 'Meta updated successfully']);
}

public function deleteCmetaById($id)
{
    // 🔹 OLD
    $old = DB::table('c_meta_data')->where('id', $id)->first();

    DB::table('c_meta_data')->where('id', $id)->delete();

    if ($old) {
        $this->logAudit('c_meta_data', $id, 'deleted', (array) $old, null);
    }

    return response()->json(['success' => true, 'message' => 'Meta deleted successfully']);
}


/* FOOTER SECTION STARTS */
public function getFooterSections(Request $request)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $conferenceId = $request->query('conference_id');
    $sections = \DB::table('c_footer_section')
        ->where('conference_id', $conferenceId)
        ->orderBy('id', 'ASC')
        ->get();

    return response()->json($sections);
}

public function saveFooterSection(Request $request)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $conferenceId = $request->conference_id;
    $params = [
        'conference_id' => $conferenceId,
        'status' => $request->status,
        'page_type' => $request->page_type,
    ];

    $exists = \DB::table('c_footer_section')
        ->where('conference_id', $conferenceId)
        ->where('page_type', $request->page_type)
        ->exists();

    if ($exists) {
        return response()->json(['message' => 'Already Exists'], 409);
    }

    \DB::table('c_footer_section')->insert($params);

    // 🔥 Audit (created)
    $row = \DB::table('c_footer_section')
        ->where('conference_id', $conferenceId)
        ->where('page_type', $request->page_type)
        ->orderByDesc('id')
        ->first();

    if ($row) {
        $this->logAudit('c_footer_section', $row->id, 'created', null, (array) $row);
    }

    return response()->json(['message' => 'Successfully Saved']);
}

public function updateFooterSection($id, Request $request)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $status = $request->status;

    // 🔹 OLD
    $old = \DB::table('c_footer_section')->where('id', $id)->first();

    \DB::table('c_footer_section')->where('id', $id)->update(['status' => $status]);

    // 🔹 NEW
    $new = \DB::table('c_footer_section')->where('id', $id)->first();
    if ($old && $new) {
        $this->logAudit('c_footer_section', $id, 'updated', (array) $old, (array) $new);
    }

    return response()->json(['message' => 'Successfully Updated']);
}
/* END FOOTER SECTION */

// =============================
// Program Speakers (Sort)
// =============================
public function getProgramSpeakersForSort(Request $request)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $conferenceId = $request->query('conference_id');
    if (!$conferenceId) return response()->json(['message' => 'Conference ID missing'], 422);

    // Mirror CI ordering (by orderpos ASC)
    $rows = \DB::table('c_program_speakers')
        ->where('conference_id', $conferenceId)
        ->orderBy('orderpos', 'ASC')
        ->get();

    return response()->json($rows);
}

public function saveProgramSpeakersSortOrder(Request $request)
{
    $user = \App\Models\Conferenceadmin\User::where('api_token', $request->bearerToken())->first();
    if (!$user) return response()->json(['message' => 'Unauthorized'], 401);

    $request->validate([
        'conference_id' => 'required|integer',
        'sort_order'    => 'required|string', // "12,7,4,19"
    ]);

    $ids = preg_split('/\s*,\s*/', trim($request->input('sort_order')), -1, PREG_SPLIT_NO_EMPTY);

    // Keep it close to CI: set as index+101 (i+100+1)
    foreach ($ids as $i => $id) {
        // 🔹 OLD
        $old = \DB::table('c_program_speakers')
            ->where('id', $id)
            ->where('conference_id', $request->conference_id)
            ->first();

        \DB::table('c_program_speakers')
            ->where('id', $id)
            ->where('conference_id', $request->conference_id)
            ->update(['orderpos' => ($i + 101)]);

        // 🔹 NEW
        $new = \DB::table('c_program_speakers')->where('id', $id)->first();
        if ($old && $new) {
            $this->logAudit('c_program_speakers', $id, 'updated', (array) $old, (array) $new);
        }
    }

    return response()->json(['message' => 'Successfully Changed The Order']);
}



// =====================================================
// PROGRAM SPEAKERS + SPEAKERS CONTENT API FUNCTIONS
// =====================================================

public function getProgramSpeakers(Request $request)
{
    $conference_id = $request->input('conference_id');

    $speakers = DB::table('c_program_speakers')
        ->where('conference_id', $conference_id)
        ->orderBy('id', 'DESC')
        ->get();

    return response()->json(['data' => $speakers]);
}

public function getProgramSpeakerById($id, Request $request)
{
    $conference_id = $request->input('conference_id');

    $speaker = DB::table('c_program_speakers')
        ->where('id', $id)
        ->where('conference_id', $conference_id)
        ->first();

    return response()->json($speaker);
}

public function storeProgramSpeaker(Request $request)
{
    $conference_id = $request->input('conference_id');

    // ✅ Get unique folder name for this conference
    $uniqueFolder = DB::table('conferences')->where('id', $conference_id)->value('unique_id');
    $uploadPath = "/home/scisynopsisconfe/public_html/uploads/conferences/{$uniqueFolder}";

    if (!File::exists($uploadPath)) {
        File::makeDirectory($uploadPath, 0777, true);
    }

    $speaker_img = null;
    if ($request->hasFile('mainImage')) {
        $file = $request->file('mainImage');
        $speaker_img = time() . '.' . $file->getClientOriginalExtension();
        $file->move($uploadPath, $speaker_img);
    }

    $spk_start_time = preg_replace('/[^0-9]/', '', $request->input('spk_start_time', ''));
    $spk_end_time = preg_replace('/[^0-9]/', '', $request->input('spk_end_time', ''));
    $spk_time_order = $spk_start_time . $spk_end_time;
    $alias_name = strtolower(preg_replace('/\s+/', '-', trim($request->input('speaker_name'))));

    DB::table('c_program_speakers')->insert([
        'conference_id' => $conference_id,
        'sch_day' => $request->input('sch_day'),
        'sTitle' => $request->input('sTitle'),
        'subTitle' => $request->input('subTitle'),
        'spk_start_time' => $request->input('spk_start_time'),
        'spk_end_time' => $request->input('spk_end_time'),
        'spk_time_order' => $spk_time_order,
        'speaker_name' => trim($request->input('speaker_name')),
        'alias_name' => $alias_name,
        'affiliation' => $request->input('affiliation'),
        'continent_id' => $request->input('continent_id'),
        'country_id' => $request->input('country_id'),
        'city_id' => $request->input('city_id'),
        'hospital_id' => $request->input('hospital_id'),
        'university_id' => $request->input('university_id'),
        'speaker_img' => $speaker_img,
        'created_date' => now(),
    ]);

    // 🔥 Audit (created)
    $row = DB::table('c_program_speakers')
        ->where('conference_id', $conference_id)
        ->where('alias_name', $alias_name)
        ->orderByDesc('id')
        ->first();

    if ($row) {
        $this->logAudit('c_program_speakers', $row->id, 'created', null, (array) $row);
    }

    return response()->json(['success' => true, 'message' => 'Speaker saved successfully.']);
}

public function updateProgramSpeaker(Request $request)
{
    $conference_id = $request->input('conference_id');
    $id = $request->input('id');

    $speaker = DB::table('c_program_speakers')
        ->where('id', $id)
        ->where('conference_id', $conference_id)
        ->first();

    if (!$speaker) {
        return response()->json(['error' => 'Speaker not found'], 404);
    }

    // ✅ Use unique_id folder for this conference
    $uniqueFolder = DB::table('conferences')->where('id', $conference_id)->value('unique_id');
    $uploadPath = "/home/scisynopsisconfe/public_html/uploads/conferences/{$uniqueFolder}";

    if (!File::exists($uploadPath)) {
        File::makeDirectory($uploadPath, 0777, true);
    }

    // 🔹 OLD
    $old = $speaker;

    $speaker_img = $speaker->speaker_img;
    if ($request->hasFile('mainImage')) {
        if ($speaker_img && File::exists($uploadPath . '/' . $speaker_img)) {
            @unlink($uploadPath . '/' . $speaker_img);
        }
        $file = $request->file('mainImage');
        $speaker_img = time() . '.' . $file->getClientOriginalExtension();
        $file->move($uploadPath, $speaker_img);
    }

    $spk_start_time = preg_replace('/[^0-9]/', '', $request->input('spk_start_time', ''));
    $spk_end_time = preg_replace('/[^0-9]/', '', $request->input('spk_end_time', ''));
    $spk_time_order = $spk_start_time . $spk_end_time;
    $alias_name = strtolower(preg_replace('/\s+/', '-', trim($request->input('speaker_name'))));

    DB::table('c_program_speakers')
        ->where('id', $id)
        ->update([
            'sch_day' => $request->input('sch_day'),
            'sTitle' => $request->input('sTitle'),
            'subTitle' => $request->input('subTitle'),
            'spk_start_time' => $request->input('spk_start_time'),
            'spk_end_time' => $request->input('spk_end_time'),
            'spk_time_order' => $spk_time_order,
            'speaker_name' => trim($request->input('speaker_name')),
            'alias_name' => $alias_name,
            'affiliation' => $request->input('affiliation'),
            'continent_id' => $request->input('continent_id'),
            'country_id' => $request->input('country_id'),
            'city_id' => $request->input('city_id'),
            'hospital_id' => $request->input('hospital_id'),
            'university_id' => $request->input('university_id'),
            'speaker_img' => $speaker_img,
        ]);

    // 🔹 NEW
    $new = DB::table('c_program_speakers')->where('id', $id)->first();
    if ($new) {
        $this->logAudit('c_program_speakers', $id, 'updated', (array) $old, (array) $new);
    }

    return response()->json(['success' => true, 'message' => 'Speaker updated successfully.']);
}

public function deleteProgramSpeaker($id, Request $request)
{
    $conference_id = $request->input('conference_id');
    $uniqueFolder = DB::table('conferences')->where('id', $conference_id)->value('unique_id');
    $uploadPath = "/home/scisynopsisconfe/public_html/uploads/conferences/{$uniqueFolder}";

    $speaker = DB::table('c_program_speakers')->where('id', $id)->where('conference_id', $conference_id)->first();
    if (!$speaker) return response()->json(['error' => 'Not found'], 404);

    if ($speaker->speaker_img && File::exists($uploadPath . '/' . $speaker->speaker_img)) {
        @unlink($uploadPath . '/' . $speaker->speaker_img);
    }

    DB::table('c_program_speakers')->where('id', $id)->delete();

    // 🔥 Audit (deleted)
    $this->logAudit('c_program_speakers', $id, 'deleted', (array) $speaker, null);

    return response()->json(['success' => true, 'message' => 'Speaker deleted successfully.']);
}

// =====================================================
// SPEAKERS CONTENT FUNCTIONS
// =====================================================

public function getSpeakersContent($speaker_id, Request $request)
{
    $conference_id = $request->input('conference_id');
    $data = DB::table('c_program_content')
        ->where('conference_id', $conference_id)
        ->where('c_program_id', $speaker_id)
        ->orderBy('id', 'DESC')
        ->get();

    return response()->json(['data' => $data]);
}

public function getSpeakersContentById($id, Request $request)
{
    $conference_id = $request->input('conference_id');
    $row = DB::table('c_program_content')
        ->where('conference_id', $conference_id)
        ->where('id', $id)
        ->first();

    return response()->json($row);
}

public function storeSpeakersContent(Request $request)
{
    $conference_id = $request->input('conference_id');
    $speaker_id = $request->input('c_program_id');

    DB::table('c_program_content')->insert([
        'conference_id' => $conference_id,
        'c_program_id' => $speaker_id,
        'content_title' => $request->input('content_title'),
        'content_desc' => $request->input('content_desc'),
        'created_date' => now(),
    ]);

    // 🔥 Audit (created)
    $row = DB::table('c_program_content')
        ->where('conference_id', $conference_id)
        ->where('c_program_id', $speaker_id)
        ->orderByDesc('id')
        ->first();

    if ($row) {
        $this->logAudit('c_program_content', $row->id, 'created', null, (array) $row);
    }

    return response()->json(['success' => true, 'message' => 'Content saved successfully.']);
}

public function updateSpeakersContent(Request $request)
{
    $conference_id = $request->input('conference_id');
    $id = $request->input('id');

    // 🔹 OLD
    $old = DB::table('c_program_content')
        ->where('id', $id)
        ->where('conference_id', $conference_id)
        ->first();

    DB::table('c_program_content')
        ->where('id', $id)
        ->where('conference_id', $conference_id)
        ->update([
            'content_title' => $request->input('content_title'),
            'content_desc' => $request->input('content_desc'),
        ]);

    // 🔹 NEW
    $new = DB::table('c_program_content')->where('id', $id)->first();
    if ($old && $new) {
        $this->logAudit('c_program_content', $id, 'updated', (array) $old, (array) $new);
    }

    return response()->json(['success' => true, 'message' => 'Content updated successfully.']);
}

public function deleteSpeakersContent($id, Request $request)
{
    $conference_id = $request->input('conference_id');

    // 🔹 OLD
    $old = DB::table('c_program_content')
        ->where('id', $id)
        ->where('conference_id', $conference_id)
        ->first();

    DB::table('c_program_content')
        ->where('id', $id)
        ->where('conference_id', $conference_id)
        ->delete();

    if ($old) {
        $this->logAudit('c_program_content', $id, 'deleted', (array) $old, null);
    }

    return response()->json(['success' => true, 'message' => 'Content deleted successfully.']);
}



public function pastGallery(Request $request)
{
    $conferenceId = $request->input('conference_id');

    // Get other active conferences
    $conferences = DB::table('conferences')
        ->where('status', 1)
        ->where('id', '!=', $conferenceId)
        ->where('short_name', '!=', '')
        ->orderBy('short_name', 'ASC')
        ->select('id', 'short_name')
        ->get();

    // Get existing past gallery record
    $prev = DB::table('c_home_past_gallery')
        ->where('conference_id', $conferenceId)
        ->where('page_type', 1)
        ->first();

    return response()->json([
        'conferences' => $conferences,
        'prev' => $prev
    ]);
}

public function savePastGallery(Request $request)
{
    $conferenceId = $request->input('conference_id');
    $pastId = $request->input('past_id');
    $bannerId = $request->input('bannerid');
    $table = 'c_home_past_gallery';

    if (empty($bannerId)) {
        $insert = DB::table($table)->insert([
            'conference_id' => $conferenceId,
            'past_conference_id' => $pastId,
            'status' => 1,
            'page_type' => 1,
        ]);

        if ($insert) {
            // 🔥 Audit (created)
            $row = DB::table($table)
                ->where('conference_id', $conferenceId)
                ->where('past_conference_id', $pastId)
                ->where('page_type', 1)
                ->orderByDesc('id')
                ->first();

            if ($row) {
                $this->logAudit($table, $row->id, 'created', null, (array) $row);
            }
        }
    } else {
        // 🔹 OLD
        $old = DB::table($table)->where('id', $bannerId)->first();

        $insert = DB::table($table)
            ->where('id', $bannerId)
            ->update(['past_conference_id' => $pastId]);

        if ($insert) {
            // 🔹 NEW
            $new = DB::table($table)->where('id', $bannerId)->first();
            if ($old && $new) {
                $this->logAudit($table, $bannerId, 'updated', (array) $old, (array) $new);
            }
        }
    }

    return $insert
        ? response()->json(['success' => true, 'message' => 'Successfully Data Updated'])
        : response()->json(['success' => false, 'message' => 'Failed to Update data'], 500);
}

public function resetPastGallery(Request $request, $pastGalleryId)
{
    $conferenceId = $request->input('conference_id');

    // 🔹 OLD rows (may be multiple)
    $rows = DB::table('c_home_past_gallery')
        ->where([
            'conference_id' => $conferenceId,
            'past_conference_id' => $pastGalleryId,
            'page_type' => 1,
        ])
        ->get();

    $deleted = DB::table('c_home_past_gallery')
        ->where([
            'conference_id' => $conferenceId,
            'past_conference_id' => $pastGalleryId,
            'page_type' => 1,
        ])
        ->delete();

    if ($deleted) {
        foreach ($rows as $row) {
            $this->logAudit('c_home_past_gallery', $row->id, 'deleted', (array) $row, null);
        }
    }

    return $deleted
        ? response()->json(['success' => true, 'message' => 'Successfully reset'])
        : response()->json(['success' => false, 'message' => 'Failed to reset data'], 500);
}

public function pastAllGallery(Request $request)
{
    $conferenceId = $request->input('conference_id');

    // 🟩 Get folder name for uploads
    $folderName = \DB::table('conferences')
        ->where('id', $conferenceId)
        ->value('unique_id');

    // 🟩 Fetch gallery records
    $pastGallery = \DB::table('c_home_past_gallery')
        ->where('conference_id', $conferenceId)
        ->select('id', 'conference_id', 'past_conference_id', 'page_type')
        ->orderBy('id', 'desc')
        ->get();

    // 🟩 Add related conference info (short_name)
    $pastGallery->transform(function ($item) {
        $item->confInfo = \DB::table('conferences')
            ->where('id', $item->past_conference_id)
            ->select('id', 'short_name')
            ->get();
        return $item;
    });

    // 🟩 Return in DataTable-compatible format
    return response()->json(['data' => $pastGallery]);
}

public function pastAllGalleryOptions(Request $request)
{
    $conferenceId = $request->input('conference_id');

    $conferences = DB::table('conferences')
        ->where('status', 1)
        ->where('id', '!=', $conferenceId)
        ->orderBy('short_name', 'ASC')
        ->select('id', 'short_name')
        ->get();

    return response()->json(['conferences' => $conferences]);
}

public function savePastAllGallery(Request $request)
{
    $conferenceId = $request->input('conference_id');
    $pastId = $request->input('past_id');

    $insert = DB::table('c_home_past_gallery')->insert([
        'conference_id' => $conferenceId,
        'past_conference_id' => $pastId,
        'status' => 1,
        'page_type' => 2,
    ]);

    if ($insert) {
        // 🔥 Audit (created)
        $row = DB::table('c_home_past_gallery')
            ->where('conference_id', $conferenceId)
            ->where('past_conference_id', $pastId)
            ->where('page_type', 2)
            ->orderByDesc('id')
            ->first();

        if ($row) {
            $this->logAudit('c_home_past_gallery', $row->id, 'created', null, (array) $row);
        }
    }

    return $insert
        ? response()->json(['success' => true, 'message' => 'Successfully Data Updated'])
        : response()->json(['success' => false, 'message' => 'Failed to Update data'], 500);
}

public function deletePastAllGallery(Request $request, $id)
{
    // 🔹 OLD
    $old = DB::table('c_home_past_gallery')->where('id', $id)->first();

    $deleted = DB::table('c_home_past_gallery')->where('id', $id)->delete();

    if ($deleted && $old) {
        $this->logAudit('c_home_past_gallery', $id, 'deleted', (array) $old, null);
    }

    return $deleted
        ? response()->json(['success' => true, 'message' => 'Successfully Deleted'])
        : response()->json(['success' => false, 'message' => 'Failed to Delete data'], 500);
}

// --------------- Helpers ----------------
protected function resolveFolderForConference($conferenceId)
{
    $conf = DB::table('conferences')->where('id', $conferenceId)->first();
    if (!$conf) return null;
    // replicate CI style: folder_name_id or unique_id fallback
    $folderName = trim(($conf->folder_name ?? '') . '_' . ($conf->id ?? $conferenceId));
    return $folderName;
}

protected function ensureUploadPath($folderName, $table)
{
    $uploadBase = public_path("uploads/conferences/{$folderName}/{$table}");
    if (!File::exists($uploadBase)) {
        File::makeDirectory($uploadBase, 0777, true);
    }
    return $uploadBase;
}

protected function buildFilePath($folderName, $table, $filename)
{
    return "uploads/conferences/{$folderName}/{$table}/{$filename}";
}

    // --------------- List ----------------
  // ✅ List records (for any table)
 public function dynamicIndex(Request $request, $table = null)
{
    $table = $table ?? $request->get('table');
    if (!$table || !Schema::hasTable($table)) {
        return response()->json(['status' => false, 'message' => 'Invalid table'], 400);
    }

    $query = DB::table($table)->orderBy('id', 'desc');
    if ($request->has('conference_id')) {
        $query->where('conference_id', $request->conference_id);
    }

    return response()->json($query->get());
}

// ✅ Show single record
public function dynamicShow($table, $id)
{
    if (!Schema::hasTable($table)) {
        return response()->json(['status' => false, 'message' => 'Invalid table'], 400);
    }
    $record = DB::table($table)->find($id);
    return response()->json($record);
}

public function dynamicCreate($table)
{
    switch ($table) {
        case 'c_venue':
            return response()->json([
                'title' => 'Add Venue Details',
                'fields' => [
                    [
                        'label' => 'Venue Page',
                        'type' => 'radio',
                        'name' => 'venue_page',
                        'options' => [
                            ['label' => 'New', 'value' => 1],
                            ['label' => 'Old', 'value' => 0],
                        ],
                    ],
                    ['label' => 'Venue', 'type' => 'text', 'name' => 'venue_title'],

                    // ✅ WebP + dimension-aware image fields
                    [
                        'label' => 'Home Page Venue Image',
                        'type' => 'file',
                        'name' => 'venueImage',
                        'accept' => 'image/webp',
                        'note' => 'Required dimensions: 500 x 305px (WEBP)',
                        'dimensions' => ['width' => 500, 'height' => 305],
                    ],
                    [
                        'label' => 'Inner Page Image',
                        'type' => 'file',
                        'name' => 'venueImage2',
                        'accept' => 'image/webp',
                        'note' => 'Required dimensions: 540 x 356px (WEBP)',
                        'dimensions' => ['width' => 540, 'height' => 356],
                    ],
                    [
                        'label' => 'Floor Map',
                        'type' => 'file',
                        'name' => 'venueImage3',
                        'accept' => 'image/webp',
                        'note' => 'Required dimensions: 1000 x 356px (WEBP)',
                        'dimensions' => ['width' => 1000, 'height' => 356],
                    ],
                    [
                        'label' => 'Acceptance Letter',
                        'type' => 'file',
                        'name' => 'letter_doc',
                        'accept' => '.doc,.docx,.pdf',
                        'note' => 'Upload only DOC, DOCX, or PDF files (max 5MB)',
                    ],

                    // ✅ CKEditor textareas
                    ['label' => 'Venue Address', 'type' => 'textarea', 'name' => 'venue_address'],
                    ['label' => 'Venue Content', 'type' => 'textarea', 'name' => 'trans_content'],
                    ['label' => 'Phone Numbers', 'type' => 'textarea', 'name' => 'venue_phonenumbers'],
                    ['label' => 'Transportation Map', 'type' => 'textarea', 'name' => 'trans_map'],
                    ['label' => 'About City', 'type' => 'textarea', 'name' => 'about_city'],
                ],
            ]);

        // ✅ NEW: venue_page
        case 'venue_page':
            return response()->json([
                'title' => 'Add FAQ',
                'fields' => [
                    [
                        'label' => 'Venue Type',
                        'type' => 'select',
                        'name' => 'venue_type',
                        'required' => true,
                        'options' => [
                            ['label' => 'FAQ', 'value' => 'FAQ'],
                            // ✅ Add more later if needed (e.g., 'Contact', 'Accommodation')
                        ],
                        'note' => 'Currently only FAQ is supported',
                    ],
                    [
                        'label' => 'Title',
                        'type' => 'text',
                        'name' => 'title',
                        'required' => true,
                        'placeholder' => 'e.g., How do I register?',
                    ],
                    [
                        'label' => 'Description',
                        'type' => 'richtext', // or 'textarea' if CKEditor is auto-applied by frontend
                        'name' => 'testimonial_desc',
                        'required' => false,
                        'rows' => 6,
                        'placeholder' => 'Answer content (supports rich formatting)',
                    ],
                ],
            ]);

        default:
            return response()->json(['status' => false, 'message' => 'Invalid table']);
    }
}

public function dynamicStore(Request $request, $table)
{
    try {
        switch ($table) {
            case 'c_venue':
                $conferenceId = $request->input('conference_id');
                if (!$conferenceId) {
                    return response()->json(['status' => false, 'message' => 'conference_id is required'], 400);
                }

                $folderName = \DB::table('conferences')->where('id', $conferenceId)->value('unique_id');
                if (!$folderName) {
                    return response()->json(['status' => false, 'message' => 'Invalid conference ID'], 400);
                }

                $uploadPath = "/home/scisynopsisconfe/public_html/uploads/conferences/{$folderName}";
                if (!file_exists($uploadPath)) {
                    mkdir($uploadPath, 0755, true);
                }

                // ✅ Validate — keep frontend names (venueImage, etc.)
                $validated = $request->validate([
                    'venue_page' => 'required|in:0,1',
                    'venue_title' => 'required|string|max:255',
                    'venueImage' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
                    'venueImage2' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
                    'venueImage3' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
                    'letter_doc' => 'nullable|mimes:pdf,doc,docx|max:5120',
                    'venue_address' => 'nullable|string',
                    'trans_content' => 'nullable|string',
                    'venue_phonenumbers' => 'nullable|string',
                    'trans_map' => 'nullable|string',
                    'about_city' => 'nullable|string',
                    'check_in_time' => 'nullable|string',
                    'check_out_time' => 'nullable|string',
                ]);

                // ✅ Build data with YOUR DB column names
                $data = [
                    'conference_id' => $conferenceId,
                    'venue_title' => $validated['venue_title'],
                    'venue_page' => $validated['venue_page'],
                    'venue_address' => $validated['venue_address'] ?? null,
                    'trans_content' => $validated['trans_content'] ?? null,
                    'venue_phonenumbers' => $validated['venue_phonenumbers'] ?? null,
                    'trans_map' => $validated['trans_map'] ?? null,
                    'about_city' => $validated['about_city'] ?? null,
                    'check_in_time' => $validated['check_in_time'] ?? null,
                    'check_out_time' => $validated['check_out_time'] ?? null,
                    'created_date' => now()->toDateTimeString(),

                    // ✅ Map frontend → DB: venueImage → venue_img, etc.
                    'venue_img' => null,
                    'venue_img2' => null,
                    'venue_img3' => null,
                    'letter_doc' => null,
                ];

                // ✅ Handle file uploads
                $files = [
                    'venueImage'  => 'venue_img',
                    'venueImage2' => 'venue_img2',
                    'venueImage3' => 'venue_img3',
                    'letter_doc'  => 'letter_doc',
                ];

                foreach ($files as $inputName => $columnName) {
                    if ($request->hasFile($inputName)) {
                        $file = $request->file($inputName);
                        $filename = time() . '_' . $inputName . '.' . $file->getClientOriginalExtension();
                        $file->move($uploadPath, $filename);
                        $data[$columnName] = "uploads/conferences/{$folderName}/venue/{$filename}";
                    }
                }

                // 🔹 insert and audit
                $insertId = \DB::table('c_venue')->insertGetId($data);

                $new = \DB::table('c_venue')->find($insertId);
                if ($new) {
                    $this->logAudit('c_venue', $insertId, 'created', null, (array) $new);
                }

                return response()->json([
                    'status' => true,
                    'message' => 'Venue added successfully'
                ]);

            // ✅ NEW: venue_page
            case 'venue_page':
                $conferenceId = $request->input('conference_id');
                if (!$conferenceId) {
                    return response()->json(['status' => false, 'message' => 'conference_id is required'], 400);
                }

                // Optional: validate conference exists (security)
                $exists = DB::table('conferences')->where('id', $conferenceId)->exists();
                if (!$exists) {
                    return response()->json(['status' => false, 'message' => 'Invalid conference ID'], 400);
                }

                $validated = $request->validate([
                    'title' => 'required|string|max:255',
                    'venue_type' => 'required',
                    'testimonial_desc' => 'nullable|string',
                ]);

                $data = [
                    'conference_id' => $conferenceId,
                    'title' => $validated['title'],
                    'venue_type' => $validated['venue_type'],
                    'testimonial_desc' => $validated['testimonial_desc'] ?? null,
                ];

                $id = DB::table('venue_page')->insertGetId($data);

                $new = DB::table('venue_page')->find($id);
                if ($new) {
                    $this->logAudit('venue_page', $id, 'created', null, (array) $new);
                }

                return response()->json([
                    'status' => true,
                    'message' => 'Venue page added successfully',
                    'id' => $id,
                ]);

            default:
                return response()->json(['status' => false, 'message' => 'Invalid table'], 400);
        }

    } catch (\Illuminate\Validation\ValidationException $e) {
        return response()->json(['status' => false, 'errors' => $e->errors()], 422);
    } catch (\Exception $e) {
        \Log::error('DynamicStore Error: ' . $e->getMessage());
        return response()->json([
            'status' => false,
            'message' => config('app.debug') ? $e->getMessage() : 'Server error'
        ], 500);
    }
}


    // ✅ Update
 // ✅ Update with full file handling, validation, and DB column mapping (mirrors dynamicStore)
// ✅ Update with full file handling, validation, and DB column mapping
public function dynamicUpdate(Request $request, $table, $id)
{
    // ✅ Ensure record exists
    $existing = DB::table($table)->find($id);
    if (!$existing) {
        return response()->json(['status' => false, 'message' => 'Record not found'], 404);
    }

    try {
        switch ($table) {
            case 'c_venue':
                $conferenceId = $request->input('conference_id') ?? $existing->conference_id;
                if (!$conferenceId) {
                    return response()->json(['status' => false, 'message' => 'conference_id is required'], 400);
                }

                $folderName = DB::table('conferences')->where('id', $conferenceId)->value('unique_id');
                if (!$folderName) {
                    return response()->json(['status' => false, 'message' => 'Invalid conference ID'], 400);
                }

                // ✅ Base upload path (absolute on server)
                $baseUploadPath = "/home/scisynopsisconfe/public_html/uploads/conferences/{$folderName}";
                $relativeUploadDir = "uploads/conferences/{$folderName}"; // For DB & URLs

                // ✅ Ensure directory (and subfolder 'venue') exist
                $uploadPath = $baseUploadPath . '/venue';
                if (!file_exists($uploadPath)) {
                    mkdir($uploadPath, 0755, true);
                }

                // ✅ Validate — allow nullable; file rules auto-skip if not uploaded
                $validated = $request->validate([
                    'venue_page' => 'nullable|in:0,1',
                    'venue_title' => 'nullable|string|max:255',
                    'venueImage' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
                    'venueImage2' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
                    'venueImage3' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
                    'letter_doc' => 'nullable|mimes:pdf,doc,docx|max:5120',
                    'venue_address' => 'nullable|string',
                    'trans_content' => 'nullable|string',
                    'venue_phonenumbers' => 'nullable|string',
                    'trans_map' => 'nullable|string',
                    'about_city' => 'nullable|string',
                    'check_in_time' => 'nullable|string',
                    'check_out_time' => 'nullable|string',
                ]);

                // ✅ Build update data (retain existing if not provided)
                $data = [
                    'conference_id' => $conferenceId,
                    'venue_title' => $validated['venue_title'] ?? $existing->venue_title,
                    'venue_page' => $validated['venue_page'] ?? $existing->venue_page,
                    'venue_address' => $validated['venue_address'] ?? $existing->venue_address,
                    'trans_content' => $validated['trans_content'] ?? $existing->trans_content,
                    'venue_phonenumbers' => $validated['venue_phonenumbers'] ?? $existing->venue_phonenumbers,
                    'trans_map' => $validated['trans_map'] ?? $existing->trans_map,
                    'about_city' => $validated['about_city'] ?? $existing->about_city,
                    'check_in_time' => $validated['check_in_time'] ?? $existing->check_in_time,
                    'check_out_time' => $validated['check_out_time'] ?? $existing->check_out_time,
                ];

                // ✅ File handling: new upload → hidden fallback → preserve existing
                $fileMap = [
                    'venueImage'  => 'venue_img',
                    'venueImage2' => 'venue_img2',
                    'venueImage3' => 'venue_img3',
                    'letter_doc'  => 'letter_doc',
                ];

                foreach ($fileMap as $inputName => $dbColumn) {
                    if ($request->hasFile($inputName)) {
                        $file = $request->file($inputName);
                        if ($file->isValid()) {
                            // Generate safe filename
                            $ext = $file->getClientOriginalExtension();
                            $filename = time() . '_' . Str::random(6) . '.' . $ext;
                            $fullPath = $uploadPath . '/' . $filename;

                            // Move to server (absolute path)
                            $file->move($uploadPath, $filename);

                            // Store relative path in DB
                            $data[$dbColumn] = "{$relativeUploadDir}/venue/{$filename}";

                            // ✅ Delete old file if exists
                            $oldFilePath = $existing->{$dbColumn};
                            if ($oldFilePath && file_exists(public_path($oldFilePath))) {
                                @unlink(public_path($oldFilePath));
                            }
                        }
                    } else {
                        // Use hidden input (e.g., hiddenVenueImage) or keep existing value
                        $hiddenKey = 'hidden' . ucfirst($inputName); // e.g., hiddenVenueImage
                        $data[$dbColumn] = $request->input($hiddenKey, $existing->{$dbColumn});
                    }
                }

                // ✅ Perform update
                DB::table('c_venue')->where('id', $id)->update($data);

                // ✅ Fetch updated record for response (optional but helpful)
                $updatedRecord = DB::table('c_venue')->find($id);

                // 🔹 Audit log
                if ($updatedRecord) {
                    $this->logAudit('c_venue', $id, 'updated', (array) $existing, (array) $updatedRecord);
                }

                return response()->json([
                    'status' => true,
                    'message' => 'Venue updated successfully',
                    'data' => $updatedRecord,
                ]);
                break;

            // ✅ NEW: venue_page
            case 'venue_page':
                $validated = $request->validate([
                    'title' => 'nullable|string|max:255',
                    'venue_type' => 'nullable',
                    'testimonial_desc' => 'nullable|string',
                ]);

                $data = [
                    'title' => $validated['title'] ?? $existing->title,
                    'venue_type' => $validated['venue_type'] ?? $existing->venue_type,
                    'testimonial_desc' => $validated['testimonial_desc'] ?? $existing->testimonial_desc,
                ];

                DB::table('venue_page')->where('id', $id)->update($data);

                $updated = DB::table('venue_page')->find($id);

                // 🔹 Audit log
                if ($updated) {
                    $this->logAudit('venue_page', $id, 'updated', (array) $existing, (array) $updated);
                }

                return response()->json([
                    'status' => true,
                    'message' => 'Venue page updated successfully',
                    'data' => $updated,
                ]);

            default:
                return response()->json(['status' => false, 'message' => 'Update not implemented for this table'], 400);
        }

    } catch (\Illuminate\Validation\ValidationException $e) {
        return response()->json(['status' => false, 'errors' => $e->errors()], 422);
    } catch (\Exception $e) {
        \Log::error("DynamicUpdate Error [table={$table}, id={$id}]: " . $e->getMessage());
        return response()->json([
            'status' => false,
            'message' => config('app.debug') ? $e->getMessage() : 'Failed to update record'
        ], 500);
    }
}

// ✅ Delete
public function dynamicDelete($table, $id)
{
    if (!Schema::hasTable($table)) {
        return response()->json(['status' => false, 'message' => 'Invalid table'], 400);
    }

    // 🔹 Capture old row for audit (do NOT change behaviour if null)
    $existing = DB::table($table)->find($id);

    DB::table($table)->where('id', $id)->delete();

    if ($existing) {
        $this->logAudit($table, $id, 'deleted', (array) $existing, null);
    }

    return response()->json(['status' => true, 'message' => 'Deleted']);
}

// --------------- Store single file ----------------
public function dynamicStoreWithSingleFile(Request $req)
{
    $table = $req->input('table');
    $conferenceId = $req->input('conference_id');
    $fileField = $req->input('file_field','file'); // name of file input
    if (!$table || !Schema::hasTable($table)) return response()->json(['success'=>false,'message'=>'Invalid table'],400);

    $folderName = $this->resolveFolderForConference($conferenceId) ?? ($req->input('folderName') ?? 'default');
    $uploadBase = $this->ensureUploadPath($folderName, $table);

    $data = $req->except(['_token','table','folderName','file_field','multi_fields']);
    if ($req->hasFile($fileField)) {
        $f = $req->file($fileField);
        $name = time().'_'.uniqid().'_'.$f->getClientOriginalName();
        $f->move($uploadBase, $name);
        $data[$fileField] = $this->buildFilePath($folderName, $table, $name);
    }

    $data['created_at'] = now();
    $id = DB::table($table)->insertGetId($data);

    // 🔹 Audit log
    $new = DB::table($table)->find($id);
    if ($new) {
        $this->logAudit($table, $id, 'created', null, (array) $new);
    }

    return response()->json(['success'=>true,'id'=>$id,'message'=>'Saved']);
}

// --------------- Update single file ----------------
public function dynamicUpdateWithSingleFile(Request $req, $id)
{
    $table = $req->input('table');
    $conferenceId = $req->input('conference_id');
    $fileField = $req->input('file_field','file');
    if (!$table || !Schema::hasTable($table)) return response()->json(['success'=>false,'message'=>'Invalid table'],400);

    $folderName = $this->resolveFolderForConference($conferenceId) ?? ($req->input('folderName') ?? 'default');
    $uploadBase = $this->ensureUploadPath($folderName, $table);

    $existing = DB::table($table)->where('id',$id)->first();
    if (!$existing) {
        return response()->json(['success' => false, 'message' => 'Record not found'], 404);
    }

    $data = $req->except(['_token','table','folderName','file_field','multi_fields']);

    if ($req->hasFile($fileField)) {
        // delete old if exists and path is relative
        if (!empty($existing->$fileField)) {
            $maybePath = public_path($existing->$fileField);
            if (File::exists($maybePath)) File::delete($maybePath);
        }
        $f = $req->file($fileField);
        $name = time().'_'.uniqid().'_'.$f->getClientOriginalName();
        $f->move($uploadBase, $name);
        $data[$fileField] = $this->buildFilePath($folderName, $table, $name);
    }

    DB::table($table)->where('id',$id)->update($data);

    // 🔹 Audit log
    $updated = DB::table($table)->where('id', $id)->first();
    if ($updated) {
        $this->logAudit($table, $id, 'updated', (array) $existing, (array) $updated);
    }

    return response()->json(['success'=>true,'message'=>'Updated']);
}

    // --------------- Store multiple files ----------------
    // multi_fields param: JSON array of input-names (e.g. ["venueImage","gallery_files"])
public function dynamicStoreWithMultipleFiles(Request $req)
{
    $table = $req->input('table');
    $conferenceId = $req->input('conference_id');
    $multiFields = $req->input('multi_fields', []); // array or comma separated
    if (is_string($multiFields)) $multiFields = json_decode($multiFields, true) ?? explode(',', $multiFields);
    if (!$table || !Schema::hasTable($table)) return response()->json(['success'=>false,'message'=>'Invalid table'],400);

    $folderName = $this->resolveFolderForConference($conferenceId) ?? ($req->input('folderName') ?? 'default');
    $uploadBase = $this->ensureUploadPath($folderName, $table);

    $data = $req->except(['_token','table','folderName','file_field','multi_fields']);

    foreach ($multiFields as $field) {
        $field = trim($field);
        $saved = [];
        if ($req->hasFile($field)) {
            $files = $req->file($field);
            // if single file provided, normalize to array
            if (!is_array($files)) $files = [$files];
            foreach ($files as $f) {
                $name = time().'_'.uniqid().'_'.$f->getClientOriginalName();
                $f->move($uploadBase, $name);
                $saved[] = $this->buildFilePath($folderName, $table, $name);
            }
        }
        // if only one file expected, you might want to keep string; we store JSON array to be generic
        $data[$field] = json_encode($saved);
    }

    $data['created_at'] = now();
    $id = DB::table($table)->insertGetId($data);

    // 🔹 Audit log (created)
    $new = DB::table($table)->find($id);
    if ($new && method_exists($this, 'logAudit')) {
        $this->logAudit($table, (int)$id, 'created', null, (array)$new);
    }

    return response()->json(['success'=>true,'id'=>$id,'message'=>'Saved']);
}

// --------------- Update multiple files ----------------
public function dynamicUpdateWithMultipleFiles(Request $req, $id)
{
    $table = $req->input('table');
    $conferenceId = $req->input('conference_id');
    $multiFields = $req->input('multi_fields', []);
    if (is_string($multiFields)) $multiFields = json_decode($multiFields, true) ?? explode(',', $multiFields);
    if (!$table || !Schema::hasTable($table)) return response()->json(['success'=>false,'message'=>'Invalid table'],400);

    $folderName = $this->resolveFolderForConference($conferenceId) ?? ($req->input('folderName') ?? 'default');
    $uploadBase = $this->ensureUploadPath($folderName, $table);

    $existing = DB::table($table)->where('id',$id)->first();
    $data = $req->except(['_token','table','folderName','file_field','multi_fields']);

    foreach ($multiFields as $field) {
        $field = trim($field);
        $saved = [];
        if ($req->hasFile($field)) {
            $files = $req->file($field);
            if (!is_array($files)) $files = [$files];
            // Optionally delete old files: skip if you want to keep old ones
            // Delete existing if exists and it's JSON array
            if (!empty($existing->$field)) {
                $decoded = json_decode($existing->$field, true);
                if (is_array($decoded)) {
                    foreach ($decoded as $old) {
                        $path = public_path($old);
                        if (File::exists($path)) File::delete($path);
                    }
                } else {
                    $path = public_path($existing->$field);
                    if (File::exists($path)) File::delete($path);
                }
            }
            foreach ($files as $f) {
                $name = time().'_'.uniqid().'_'.$f->getClientOriginalName();
                $f->move($uploadBase, $name);
                $saved[] = $this->buildFilePath($folderName, $table, $name);
            }
        } else {
            // keep existing value when no new file uploaded
            $saved = $existing->$field ?? [];
            if (is_string($saved) && @json_decode($saved) !== null) $saved = json_decode($saved, true);
        }
        $data[$field] = json_encode($saved);
    }

    DB::table($table)->where('id',$id)->update($data);

    // 🔹 Audit log (updated)
    $updated = DB::table($table)->where('id',$id)->first();
    if ($updated && method_exists($this, 'logAudit')) {
        $this->logAudit($table, (int)$id, 'updated', (array)$existing, (array)$updated);
    }

    return response()->json(['success'=>true,'message'=>'Updated']);
}



public function getPastEditions(Request $request)
{
    // --- Auth same as conference-info ---
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    $conferenceId = $request->query('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    $conf = \DB::table('conferences')->where('id', $conferenceId)->first();
    if (!$conf) {
        return response()->json(['message' => 'Conference not found'], 404);
    }

    $folder = $conf->folder_name ?? '';

    // absolute URL like CI3: /uploads/{folderName}/{past_editions_logo}
    $logoUrl = ($conf->past_editions_logo && $folder)
        ? "https://scisynopsisconferences.com/uploads/{$folder}/{$conf->past_editions_logo}"
        : null;

    return response()->json([
        'id'                  => $conf->id,
        'folder_name'         => $folder,
        'past_editions_logo'  => $logoUrl,
        'raw_past_editions_logo' => $conf->past_editions_logo, // optional, if you ever need raw name
    ]);
}

public function savePastEditions(Request $request)
{
    // --- Auth same as conference-info ---
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    $conferenceId = $request->input('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }


    if ($request->hasFile('mainImage')) {
        $file = $request->file('mainImage');

        \Log::info('UPLOAD DEBUG', [
            'original_name' => $file->getClientOriginalName(),
            'extension'     => $file->getClientOriginalExtension(),
            'mime_type'     => $file->getMimeType(),            // server-detected MIME
            'client_mime'   => $file->getClientMimeType(),      // browser-reported MIME
            'size'          => $file->getSize(),
            'is_valid'      => $file->isValid()
        ]);
    }
    $conf = \DB::table('conferences')->where('id', $conferenceId)->first();
    if (!$conf) {
        return response()->json(['message' => 'Conference not found'], 404);
    }

    // Validation: WEBP only, and dimension check can be done client-side
    $request->validate([
        'mainImage' => 'nullable|file|mimes:webp|mimetypes:image/webp,image/x-webp',
        'del_img'   => 'nullable|boolean',
    ]);

    $folderName = $conf->folder_name ?: '';
    if (!$folderName) {
        // Fallback: you can adjust this if folder_name is guaranteed
        $folderName = 'default';
    }

    $uploadsPath = "/home/scisynopsisconfe/public_html/uploads/{$folderName}";
    if (!file_exists($uploadsPath)) {
        @mkdir($uploadsPath, 0777, true);
    }

    $params = [];

    // File upload (similar logic to CI3 but simplified)
    if ($request->hasFile('mainImage')) {
        $file = $request->file('mainImage');
        $filename = time() . '.' . $file->getClientOriginalExtension();
        $file->move($uploadsPath, $filename);
        $params['past_editions_logo'] = $filename;
    }

    // Delete flag (same behaviour as CI3's del_img = 1)
    if ($request->boolean('del_img')) {
        $params['past_editions_logo'] = '';
    }

    if (!empty($params)) {
        // 🔹 OLD
        $old = \DB::table('conferences')->where('id', $conferenceId)->first();

        \DB::table('conferences')->where('id', $conferenceId)->update($params);

        // 🔹 NEW
        $new = \DB::table('conferences')->where('id', $conferenceId)->first();
        if ($old && $new) {
            $this->logAudit('conferences', $conferenceId, 'updated', (array) $old, (array) $new);
        }
    }

    return response()->json([
        'message'        => 'Past editions updated successfully',
        'conference_id'  => $conferenceId,
    ]);
}

public function getContactInfo(Request $request)
{
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    $conferenceId = $request->query('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    $row = DB::table('c_address')
        ->where('conference_id', $conferenceId)
        ->orderBy('id', 'asc')
        ->first();

    if (!$row) {
        // no data yet – return defaults for Vue
        return response()->json([
            'id'              => null,
            'conference_id'   => (int) $conferenceId,
            'contact_phone'   => '',
            'contact_email'   => '',
            'mail_heading_2'  => '',
            'secondary_mail'  => '',
            'mail_heading_3'  => '',
            'sponsors_email'  => '',
            'whatsapp_phone'  => '',
            'tollfree_number' => '',
        ]);
    }

    return response()->json([
// ... no audit for pure GET
        'id'              => $row->id,
        'conference_id'   => $row->conference_id,
        'contact_phone'   => $row->primary_phone,
        'contact_email'   => $row->primary_mail,
        'mail_heading_2'  => $row->mail_heading_2,
        'secondary_mail'  => $row->secondary_mail,
        'mail_heading_3'  => $row->mail_heading_3,
        'sponsors_email'  => $row->sponsors_email,
        'whatsapp_phone'  => $row->whatsapp_number,
        'tollfree_number' => $row->fax_number,
    ]);
}

public function saveContactInfo(Request $request)
{
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    $conferenceId = $request->input('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    // Basic validation (extend if you like)
    $request->validate([
        'contact_phone'   => 'nullable|string|max:255',
        'whatsapp_phone'  => 'nullable|string|max:255',
        'tollfree_number' => 'nullable|string|max:255',
        'contact_email'   => 'nullable|email|max:255',
        'secondary_mail'  => 'nullable|email|max:255',
        'sponsors_email'  => 'nullable|email|max:255',
        'mail_heading_2'  => 'nullable|string|max:255',
        'mail_heading_3'  => 'nullable|string|max:255',
    ]);

    $params = [
        'conference_id'   => $conferenceId,
        'primary_phone'   => $request->input('contact_phone'),
        'whatsapp_number' => $request->input('whatsapp_phone'),
        'fax_number'      => $request->input('tollfree_number'),
        'primary_mail'    => $request->input('contact_email'),
        'mail_heading_2'  => $request->input('mail_heading_2'),
        'secondary_mail'  => $request->input('secondary_mail'),
        'mail_heading_3'  => $request->input('mail_heading_3'),
        'sponsors_email'  => $request->input('sponsors_email'),
        'status'          => 1,
        'created_date'    => now(), // same as CI: always set
    ];

    $id = $request->input('bannersid');

    if ($id) {
        $exists = DB::table('c_address')
            ->where('conference_id', $conferenceId)
            ->where('id', $id)
            ->exists();

        if ($exists) {
            // 🔹 OLD
            $old = DB::table('c_address')->where('id', $id)->first();

            DB::table('c_address')
                ->where('id', $id)
                ->update($params);

            $recordId = $id;

            // 🔹 NEW
            $new = DB::table('c_address')->where('id', $recordId)->first();
            if ($old && $new) {
                $this->logAudit('c_address', $recordId, 'updated', (array) $old, (array) $new);
            }
        } else {
            $recordId = DB::table('c_address')->insertGetId($params);

            // 🔥 created
            $row = DB::table('c_address')->where('id', $recordId)->first();
            if ($row) {
                $this->logAudit('c_address', $recordId, 'created', null, (array) $row);
            }
        }
    } else {
        $existing = DB::table('c_address')
            ->where('conference_id', $conferenceId)
            ->first();

        if ($existing) {
            // 🔹 OLD
            $old = $existing;

            DB::table('c_address')
                ->where('id', $existing->id)
                ->update($params);

            $recordId = $existing->id;

            // 🔹 NEW
            $new = DB::table('c_address')->where('id', $recordId)->first();
            if ($old && $new) {
                $this->logAudit('c_address', $recordId, 'updated', (array) $old, (array) $new);
            }
        } else {
            $recordId = DB::table('c_address')->insertGetId($params);

            // 🔥 created
            $row = DB::table('c_address')->where('id', $recordId)->first();
            if ($row) {
                $this->logAudit('c_address', $recordId, 'created', null, (array) $row);
            }
        }
    }

    return response()->json([
        'message'       => 'Contact info saved successfully',
        'id'            => $recordId,
        'conference_id' => $conferenceId,
    ]);
}

public function deleteContactInfo(Request $request, $id)
{
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    $conferenceId = $request->query('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    // 🔹 OLD
    $old = DB::table('c_address')
        ->where('conference_id', $conferenceId)
        ->where('id', $id)
        ->first();

    $deleted = DB::table('c_address')
        ->where('conference_id', $conferenceId)
        ->where('id', $id)
        ->delete();

    if (!$deleted) {
        return response()->json(['message' => 'Contact info not found'], 404);
    }

    if ($old) {
        $this->logAudit('c_address', $id, 'deleted', (array) $old, null);
    }

    return response()->json([
        'message'       => 'Contact info deleted successfully',
        'id'            => (int) $id,
        'conference_id' => (int) $conferenceId,
    ]);
}

public function getSocialLinks(Request $request)
{
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$user) {
      return response()->json(['message' => 'Unauthorized'], 401);
    }

    $conferenceId = $request->query('conference_id');
    if (!$conferenceId) {
      return response()->json(['message' => 'Conference ID missing'], 422);
    }

    // In CI you used getAllInfo("c_social") but the view only used $info[0].
    // Here we treat it as 1 row per conference.
    $row = DB::table('c_social')
        ->where('conference_id', $conferenceId)
        ->orderBy('id', 'asc')
        ->first();

    if (!$row) {
        return response()->json([
            'id'            => null,
            'conference_id' => (int) $conferenceId,
            'content_title' => '', // Facebook
            'sub_title'     => '', // Twitter
            'short_desc'    => '', // Linkedin
            'long_desc'     => '', // Youtube
            'cms_logo'      => '',
            'cms_img'       => '', // Instagram
        ]);
    }

    return response()->json([
        'id'            => $row->id,
        'conference_id' => $row->conference_id,
        'content_title' => $row->content_title,
        'sub_title'     => $row->sub_title,
        'short_desc'    => $row->short_desc,
        'long_desc'     => $row->long_desc,
        'cms_logo'      => $row->cms_logo,
        'cms_img'       => $row->cms_img,
    ]);
}

public function saveSocialLinks(Request $request)
{
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    $conferenceId = $request->input('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    $request->validate([
        'content_title' => 'nullable|string|max:255',
        'sub_title'     => 'nullable|string|max:255',
        'short_desc'    => 'nullable|string|max:255',
        'long_desc'     => 'nullable|string|max:255',
        'cms_logo'      => 'nullable|string|max:255',
        'cms_img'       => 'nullable|string|max:255',
    ]);

    $params = [
        'conference_id' => $conferenceId,
        'content_title' => $request->input('content_title'),
        'sub_title'     => $request->input('sub_title'),
        'short_desc'    => $request->input('short_desc'),
        'long_desc'     => $request->input('long_desc'),
        'cms_logo'      => $request->input('cms_logo'),
        'cms_img'       => $request->input('cms_img'),
        'created_date'  => now(),
    ];

    $id = $request->input('bannersid');

    if ($id) {
        $exists = DB::table('c_social')
            ->where('conference_id', $conferenceId)
            ->where('id', $id)
            ->exists();

        if ($exists) {
            // 🔹 OLD
            $old = DB::table('c_social')->where('id', $id)->first();

            DB::table('c_social')
                ->where('id', $id)
                ->update($params);

            $recordId = $id;

            // 🔹 NEW
            $new = DB::table('c_social')->where('id', $recordId)->first();
            if ($old && $new) {
                $this->logAudit('c_social', $recordId, 'updated', (array) $old, (array) $new);
            }
        } else {
            $recordId = DB::table('c_social')->insertGetId($params);

            // 🔥 created
            $row = DB::table('c_social')->where('id', $recordId)->first();
            if ($row) {
                $this->logAudit('c_social', $recordId, 'created', null, (array) $row);
            }
        }
    } else {
        // Upsert by conference_id (1 row per conference)
        $existing = DB::table('c_social')
            ->where('conference_id', $conferenceId)
            ->first();

        if ($existing) {
            // 🔹 OLD
            $old = $existing;

            DB::table('c_social')
                ->where('id', $existing->id)
                ->update($params);

            $recordId = $existing->id;

            // 🔹 NEW
            $new = DB::table('c_social')->where('id', $recordId)->first();
            if ($old && $new) {
                $this->logAudit('c_social', $recordId, 'updated', (array) $old, (array) $new);
            }
        } else {
            $recordId = DB::table('c_social')->insertGetId($params);

            // 🔥 created
            $row = DB::table('c_social')->where('id', $recordId)->first();
            if ($row) {
                $this->logAudit('c_social', $recordId, 'created', null, (array) $row);
            }
        }
    }

    return response()->json([
        'message'       => 'Social links saved successfully',
        'id'            => $recordId,
        'conference_id' => $conferenceId,
    ]);
}

public function deleteSocialLinks(Request $request, $id)
{
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    $conferenceId = $request->query('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    // 🔹 OLD
    $old = DB::table('c_social')
        ->where('conference_id', $conferenceId)
        ->where('id', $id)
        ->first();

    $deleted = DB::table('c_social')
        ->where('conference_id', $conferenceId)
        ->where('id', $id)
        ->delete();

    if (!$deleted) {
        return response()->json(['message' => 'Social links not found'], 404);
    }

    if ($old) {
        $this->logAudit('c_social', $id, 'deleted', (array) $old, null);
    }

    return response()->json([
        'message'       => 'Social links deleted successfully',
        'id'            => (int) $id,
        'conference_id' => (int) $conferenceId,
    ]);
}

public function getTawkScript(Request $request)
{
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    $conferenceId = $request->query('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    // In CI you did getAllInfo("c_tawk_scpt"), but used only $banners[0]
    $row = DB::table('c_tawk_scpt')
        ->where('conference_id', $conferenceId)
        ->orderBy('id', 'asc')
        ->first();

    if (!$row) {
        return response()->json([
            'id'            => null,
            'conference_id' => (int) $conferenceId,
            'tawk_embed'    => '',
        ]);
    }

    return response()->json([
        'id'            => $row->id,
        'conference_id' => $row->conference_id,
        'tawk_embed'    => $row->tawk_embed,
    ]);
}

public function saveTawkScript(Request $request)
{
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    $conferenceId = $request->input('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    // you can relax this validation if you paste raw <script> tags etc.
    $request->validate([
        'tawk_embed' => 'nullable|string',
    ]);

    $params = [
        'conference_id' => $conferenceId,
        'tawk_embed'    => $request->input('tawk_embed'),
        'created_date'  => now(),
    ];

    $id = $request->input('bannersid');

    if ($id) {
        $exists = DB::table('c_tawk_scpt')
            ->where('conference_id', $conferenceId)
            ->where('id', $id)
            ->exists();

        if ($exists) {
            // 🔹 OLD
            $old = DB::table('c_tawk_scpt')->where('id', $id)->first();

            DB::table('c_tawk_scpt')
                ->where('id', $id)
                ->update($params);

            $recordId = $id;

            // 🔹 NEW
            $new = DB::table('c_tawk_scpt')->where('id', $recordId)->first();
            if ($old && $new) {
                $this->logAudit('c_tawk_scpt', $recordId, 'updated', (array) $old, (array) $new);
            }
        } else {
            $recordId = DB::table('c_tawk_scpt')->insertGetId($params);

            // 🔥 created
            $row = DB::table('c_tawk_scpt')->where('id', $recordId)->first();
            if ($row) {
                $this->logAudit('c_tawk_scpt', $recordId, 'created', null, (array) $row);
            }
        }
    } else {
        // upsert one row per conference
        $existing = DB::table('c_tawk_scpt')
            ->where('conference_id', $conferenceId)
            ->first();

        if ($existing) {
            // 🔹 OLD
            $old = $existing;

            DB::table('c_tawk_scpt')
                ->where('id', $existing->id)
                ->update($params);

            $recordId = $existing->id;

            // 🔹 NEW
            $new = DB::table('c_tawk_scpt')->where('id', $recordId)->first();
            if ($old && $new) {
                $this->logAudit('c_tawk_scpt', $recordId, 'updated', (array) $old, (array) $new);
            }
        } else {
            $recordId = DB::table('c_tawk_scpt')->insertGetId($params);

            // 🔥 created
            $row = DB::table('c_tawk_scpt')->where('id', $recordId)->first();
            if ($row) {
                $this->logAudit('c_tawk_scpt', $recordId, 'created', null, (array) $row);
            }
        }
    }

    return response()->json([
        'message'       => 'Live chat script saved successfully',
        'id'            => $recordId,
        'conference_id' => $conferenceId,
    ]);
}

public function deleteTawkScript(Request $request)
{
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    $conferenceId = $request->query('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    // Mirror CI's deleteContentByTable but scoped to this conference
    // 🔹 OLD rows
    $rows = DB::table('c_tawk_scpt')
        ->where('conference_id', $conferenceId)
        ->get();

    $deleted = DB::table('c_tawk_scpt')
        ->where('conference_id', $conferenceId)
        ->delete();

    if (!$deleted) {
        return response()->json(['message' => 'Live chat script not found'], 404);
    }

    foreach ($rows as $row) {
        $this->logAudit('c_tawk_scpt', $row->id, 'deleted', (array) $row, null);
    }

    return response()->json([
        'message'       => 'Live chat script deleted successfully',
        'conference_id' => (int) $conferenceId,
    ]);
}

public function getHomeScript(Request $request)
{
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    $conferenceId = $request->query('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    // CI used getAllInfo, but view only cares about [0]; treat as 1 row per conference
    $row = DB::table('c_homepage_script')
        ->where('conference_id', $conferenceId)
        ->orderBy('id', 'asc')
        ->first();

    if (!$row) {
        return response()->json([
            'id'            => null,
            'conference_id' => (int) $conferenceId,
            'home_script'   => '',
        ]);
    }

    return response()->json([
        'id'            => $row->id,
        'conference_id' => $row->conference_id,
        'home_script'   => $row->home_script,
    ]);
}

public function saveHomeScript(Request $request)
{
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    $conferenceId = $request->input('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    $request->validate([
        'home_script' => 'nullable|string',
    ]);

    $params = [
        'conference_id' => $conferenceId,
        'home_script'   => $request->input('home_script'),
        'created_date'  => now(),
    ];

    $id = $request->input('bannersid');

    if ($id) {
        $exists = DB::table('c_homepage_script')
            ->where('conference_id', $conferenceId)
            ->where('id', $id)
            ->exists();

        if ($exists) {
            // 🔹 OLD
            $old = DB::table('c_homepage_script')->where('id', $id)->first();

            DB::table('c_homepage_script')
                ->where('id', $id)
                ->update($params);

            $recordId = $id;

            // 🔹 NEW
            $new = DB::table('c_homepage_script')->where('id', $recordId)->first();
            if ($old && $new) {
                $this->logAudit('c_homepage_script', $recordId, 'updated', (array) $old, (array) $new);
            }
        } else {
            $recordId = DB::table('c_homepage_script')->insertGetId($params);

            // 🔥 created
            $row = DB::table('c_homepage_script')->where('id', $recordId)->first();
            if ($row) {
                $this->logAudit('c_homepage_script', $recordId, 'created', null, (array) $row);
            }
        }
    } else {
        // upsert one row per conference
        $existing = DB::table('c_homepage_script')
            ->where('conference_id', $conferenceId)
            ->first();

        if ($existing) {
            // 🔹 OLD
            $old = $existing;

            DB::table('c_homepage_script')
                ->where('id', $existing->id)
                ->update($params);

            $recordId = $existing->id;

            // 🔹 NEW
            $new = DB::table('c_homepage_script')->where('id', $recordId)->first();
            if ($old && $new) {
                $this->logAudit('c_homepage_script', $recordId, 'updated', (array) $old, (array) $new);
            }
        } else {
            $recordId = DB::table('c_homepage_script')->insertGetId($params);

            // 🔥 created
            $row = DB::table('c_homepage_script')->where('id', $recordId)->first();
            if ($row) {
                $this->logAudit('c_homepage_script', $recordId, 'created', null, (array) $row);
            }
        }
    }

    return response()->json([
        'message'       => 'Google tracking code saved successfully',
        'id'            => $recordId,
        'conference_id' => $conferenceId,
    ]);
}

public function deleteHomeScript(Request $request)
{
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    $conferenceId = $request->query('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    // Mirror CI deleteContentByTable, but scoped to this conference
    // 🔹 OLD rows
    $rows = DB::table('c_homepage_script')
        ->where('conference_id', $conferenceId)
        ->get();

    $deleted = DB::table('c_homepage_script')
        ->where('conference_id', $conferenceId)
        ->delete();

    if (!$deleted) {
        return response()->json(['message' => 'Google tracking code not found'], 404);
    }

    foreach ($rows as $row) {
        $this->logAudit('c_homepage_script', $row->id, 'deleted', (array) $row, null);
    }

    return response()->json([
        'message'       => 'Google tracking code deleted successfully',
        'conference_id' => (int) $conferenceId,
    ]);
}

public function getMessengerScript(Request $request)
{
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    $conferenceId = $request->query('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    $row = DB::table('messenger_script')
        ->where('conference_id', $conferenceId)
        ->orderBy('id', 'asc')
        ->first();

    if (!$row) {
        return response()->json([
            'id'               => null,
            'conference_id'    => (int) $conferenceId,
            'messenger_script' => '',
        ]);
    }

    return response()->json([
        'id'               => $row->id,
        'conference_id'    => $row->conference_id,
        'messenger_script' => $row->messenger_script,
    ]);
}

public function saveMessengerScript(Request $request)
{
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    $conferenceId = $request->input('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    $request->validate([
        'chat_script' => 'nullable|string',
    ]);

    $params = [
        'conference_id'    => $conferenceId,
        'messenger_script' => $request->input('chat_script'),
        'status'           => 1,
        'created_date'     => now(),
    ];

    $id = $request->input('bannersid');

    if ($id) {
        $exists = DB::table('messenger_script')
            ->where('conference_id', $conferenceId)
            ->where('id', $id)
            ->exists();

        if ($exists) {
            // 🔹 OLD
            $old = DB::table('messenger_script')->where('id', $id)->first();

            DB::table('messenger_script')
                ->where('id', $id)
                ->update($params);

            $recordId = $id;

            // 🔹 NEW
            $new = DB::table('messenger_script')->where('id', $recordId)->first();
            if ($old && $new) {
                $this->logAudit('messenger_script', $recordId, 'updated', (array) $old, (array) $new);
            }
        } else {
            $recordId = DB::table('messenger_script')->insertGetId($params);

            // 🔥 created
            $row = DB::table('messenger_script')->where('id', $recordId)->first();
            if ($row) {
                $this->logAudit('messenger_script', $recordId, 'created', null, (array) $row);
            }
        }
    } else {
        $existing = DB::table('messenger_script')
            ->where('conference_id', $conferenceId)
            ->first();

        if ($existing) {
            // 🔹 OLD
            $old = $existing;

            DB::table('messenger_script')
                ->where('id', $existing->id)
                ->update($params);

            $recordId = $existing->id;

            // 🔹 NEW
            $new = DB::table('messenger_script')->where('id', $recordId)->first();
            if ($old && $new) {
                $this->logAudit('messenger_script', $recordId, 'updated', (array) $old, (array) $new);
            }
        } else {
            $recordId = DB::table('messenger_script')->insertGetId($params);

            // 🔥 created
            $row = DB::table('messenger_script')->where('id', $recordId)->first();
            if ($row) {
                $this->logAudit('messenger_script', $recordId, 'created', null, (array) $row);
            }
        }
    }

    return response()->json([
        'message'       => 'Successfully saved data',
        'id'            => $recordId,
        'conference_id' => $conferenceId,
    ]);
}

public function deleteMessengerScript(Request $request, $id)
{
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    $conferenceId = $request->query('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    // 🔹 OLD
    $old = DB::table('messenger_script')
        ->where('conference_id', $conferenceId)
        ->where('id', $id)
        ->first();

    $deleted = DB::table('messenger_script')
        ->where('conference_id', $conferenceId)
        ->where('id', $id)
        ->delete();

    if (!$deleted) {
        return response()->json(['message' => 'Failed to delete data'], 404);
    }

    if ($old) {
        $this->logAudit('messenger_script', $id, 'deleted', (array) $old, null);
    }

    return response()->json([
        'message'       => 'Successfully deleted',
        'id'            => (int) $id,
        'conference_id' => (int) $conferenceId,
    ]);
}

public function userInvoices(Request $request)
{
    // 🔐 1. Authenticate via token (same pattern as logo())
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $adminUser = User::where('api_token', $token)->first();
    if (!$adminUser) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    // 📌 2. Get conference_id from request (NO session)
    $conferenceId = $request->input('conference_id') ?? $request->query('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    $conference = DB::table('conferences')->where('id', $conferenceId)->first();
    if (!$conference) {
        return response()->json(['message' => 'Conference not found'], 404);
    }

    // 👤 3. Get user_id from request
    $userId = $request->input('user_id') ?? $request->query('user_id');
    if (!$userId) {
        return response()->json(['message' => 'User ID missing'], 422);
    }

    // 4. Load user info
    $userInfo = DB::table('c_reg_users')
        ->where('id', $userId)
        ->get();

    if ($userInfo->isEmpty()) {
        return response()->json([
            'status'  => false,
            'message' => 'User not found',
        ], 404);
    }

    $user = $userInfo->first();

    // 5. Original CI3 queries replicated

    // Invoice info
    $invoiceInfo = DB::table('c_invoices')
        ->where('user_id', $userId)
        ->get();

    // Payment info
    $paymentInfo = DB::table('c_user_orders')
        ->where('user_id', $userId)
        ->get();

    // Reg info (group vs normal based on user_type)
    if ((int) $user->user_type === 8) {
        $regInfo = DB::table('c_user_group_registrations_dates')
            ->where('user_id', $userId)
            ->get();
    } else {
        $regInfo = DB::table('c_user_registrations_dates')
            ->where('user_id', $userId)
            ->get();
    }

    // ACP / ESPN registrations
    $regInfoacp = DB::table('c_user_registrations_acp')
        ->where('user_id', $userId)
        ->get();

    $regInfoespn = DB::table('c_user_registrations_espn')
        ->where('user_id', $userId)
        ->get();

    // Bank info (active)
    $bankinfo = DB::table('bank_details')
        ->where('status', 1)
        ->get();

    // Addon amounts
    $addonInfo = DB::table('c_user_addon_amounts')
        ->where('user_id', $userId)
        ->get();

    // Configurations (status=1, DESC)
    $configurations = DB::table('configurations')
        ->where('status', 1)
        ->orderByDesc('id')
        ->get();

    // Address from web_cms (page_type = 10)
    $address = DB::table('web_cms')
        ->where('page_type', 10)
        ->orderByDesc('id')
        ->get();

    // company_info filtered by user currency (same as CI3)
    $bankDetails = DB::table('company_info')
        ->where('currency', $user->currency)
        ->orderByDesc('id')
        ->get();

    // Conference info we already loaded above; wrap it to match old structure
    $confenceInfo = DB::table('conferences')
        ->where('id', $conferenceId)
        ->orderByDesc('id')
        ->get();

    // 6. Return JSON (no double-encoding)
    return response()->json([
        'status'         => true,
        'userInfo'       => $userInfo,
        'invoiceInfo'    => $invoiceInfo,
        'paymentInfo'    => $paymentInfo,
        'regInfo'        => $regInfo,
        'regInfoacp'     => $regInfoacp,
        'regInfoespn'    => $regInfoespn,
        'addonInfo'      => $addonInfo,
        'configurations' => $configurations,
        'bankDetails'    => $bankDetails,
        'confenceInfo'   => $confenceInfo,
        'address'        => $address,
        'bankinfo'       => $bankinfo,
        'userId'         => $userId,
        'user_type'      => (int) $user->user_type,
        'conference'     => $conference, // optional extra, since we already fetched it
    ]);
}

public function storeUserInvoice(Request $request)
{
   $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    // ✅ validate
    $validated = $request->validate([
        'conference_id'       => 'required|integer',
        'user_id'             => 'required|integer',
        'invoice_to'          => 'required|string|max:255',
        'invoice_date'        => 'required|date',
        'description'         => 'required|string',
        'registration_type'   => 'nullable|string|max:255',
        'reg_description'     => 'nullable|string|max:255',
        'currency'            => 'nullable|string|max:10',
        'amount'              => 'nullable|numeric',
        'manual_due_date'     => 'nullable|date',
        'extra_info'          => 'nullable|string',
        'total_extra_amount'  => 'nullable|numeric',
    ]);

    $conferenceId   = $validated['conference_id'];
    $userId         = $validated['user_id'];
    $invoiceDate    = date('Y-m-d', strtotime($validated['invoice_date']));
    $manualDueDate  = !empty($validated['manual_due_date'])
        ? date('Y-m-d', strtotime($validated['manual_due_date']))
        : null;

    // same pattern as CI3: Ymd + userId
    $invoiceIdValue = now()->format('Ymd') . $userId;

    $params = [
        'conference_id'       => $conferenceId,
        'user_id'             => $userId,
        'invoice_id'          => $invoiceIdValue,
        'invoice_to'          => $validated['invoice_to'],
        'invoice_date'        => $invoiceDate,
        'description'         => $validated['description'],
        'registration_type'   => $validated['registration_type'] ?? null,
        'reg_description'     => $validated['reg_description'] ?? null,
        'currency'            => $validated['currency'] ?? null,
        'amount'              => $validated['amount'] ?? null,
        'manual_due_date'     => $manualDueDate,
        'extra_info'          => $validated['extra_info'] ?? null,
        'total_extra_amount'  => $validated['total_extra_amount'] ?? null,
        'status'              => 1,
        'created_date'        => now(),   // matches old CI3 created_date
    ];

    $id = DB::table('c_invoices')->insertGetId($params);

    // 🔁 update c_reg_users.manual_due_date (your old code tried this with $manual_duedate)
    if ($manualDueDate) {
        DB::table('c_reg_users')
            ->where('id', $userId)
            ->update(['manual_due_date' => $manualDueDate]);
    }

    return response()->json([
        'status'      => true,
        'message'     => 'Invoice created successfully',
        'invoice_db_id' => $id,
        'invoice_id'  => $invoiceIdValue,
    ]);
}
public function updateUserInvoice(Request $request, $id)
{
 $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }
    // validate
    $validated = $request->validate([
        'user_id'             => 'required|integer',
        'invoice_to'          => 'required|string|max:255',
        'invoice_date'        => 'required|date',
        'description'         => 'required|string',
        'registration_type'   => 'nullable|string|max:255',
        'reg_description'     => 'nullable|string|max:255',
        'currency'            => 'nullable|string|max:10',
        'amount'              => 'nullable|numeric',
        'manual_due_date'     => 'nullable|date',
        'extra_info'          => 'nullable|string',
        'total_extra_amount'  => 'nullable|numeric',
    ]);

    $userId        = $validated['user_id'];
    $invoiceDate   = date('Y-m-d', strtotime($validated['invoice_date']));
    $manualDueDate = !empty($validated['manual_due_date'])
        ? date('Y-m-d', strtotime($validated['manual_due_date']))
        : null;

    $params = [
        'invoice_to'          => $validated['invoice_to'],
        'invoice_date'        => $invoiceDate,
        'description'         => $validated['description'],
        'registration_type'   => $validated['registration_type'] ?? null,
        'reg_description'     => $validated['reg_description'] ?? null,
        'currency'            => $validated['currency'] ?? null,
        'amount'              => $validated['amount'] ?? null,
        'manual_due_date'     => $manualDueDate,
        'extra_info'          => $validated['extra_info'] ?? null,
        'total_extra_amount'  => $validated['total_extra_amount'] ?? null,
        'created_date'        => now(), // you were overwriting created_date; kept same
    ];

    $updated = DB::table('c_invoices')->where('id', $id)->update($params);

    if ($manualDueDate) {
        DB::table('c_reg_users')
            ->where('id', $userId)
            ->update(['manual_due_date' => $manualDueDate]);
    }

    return response()->json([
        'status'  => (bool) $updated,
        'message' => $updated ? 'Invoice updated successfully' : 'No changes',
    ]);
}

public function downloadUserInvoice(Request $request, $userId)
{
    // 🔑 Token resolution — identical to your logo() method
    $token = $request->bearerToken()
        ?? $request->header('Authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    if (!$token) {
        return response()->json(['message' => 'Token missing'], 400);
    }

    // 👤 Authenticate via c_reg_users (matches your CI table)
    $user = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    // 🎯 Optional: Enforce access control (e.g., admin or self)
    // if (!$user->is_admin && $user->id != $userId) {
    //     return response()->json(['message' => 'Forbidden'], 403);
    // }

    // 🔢 Ensure $userId is int
    $userId = (int) $userId;

    // 📥 Fetch data — identical to your CI model calls
    $invoiceInfo = DB::table('c_invoices')
        ->where('user_id', $userId)
        ->orderBy('id', 'DESC')
        ->first();

    $userInfo = DB::table('c_reg_users')
        ->where('id', $userId)
        ->first();

    if (!$userInfo) {
        return response()->json(['message' => 'User not found'], 404);
    }

    $paymentInfo = DB::table('c_user_orders')
        ->where('user_id', $userId)
        ->orderBy('id', 'DESC')
        ->get();

    $bankinfo = DB::table('bank_details')
        ->where('status', 1)
        ->first();

    $configurations = DB::table('configurations')
        ->where('status', 1)
        ->orderBy('id', 'DESC')
        ->first();

    $address = DB::table('web_cms')
        ->where('page_type', 10)
        ->orderBy('id', 'DESC')
        ->first();

    // Bank details from other DB (adjust connection name as needed)
    $bankDetails = DB::table('company_info')
        ->where('status', 1)
        ->orderBy('id', 'DESC')
        ->first();

    $conferenceId = $invoiceInfo ? $invoiceInfo->conference_id : null;
    $confenceInfo = $conferenceId
        ? DB::table('conferences')->where('id', $conferenceId)->first()
        : null;

    // Registration info — conditional on user_type (exact CI logic)
    if ($userInfo->user_type == 8) {
        $regInfo = DB::table('c_user_group_registrations_dates')
            ->where('user_id', $userId)
            ->orderBy('id', 'DESC')
            ->first();
    } else {
        $regInfo = DB::table('c_user_registrations_dates')
            ->where('user_id', $userId)
            ->orderBy('id', 'DESC')
            ->first();
    }

    $regInfoacp = DB::table('c_user_registrations_acp')
        ->where('user_id', $userId)
        ->orderBy('id', 'DESC')
        ->first();

    $regInfoespn = DB::table('c_user_registrations_espn')
        ->where('user_id', $userId)
        ->orderBy('id', 'DESC')
        ->first();

    $addonInfo = DB::table('c_user_addon_amounts')
        ->where('user_id', $userId)
        ->orderBy('id', 'DESC')
        ->get();

    // ✅ Return data — structured for Vue (single objects, not wrapped arrays)
    return response()->json([
        'userInfo'       => $userInfo,
        'invoiceInfo'    => $invoiceInfo,
        'paymentInfo'    => $paymentInfo,
        'regInfo'        => $regInfo,
        'regInfoacp'     => $regInfoacp,
        'regInfoespn'    => $regInfoespn,
        'addonInfo'      => $addonInfo,
        'bankinfo'       => $bankinfo,
        'configurations' => $configurations,
        'address'        => $address,
        'bankDetails'    => $bankDetails,
        'confenceInfo'   => $confenceInfo,
        'userId'         => $userId,
    ]);
}


public function userReceipts(Request $request)
{
    // 🔐 1. Authenticate via token (same pattern as userInvoices)
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $adminUser = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$adminUser) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    // 📌 2. Get conference_id from request (NO session)
    $conferenceId = $request->input('conference_id') ?? $request->query('conference_id');
    if (!$conferenceId) {
        return response()->json(['message' => 'Conference ID missing'], 422);
    }

    $conference = DB::table('conferences')->where('id', $conferenceId)->first();
    if (!$conference) {
        return response()->json(['message' => 'Conference not found'], 404);
    }

    // 👤 3. Get user_id from request
    $userId = $request->input('user_id') ?? $request->query('user_id');
    if (!$userId) {
        return response()->json(['message' => 'User ID missing'], 422);
    }

    // 4. Load user info
    $userInfo = DB::table('c_reg_users')
        ->where('id', $userId)
        ->get();

    if ($userInfo->isEmpty()) {
        return response()->json([
            'status'  => false,
            'message' => 'User not found',
        ], 404);
    }

    $user = $userInfo->first();

    // 5. Original CI3 queries replicated for receipts

    // Invoice info (for reference in receipt view)
    $invoiceInfo = DB::table('c_invoices')
        ->where('user_id', $userId)
        ->where('conference_id', $conferenceId)
        ->get();

    // Receipts info
    $receiptsInfo = DB::table('c_receipts')
        ->where('user_id', $userId)
        ->where('conference_id', $conferenceId)
        ->get();

    // Payment info
    $paymentInfo = DB::table('c_user_orders')
        ->where('user_id', $userId)
        ->get();

    // Reg info (group vs normal based on user_type – same as CI3)
    if ((int) $user->user_type === 8) {
        $regInfo = DB::table('c_user_group_registrations_dates')
            ->where('user_id', $userId)
            ->get();
    } else {
        $regInfo = DB::table('c_user_registrations_dates')
            ->where('user_id', $userId)
            ->get();
    }

    // ACP / ESPN registrations
    $regInfoacp = DB::table('c_user_registrations_acp')
        ->where('user_id', $userId)
        ->get();

    $regInfoespn = DB::table('c_user_registrations_espn')
        ->where('user_id', $userId)
        ->get();

    // Bank info (active – same as invoices)
    $bankinfo = DB::table('bank_details')
        ->where('status', 1)
        ->get();

    // Addon amounts
    $addonInfo = DB::table('c_user_addon_amounts')
        ->where('user_id', $userId)
        ->get();

    // User orders (duplicate of paymentInfo in CI3)
    $userOrders = DB::table('c_user_orders')
        ->where('user_id', $userId)
        ->get();

    // Shipping info
    $shippingInfo = DB::table('c_registration_shipping')
        ->where('regid', $userId)
        ->get();

    // Configurations (status=1, DESC)
    $configurations = DB::table('configurations')
        ->where('status', 1)
        ->orderByDesc('id')
        ->get();

    // Address from web_cms (page_type = 10)
    $address = DB::table('web_cms')
        ->where('page_type', 10)
        ->orderByDesc('id')
        ->get();

    // company_info filtered by user currency (like userInvoices)
    $bankDetails = DB::table('company_info')
        ->where('currency', $user->currency)
        ->orderByDesc('id')
        ->get();

    // Conference info (wrap the one we already loaded)
    $confenceInfo = DB::table('conferences')
        ->where('id', $conferenceId)
        ->orderByDesc('id')
        ->get();

    // 6. Return JSON (no double-encoding)
    return response()->json([
        'status'         => true,
        'userInfo'       => $userInfo,
        'invoiceInfo'    => $invoiceInfo,
        'receiptsInfo'   => $receiptsInfo,
        'paymentInfo'    => $paymentInfo,
        'regInfo'        => $regInfo,
        'regInfoacp'     => $regInfoacp,
        'regInfoespn'    => $regInfoespn,
        'addonInfo'      => $addonInfo,
        'userOrders'     => $userOrders,
        'shippingInfo'   => $shippingInfo,
        'configurations' => $configurations,
        'bankDetails'    => $bankDetails,
        'confenceInfo'   => $confenceInfo,
        'address'        => $address,
        'bankinfo'       => $bankinfo,
        'userId'         => $userId,
        'user_type'      => (int) $user->user_type,
        'conference'     => $conference, // optional extra
    ]);
}

public function storeUserReceipt(Request $request)
{
    // 🔐 same token auth pattern
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $adminUser = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$adminUser) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    // ✅ validate (mirror CI3, but keep some nullable)
    $validated = $request->validate([
        'conference_id'     => 'required|integer',
        'user_id'           => 'required|integer',
        'transaction_id'    => 'required|string|max:255',
        'receipt_date'      => 'required|date',
        'description'       => 'required|string',

        'shipping_name'     => 'nullable|string|max:255',
        'shipping_email'    => 'nullable|string|max:255',
        'shipping_phone'    => 'nullable|string|max:50',
        'shipping_address'  => 'nullable|string',

        'billing_name'      => 'nullable|string|max:255',
        'billing_email'     => 'nullable|string|max:255',
        'billing_phone'     => 'nullable|string|max:50',
        'billing_address'   => 'nullable|string',

        'registration_type' => 'nullable|string|max:255',
        'reg_description'   => 'nullable|string|max:255',
        'currency'          => 'nullable|string|max:10',
        'amount'            => 'nullable|numeric',
    ]);

    $conferenceId = $validated['conference_id'];
    $userId       = $validated['user_id'];

    $receiptDate  = date('Y-m-d', strtotime($validated['receipt_date']));
    $receiptIdVal = now()->format('Ymd') . $userId; // Ymd + userId (same as CI3)

    $params = [
        'conference_id'     => $conferenceId,
        'user_id'           => $userId,
        'receipt_id'        => $receiptIdVal,
        'transaction_id'    => $validated['transaction_id'],
        'shipping_name'     => $validated['shipping_name'] ?? null,
        'shipping_email'    => $validated['shipping_email'] ?? null,
        'shipping_phone'    => $validated['shipping_phone'] ?? null,
        'shipping_address'  => $validated['shipping_address'] ?? null,
        'billing_name'      => $validated['billing_name'] ?? null,
        'billing_email'     => $validated['billing_email'] ?? null,
        'billing_phone'     => $validated['billing_phone'] ?? null,
        'billing_address'   => $validated['billing_address'] ?? null,
        'description'       => $validated['description'],
        'receipt_date'      => $receiptDate,
        'registration_type' => $validated['registration_type'] ?? null,
        'reg_description'   => $validated['reg_description'] ?? null,
        'currency'          => $validated['currency'] ?? null,
        'amount'            => $validated['amount'] ?? null,
        'status'            => 1,
        'created_date'      => now(),
    ];

    $id = DB::table('c_receipts')->insertGetId($params);

    // 🔁 mark payment_status as success (like CI3 savereceipts)
    DB::table('c_reg_users')
        ->where('id', $userId)
        ->update(['payment_status' => 'success']);

    return response()->json([
        'status'         => true,
        'message'        => 'Receipt created successfully',
        'receipt_db_id'  => $id,
        'receipt_id'     => $receiptIdVal,
    ]);
}


public function updateUserReceipt(Request $request, $id)
{
    // 🔐 same token auth pattern
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    $adminUser = \App\Models\Conferenceadmin\User::where('api_token', $token)->first();
    if (!$adminUser) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    // ✅ validate
    $validated = $request->validate([
        'user_id'           => 'required|integer',
        'transaction_id'    => 'required|string|max:255',
        'receipt_date'      => 'required|date',
        'description'       => 'required|string',

        'shipping_name'     => 'nullable|string|max:255',
        'shipping_email'    => 'nullable|string|max:255',
        'shipping_phone'    => 'nullable|string|max:50',
        'shipping_address'  => 'nullable|string',

        'billing_name'      => 'nullable|string|max:255',
        'billing_email'     => 'nullable|string|max:255',
        'billing_phone'     => 'nullable|string|max:50',
        'billing_address'   => 'nullable|string',

        'registration_type' => 'nullable|string|max:255',
        'reg_description'   => 'nullable|string|max:255',
        'currency'          => 'nullable|string|max:10',
        'amount'            => 'nullable|numeric',
    ]);

    $userId      = $validated['user_id'];
    $receiptDate = date('Y-m-d', strtotime($validated['receipt_date']));

    $params = [
        'transaction_id'    => $validated['transaction_id'],
        'shipping_name'     => $validated['shipping_name'] ?? null,
        'shipping_email'    => $validated['shipping_email'] ?? null,
        'shipping_phone'    => $validated['shipping_phone'] ?? null,
        'shipping_address'  => $validated['shipping_address'] ?? null,
        'billing_name'      => $validated['billing_name'] ?? null,
        'billing_email'     => $validated['billing_email'] ?? null,
        'billing_phone'     => $validated['billing_phone'] ?? null,
        'billing_address'   => $validated['billing_address'] ?? null,
        'description'       => $validated['description'],
        'receipt_date'      => $receiptDate,
        'registration_type' => $validated['registration_type'] ?? null,
        'reg_description'   => $validated['reg_description'] ?? null,
        'currency'          => $validated['currency'] ?? null,
        'amount'            => $validated['amount'] ?? null,
        'created_date'      => now(), // keeping same style as invoice update
    ];

    $updated = DB::table('c_receipts')->where('id', $id)->update($params);

    // 🔁 same as CI3: update payment_status to success
    DB::table('c_reg_users')
        ->where('id', $userId)
        ->update(['payment_status' => 'success']);

    return response()->json([
        'status'  => (bool) $updated,
        'message' => $updated ? 'Receipt updated successfully' : 'No changes',
    ]);
}


}
