<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Models\{Conference, Registration, UserRegistrationDate, UserAddonAmount, RegistrationShipping};
use Stripe\Stripe;

class ChatGPTPaymentController extends Controller
{
    private function getCurrencyCode($id)
    {
        return match ((int) $id) {
            1 => 'GBP',
            2 => 'USD',
            3 => 'EUR',
            default => 'EUR'
        };
    }

    private function getCurrencySymbol($id)
    {
        return match ((int) $id) {
            1 => '£',
            2 => '$',
            3 => '€',
            default => '€'
        };
    }

    /**
     * STEP 1 — Submit registration
     */
public function submitChatGPT(Request $request): JsonResponse
{
    DB::beginTransaction();
    try {
        // 1️⃣ Setup
        $order_id = rand(9999, 9999999);
        $transaction_id = 'SCS' . now()->format('yHmsdi');
        $created_date = now();
        $currency = $request->input('ncurrsel') ?? $request->input('currsel') ?? 2;

        // 🔍 Log absolutely everything in the request
        Log::info('=== Registration Started ===', [
            'order_id' => $order_id,
            'transaction_id' => $transaction_id,
            'currency_input' => $currency,
            'full_request' => $request->all(),
        ]);

        // 2️⃣ Validate conference
        $conference = Conference::find($request->conferenceId);
        if (!$conference) {
            Log::warning('Conference not found', ['conferenceId' => $request->conferenceId]);
            return response()->json(['success' => false, 'message' => 'Conference not found'], 404);
        }

        // 3️⃣ Create base registration
        $registration = Registration::create([
            'conference_id'  => $request->conferenceId,
            'designation'    => $request->designation,
            'firstname'      => $request->firstname,
            'lastname'       => $request->lastname,
            'usr_email'      => $request->usr_email,
            'country'        => $request->country,
            'user_phone'     => $request->user_phone,
            'usr_whatsapp'   => $request->whatsapp_number,
            'company'        => $request->company,
            'category'       => $request->category,
            'address'        => $request->address,
            'payment_method' => $request->payment_meathod,
            'user_type'      => 3,
            'payment_status' => 'Pending',
            'created_date'   => $created_date,
            'transaction_id' => $transaction_id,
            'total_amount'   => 0,
            'currency'       => $currency,
            'invoiceid'           => '',
            'abstract_title'      => '',
            'extra_info'          => '',
            'total_extra_amount'  => '',
            'coupon_value'        => '',
            'aditional_details'   => '',
        ]);

        Log::info('Base registration created', ['registration_id' => $registration->id]);

        // 4️⃣ Determine base registration amount
        $curDate = date('Y-m-d');
        Log::info('Determining base registration amount', ['curDate' => $curDate]);

        $catInfo = DB::table('c_registration_categories')
            ->where(['id' => $request->selectedCatId, 'status' => 1, 'page_type' => 1])
            ->first();

        $registrationAmount = DB::table('c_registrations_amount1')
            ->where([
                'conference_id' => $request->conferenceId,
                'currency_id'   => $currency,
                'category_id'   => $request->selectedCatId,
            ])
            ->first();

        $dates = DB::table('c_registrations_dates')
            ->where(['status' => 1, 'conference_id' => $request->conferenceId])
            ->first();

        Log::info('Fetched registration data', compact('catInfo', 'registrationAmount', 'dates'));

        $totalAmt = 0;
        if ($registrationAmount && $dates) {
            if (strtotime($dates->early_date) >= strtotime($curDate)) {
                $totalAmt = $registrationAmount->early_bird_amount;
            } elseif (strtotime($dates->standard_date) >= strtotime($curDate)) {
                $totalAmt = $registrationAmount->first_call_amount;
            } elseif (strtotime($dates->onsite_date) >= strtotime($curDate)) {
                $totalAmt = $registrationAmount->final_call_amount;
            } elseif (strtotime($dates->onsite_reg_date) >= strtotime($curDate)) {
                $totalAmt = $registrationAmount->onsite_registration;
            }
        }

        $baseAmount = round(($request->selected_qty ?? 1) * ($totalAmt ?: 0), 2);
        $totalAmount = $baseAmount;

        Log::info('Base amount computed', [
            'baseAmount' => $baseAmount,
            'totalAmount' => $totalAmount,
        ]);

$conferenceName = trim("{$conference->conference_first_title} {$conference->conference_name}");
$ptrm = [
    'regid'         => $registration->id,
    'conference_id' => $request->conferenceId,
    'shipDesign'    => $request->shipdesignation,
    'shipFirstname' => $request->shipfirstname,
    'shipLastname'  => $request->shiplastname,
    'shipCompany'   => $request->shipcompany,
    'shipCountry'   => $request->shipcountry,
    'shipEmail'     => $request->shipusr_email,
    'shipPhone'     => $request->shipphone,
    'shipWhatsapp'  => $request->shipwhatsapp_number ?: '0',
    'shipAddress'   => $request->shipaddress,
    'conference_name' => $conferenceName,
    'createDate'    => now(),
];
$inns = DB::table('c_registration_shipping')->insertGetId($ptrm);

Log::info('Shipping details inserted', [
    'insert_id' => $inns,
    'ptrm' => $ptrm,
]);
        // 5️⃣ Insert base registration details
        $insert1 = DB::table('c_user_registrations_dates')->insertGetId([
            'user_id'          => $registration->id,
            'order_id'         => $order_id,
            'registration_id'  => $request->selectedCatId,
            'registration_type'=> $catInfo->category_name ?? 'General',
            'radio_type'       => $request->selectedItemId,
            'reg_qty'          => $request->selected_qty ?? 1,
            'registration_name' => $registrationAmount->registration_name,
            'reg_total_amount' => $baseAmount,
            'curency_type'     => $currency,
            'radio_amount'     => $totalAmt,
            'total_amount'     => $baseAmount,
            'payment_meathod'  => $request->payment_meathod,
            'payment_status'   => 'Pending',
            'checkin'          => $request->checkin,
            'checkout'         => $request->checkout,
            'created_date'     => $created_date,
        ]);
        Log::info('Inserted into c_user_registrations_dates', ['insert1' => $insert1]);

        // 6️⃣ Accommodation Add-on
        $addoncatid = $request->input('addoncatidnew');
        Log::info('Checking add-ons section');

        // --- Handle JSON/array-based add-ons (Vue frontend) ---
        $addons = $request->input('add_ons', []);
        if (!is_array($addons)) {
            $addons = json_decode($addons, true) ?: [];
        }

        if (!empty($addons)) {
            Log::info('Processing add_ons array from frontend', ['addons' => $addons]);

            foreach ($addons as $addon) {
                $cat_id = $addon['cat_id'] ?? null;
                $item_id = $addon['item_id'] ?? null;
                $qty = $addon['qty'] ?? 1;
                $amount = $addon['amount'] ?? 0;
                $addonTotal = round($amount * $qty, 2);

                if ($cat_id && $item_id) {
                            if (!empty($cat_id)) {
            $addoncatid = $cat_id;
            $acccatInfo = DB::table('c_registration_categories')
                ->where(['status' => 1, 'id' => $addoncatid, 'page_type' => 2])
                ->first();
}

 $accInfo = DB::table('c_accommodation_v1')
                    ->where([
                        'status' => 1,
                        'conference_id' => $request->conferenceId,
                        'currency_id' => $currency,
                        'category_id' => $addoncatid,
                        'id' => $item_id,
                    ])
                    ->first();
                    
                    
                    DB::table('c_user_addon_amounts')->insert([
                        'order_id'           => $order_id,
                        'user_id'            => $registration->id,
                        'reg_id'             => $insert1,
                        'addon_catid'        => $cat_id,
                        'addon_subcatid'     => $item_id,
                        'addon_catname'      => $acccatInfo->category_name,
                        "addon_subcatname"   => $accInfo->accommodation_name,
                        'addon_amount'       => $amount,
                        'addon_qty'          => $qty,
                        'addon_total_amount' => $addonTotal,
                        'currency_type'      => $currency,
                        'reg_date'           => $created_date,
                        'created_date'       => $created_date,
                    ]);
                    $totalAmount += $addonTotal;
                    Log::info('Accommodation add-on processed', compact('cat_id','item_id','qty','amount','addonTotal','totalAmount'));
                } elseif (!$cat_id && $item_id && $amount > 0) {
                    DB::table('c_user_registrations_acp')->insert([
                        'order_id'            => $order_id,
                        'user_id'             => $registration->id,
                        'accompanying_name'   => 'Accompanying Person',
                        'radio_amount'        => $amount,
                        'acpreg_qty'          => $qty,
                        'acpreg_total_amount' => $addonTotal,
                        'curency_type'        => $currency,
                        'acptotal_amount'     => $addonTotal,
                        'payment_meathod'     => $request->payment_meathod,
                        'reg_date'            => $created_date,
                        'created_date'        => $created_date,
                    ]);
                    $totalAmount += $addonTotal;
                    Log::info('Accompanying person add-on processed', compact('item_id','qty','amount','addonTotal','totalAmount'));
                }
            }

            Log::info('All add_ons processed', ['final_total' => $totalAmount]);
        } else {
            Log::info('No accommodation add-on selected');
        }

        // 7️⃣ Accompanying Person
        if (($request->selected_qtyan ?? 0) > 0) {
            $acpRate = $request->acpamount ?? 0;
            $acpQty = $request->selected_qtyan;
            $acpTotal = round($acpRate * $acpQty, 2);
            $totalAmount += $acpTotal;

            DB::table('c_user_registrations_acp')->insert([
                'order_id'            => $order_id,
                'user_id'             => $registration->id,
                'accompanying_name'   => 'Accompanying Person',
                'earlyType' => $request->earlyType,
                'radio_amount'        => $acpRate,
                'acpreg_qty'          => $acpQty,
                'acpreg_total_amount' => $acpTotal,
                'curency_type'        => $currency,
                'acptotal_amount'     => $acpTotal,
                'payment_meathod'     => $request->payment_meathod,
                'reg_date'            => $created_date,
                'created_date'        => $created_date,
            ]);

            Log::info('Accompanying person added', [
                'acpRate' => $acpRate,
                'acpQty' => $acpQty,
                'acpTotal' => $acpTotal,
                'running_total' => $totalAmount,
            ]);
        } else {
            Log::info('No accompanying person selected');
        }

        // 8️⃣ Extra Nights — with CI3-style logic + detailed logging
        $accommodation_name = ($request->selected_adn_name ?? '') . ' (Extra Nights)';

        if (!empty($request->addoncatidnew)) {
            $addoncatid = $request->addoncatidnew;
            $acccatInfo = DB::table('c_registration_categories')
                ->where(['status' => 1, 'id' => $addoncatid, 'page_type' => 2])
                ->first();

            $addonsubcatid = $request->input('addonsubcatid_' . $addoncatid);

            if (is_array($addonsubcatid) && isset($addonsubcatid[0]) && $addonsubcatid[0] != 0) {
                $accInfo = DB::table('c_accommodation_v1')
                    ->where([
                        'status' => 1,
                        'conference_id' => $request->conferenceId,
                        'currency_id' => $currency,
                        'category_id' => $addoncatid,
                        'id' => $addonsubcatid[0],
                    ])
                    ->get();

                if ($accInfo->count() > 0) {
                    $accommodation_name = $accInfo[0]->accommodation_name . ' (Extra Nights)';
                } else {
                    $accommodation_name = ($request->selected_adn_name ?? '') . ' (Extra Nights)';
                }
            } else {
                $accommodation_name = ($request->selected_adn_name ?? '') . ' (Extra Nights)';
            }
        }

        if (($request->selected_qtyespn ?? 0) > 0) {
            $espnRate = $request->espnamount ?? 0;
            $espnQty = $request->selected_qtyespn;
            $espnTotal = round($espnRate * $espnQty, 2);
            $totalAmount += $espnTotal;
 $earlyType = $request->earlyType;
            DB::table('c_user_registrations_espn')->insert([
                'order_id'             => $order_id,
                'user_id'              => $registration->id,
                'extra_name'           => $accommodation_name,
                'earlyType' => $request->earlyType,
                'radio_amount'         => $espnRate,
                'espnreg_qty'          => $espnQty,
                'espnreg_total_amount' => $espnTotal,
                'curency_type'         => $currency,
                'espntotal_amount'     => $espnTotal,
                'payment_meathod'      => $request->payment_meathod,
                'reg_date'             => $created_date,
                'created_date'         => $created_date,
            ]);

            Log::info('Extra Nights added', [
                'accommodation_name' => $accommodation_name,
                'espnRate' => $espnRate,
                'espnQty' => $espnQty,
                'espnTotal' => $espnTotal,
                'running_total' => $totalAmount,
            ]);
        } else {
            Log::info('No extra nights selected');
        }

        // 9️⃣ Final total
        DB::table('c_reg_users')
            ->where('id', $registration->id)
            ->update(['total_amount' => $totalAmount]);

        Log::info('Final total updated in DB', [
            'registration_id' => $registration->id,
            'totalAmount' => $totalAmount,
        ]);

        DB::commit();

        Log::info('=== Registration Completed Successfully ===', [
            'order_id' => $order_id,
            'transaction_id' => $transaction_id,
            'total' => $totalAmount,
            'currency' => $this->getCurrencyCode($currency),
        ]);

        return response()->json([
            'success'        => true,
            'order_id'       => $order_id,
            'transaction_id' => $transaction_id,
            'total'          => $totalAmount,
            'currency'       => $this->getCurrencyCode($currency),
            'symbol'         => $this->getCurrencySymbol($currency),
        ]);

    } catch (\Exception $e) {
        DB::rollBack();
        Log::error('ChatGPT Submit Error: ' . $e->getMessage(), [
            'trace' => $e->getTraceAsString(),
        ]);
        return response()->json(['success' => false, 'message' => 'Registration failed'], 500);
    }
}








    /**
     * STEP 2 — Stripe Checkout
     */
    public function handleStripePaymentChatGPT(Request $request): JsonResponse
    {
        try {
            $conference = Conference::find($request->conference_id);
            if (!$conference) {
                return response()->json(['success' => false, 'message' => 'Conference not found'], 404);
            }

            $conferenceUrl = rtrim($conference->conference_url ?? config('app.frontend_url', env('FRONTEND_URL')), '/');
            $isSandbox     = env('PAYMENT_SANDBOX_V7', true);
            $stripeSecret  = $isSandbox ? env('STRIPE_SECRET_TEST_V7') : env('STRIPE_SECRET_LIVE_V7');

            Stripe::setApiKey($stripeSecret);

            $checkout = \Stripe\Checkout\Session::create([
                'payment_method_types' => ['card'],
                'line_items'           => [[
                    'price_data' => [
                        'currency'     => strtolower($this->getCurrencyCode($request->currsel ?? 2)),
                        'product_data' => ['name' => 'Conference Registration'],
                        'unit_amount'  => intval($request->amount * 100),
                    ],
                    'quantity' => 1,
                ]],
                'mode'        => 'payment',
                'success_url' => url("/registration/chatgpt/stripe/success?session_id={CHECKOUT_SESSION_ID}&order_id={$request->order_id}&conference_url=" . urlencode($conferenceUrl)),
                'cancel_url'  => url("/registration/chatgpt/stripe/cancel?order_id={$request->order_id}&conference_url=" . urlencode($conferenceUrl)),
                'metadata'    => [
                    'order_id'        => $request->order_id,
                    'conference_id'   => $request->conference_id,
                    'conference_name' => $conference->conference_name,
                    'environment'     => $isSandbox ? 'sandbox' : 'live',
                ],
            ]);

            DB::table('c_user_registrations_dates')
                ->where('order_id', (string) $request->order_id)
                ->update([
                    'stripe_session_id' => $checkout->id,
                    'payment_status'    => 'pending',
                    'updated_at'        => now(),
                ]);

            return response()->json([
                'success'    => true,
                'url'        => $checkout->url,
                'session_id' => $checkout->id,
                'mode'       => $isSandbox ? 'test' : 'live',
            ]);
        } catch (\Exception $e) {
            Log::error('Stripe ChatGPT Error: ' . $e->getMessage());
            return response()->json(['success' => false, 'message' => 'Stripe failed: ' . $e->getMessage()], 500);
        }
    }

    public function stripeSuccess(Request $request)
    {
        $sessionId     = $request->query('session_id');
        $orderId       = $request->query('order_id');
        $conferenceUrl = rtrim($request->query('conference_url') ?? config('app.frontend_url', env('FRONTEND_URL')), '/');

        try {
            $isSandbox    = env('PAYMENT_SANDBOX_V7', true);
            $stripeSecret = $isSandbox ? env('STRIPE_SECRET_TEST_V7') : env('STRIPE_SECRET_LIVE_V7');
            Stripe::setApiKey($stripeSecret);

            $session       = \Stripe\Checkout\Session::retrieve($sessionId);
            $paymentIntent = \Stripe\PaymentIntent::retrieve($session->payment_intent);
            $userReg       = UserRegistrationDate::where('stripe_session_id', $sessionId)->first();

            if (!$userReg) {
                Log::error("No record found for stripe_session_id: $sessionId");
                return redirect()->to("{$conferenceUrl}/chatgpt-stripe-failed?order_id={$orderId}");
            }

            DB::table('c_user_registrations_dates')
                ->where('order_id', $orderId)
                ->update([
                    'payment_status'    => 'paid',
                    'payment_intent_id' => $paymentIntent->id,
                    'updated_at'        => now(),
                ]);

            DB::table('c_reg_users')
                ->where('id', $userReg->user_id)
                ->update([
                    'payment_status' => 'paid',
                    'transaction_id' => $paymentIntent->id,
                ]);

            return redirect()->to("{$conferenceUrl}/chatgpt-stripe-success?order_id={$orderId}");
        } catch (\Exception $e) {
            Log::error('Stripe Success Handler Error: ' . $e->getMessage());
            return redirect()->to("{$conferenceUrl}/chatgpt-stripe-failed?order_id={$orderId}");
        }
    }

    public function stripeCancel(Request $request)
    {
        $orderId       = $request->query('order_id');
        $conferenceUrl = rtrim($request->query('conference_url') ?? config('app.frontend_url', env('FRONTEND_URL')), '/');

        try {
            if (!$orderId) {
                Log::warning('Stripe cancel missing order_id.');
                return redirect()->to("{$conferenceUrl}/chatgpt-stripe-cancel");
            }

            $userReg = DB::table('c_user_registrations_dates')->where('order_id', $orderId)->first();
            if ($userReg) {
                DB::table('c_user_registrations_dates')
                    ->where('order_id', $orderId)
                    ->update([
                        'payment_meathod' => 1,
                        'payment_status'  => 'Cancelled',
                        'updated_at'      => now(),
                    ]);

                DB::table('c_reg_users')
                    ->where('id', $userReg->user_id)
                    ->update([
                        'payment_status' => 'Cancelled',
                    ]);
            }

            return redirect()->to("{$conferenceUrl}/chatgpt-stripe-cancel?order_id={$orderId}");
        } catch (\Exception $e) {
            Log::error('Stripe Cancel Handler Error: ' . $e->getMessage());
            return redirect()->to("{$conferenceUrl}/chatgpt-stripe-cancel?order_id={$orderId}");
        }
    }

    /**
     * STEP 3 — PayPal Redirect
     */
    public function handlePaypalPaymentChatGPT(Request $request): JsonResponse
    {
        try {
            $conference = Conference::find($request->conference_id);
            if (!$conference) {
                return response()->json(['success' => false, 'message' => 'Conference not found'], 404);
            }

            $sandbox        = env('PAYMENT_SANDBOX_V7', true);
            $paypalBusiness = $sandbox ? env('PAYPAL_BUSINESS_SANDBOX_V7') : env('PAYPAL_BUSINESS_LIVE_V7');
            $paypalUrl      = $sandbox
                ? 'https://www.sandbox.paypal.com/cgi-bin/webscr'
                : 'https://www.paypal.com/cgi-bin/webscr';

            $conferenceUrl = rtrim($conference->conference_url ?? config('app.frontend_url', env('FRONTEND_URL')), '/');
            $currencyCode  = $this->getCurrencyCode($request->currsel ?? 2);

            $query = http_build_query([
                'cmd'           => '_xclick',
                'business'      => $paypalBusiness,
                'item_name'     => 'Conference Registration',
                'amount'        => number_format($request->amount, 2, '.', ''),
                'currency_code' => $currencyCode,
                'return'        => url("/registration/chatgpt/paypal/success?order_id={$request->order_id}&conference_url=" . urlencode($conferenceUrl)),
                'cancel_return' => url("/registration/chatgpt/paypal/cancel?order_id={$request->order_id}&conference_url=" . urlencode($conferenceUrl)),
                'notify_url'    => url('/api/paypal/ipn-handler'),
                'custom'        => $request->order_id,
            ]);

            return response()->json([
                'success'      => true,
                'redirect_url' => "{$paypalUrl}?{$query}",
            ]);
        } catch (\Exception $e) {
            Log::error('PayPal ChatGPT Error: ' . $e->getMessage());
            return response()->json(['success' => false, 'message' => 'PayPal failed'], 500);
        }
    }

    /**
     * ✅ PayPal Redirect Success — user-facing
     */
    public function paypalSuccess(Request $request)
    {
        $orderId       = $request->query('order_id');
        $conferenceUrl = rtrim($request->query('conference_url') ?? config('app.frontend_url', env('FRONTEND_URL')), '/');

        try {
            Log::info('PayPal Success Redirect Data:', $request->query());

            if (!$orderId) {
                Log::warning('PayPal success missing order_id.');
                return redirect()->to("{$conferenceUrl}/chatgpt-paypal-failed");
            }

            // Mark as pending until IPN confirms completion
            DB::table('c_user_registrations_dates')
                ->where('order_id', $orderId)
                ->update([
                    'payment_meathod' => 2,
                    'payment_status'  => 'Pending',
                    'updated_at'      => now(),
                ]);

            return redirect()->to("{$conferenceUrl}/chatgpt-paypal-success?order_id={$orderId}");
        } catch (\Exception $e) {
            Log::error('PayPal Success Handler Error: ' . $e->getMessage());
            return redirect()->to("{$conferenceUrl}/chatgpt-paypal-failed?order_id={$orderId}");
        }
    }

    /**
     * ✅ PayPal IPN — background verification
     */
    public function paypalIpnHandler(Request $request)
    {
        Log::info('PayPal IPN Data Received:', $request->all());

        $orderId       = $request->input('custom');
        $txnId         = $request->input('txn_id');
        $payerId       = $request->input('payer_id');
        $paymentStatus = $request->input('payment_status');
        $mcGross       = $request->input('mc_gross');
        $mcCurrency    = $request->input('mc_currency');

        if (!$orderId) {
            Log::warning('PayPal IPN missing order_id.');
            return response('ignored', 200);
        }

        DB::table('c_user_registrations_dates')
            ->where('order_id', $orderId)
            ->update([
                'payment_meathod' => 2,
                'payment_status'  => $paymentStatus === 'Completed' ? 'Completed' : 'Pending',
                'updated_at'      => now(),
            ]);

        $userId = DB::table('c_user_registrations_dates')->where('order_id', $orderId)->value('user_id');
        if ($userId) {
            DB::table('c_reg_users')
                ->where('id', $userId)
                ->update([
                    'payment_status' => $paymentStatus === 'Completed' ? 'Completed' : 'Pending',
                    'transaction_id' => $txnId,
                    'total_amount'   => $mcGross,
                ]);
        }

        Log::info("PayPal IPN processed for order_id: {$orderId}, txn_id: {$txnId}");
        return response('OK', 200);
    }

    public function paypalCancel(Request $request)
    {
        $orderId       = $request->query('order_id');
        $conferenceUrl = rtrim($request->query('conference_url') ?? config('app.frontend_url', env('FRONTEND_URL')), '/');

        try {
            if (!$orderId) {
                Log::warning('PayPal cancel missing order_id.');
                return redirect()->to("{$conferenceUrl}/chatgpt-paypal-cancel");
            }

            $userReg = DB::table('c_user_registrations_dates')->where('order_id', $orderId)->first();
            if ($userReg) {
                DB::table('c_user_registrations_dates')
                    ->where('order_id', $orderId)
                    ->update([
                        'payment_meathod' => 2,
                        'payment_status'  => 'Cancelled',
                        'updated_at'      => now(),
                    ]);

                DB::table('c_reg_users')
                    ->where('id', $userReg->user_id)
                    ->update([
                        'payment_status' => 'Cancelled',
                    ]);
            }

            return redirect()->to("{$conferenceUrl}/chatgpt-paypal-cancel?order_id={$orderId}");
        } catch (\Exception $e) {
            Log::error('PayPal Cancel Handler Error: ' . $e->getMessage());
            return redirect()->to("{$conferenceUrl}/chatgpt-paypal-cancel?order_id={$orderId}");
        }
    }
}
