<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Conference;
use App\Models\Category;
use App\Models\Speaker;
use Illuminate\Http\JsonResponse;

class AllSpeakersApiController extends Controller
{
    public function show($id): JsonResponse
    {
        $conference = Conference::with([
            'banners' => fn($q) => $q->where('page_type', 24)->where('status', 1)->orderByDesc('id'),
            'AllSpeakersmetaData'
        ])->find($id);

        if (!$conference) {
            return response()->json(['error' => 'Conference not found'], 404);
        }

        $meta = $conference->AllSpeakersmetaData;
        $banners = $conference->banners;

        // Get all categories, ordered
        $categories = Category::orderBy('sort_order')->get();

        // Preload speakers for this conference WITH their countries
        $speakers = Speaker::with('country')
            ->where('conference_id', $id)
            ->orderBy('orderpos')
            ->get()
            ->groupBy('speaker_type'); // Group by category ID

        // Build grouped speakers array
        $groupedSpeakers = $categories->map(function ($category) use ($speakers) {
            $catSpeakers = $speakers->get($category->id, collect())->map(function ($speaker) {
                return [
                    'id' => $speaker->id,
                    'speaker_img' => $speaker->speaker_img,
                    'alias_name' => $speaker->alias_name,
                    'speaker_name' => $speaker->speaker_name,
                    'speaker_title' => $speaker->speaker_title,
                    'affliation' => $speaker->affliation,
                    'speaker_type' => $speaker->speaker_type,
                    'countryInfo' => $speaker->country ? [['country_name' => $speaker->country->country_name]] : []
                ];
            });

            return [
                'category_id' => $category->id,
                'category_name' => $category->category_name,
                'speakers' => $catSpeakers->values()
            ];
        })->values();

        return response()->json([
            'success' => true,
            'cntlrname' => 'all-speakers',
            'allspeakersData' => [
                'groupedSpeakers' => $groupedSpeakers,
                'banners' => $banners
            ],
            'meta' => [
                'title' => $meta?->meta_title ?? 'All Speakers – Conference',
                'description' => $meta?->meta_description ?? 'All speakers grouped by category.',
                'keywords' => $meta?->meta_keywords ?? 'all speakers, conference'
            ],
            'conferenceInfo' => $conference
        ]);
    }
}