<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Helpers\AdminAuditLogger; // âœ… NEW

class AdminController extends Controller
{
    use AdminAuditLogger; // âœ… Use new trait

    /**
     * LOGIN (CI3 pattern with SHA1)
     * POST /backend/api/admin/login
     */
    public function login(Request $request)
    {
        $email    = $request->input('email');
        $password = sha1($request->input('password')); // same as CI3 checkUser()

        $user = DB::table('users')
            ->where('email', $email)
            ->where('password', $password)
            ->where('status', 1)
            ->whereIn('usertype', [1]) // only Super Admin like your CI3 checkUser
            ->first();

        if (!$user) {
            return response()->json(['message' => 'Invalid Username / Password'], 401);
        }

        // OLD row
        $old = (array) $user;

        // Generate new API token
        $token = bin2hex(random_bytes(32));

        DB::table('users')
            ->where('id', $user->id)
            ->update(['api_token' => $token]);

        // NEW row
        $newUser = DB::table('users')->where('id', $user->id)->first();

        // ðŸ§¾ ADMIN audit (conference_id = 0 by default)
        $this->logAdminAudit(
            'users',
            $user->id,
            'updated',
            $old,
            (array) $newUser
        );

        // role text like your header.php
        if ((string)$newUser->usertype === '1') {
            $reso = 'Super Admin';
        } elseif ((string)$newUser->usertype === '2') {
            $reso = 'Admin';
        } else {
            $reso = '';
        }
        $newUser->role_text = $reso;

        return response()->json([
            'token' => $token,
            'user'  => $newUser,
        ]);
    }

    /**
     * LOGOUT
     * POST /backend/api/admin/logout
     */
    public function logout(Request $request)
    {
        // resolve admin by api_token
        $token = $request->bearerToken()
            ?? $request->header('authorization')
            ?? $request->input('token')
            ?? $request->query('token');

        $user = null;

        if ($token) {
            $user = DB::table('users')
                ->where('api_token', $token)
                ->where('status', 1)
                ->whereIn('usertype', [1, 2])
                ->first();
        }

        if ($user) {
            $old = (array) $user;

            DB::table('users')
                ->where('id', $user->id)
                ->update(['api_token' => null]);

            $new = DB::table('users')->where('id', $user->id)->first();

            $this->logAdminAudit(
                'users',
                $user->id,
                'updated',
                $old,
                (array) $new
            );
        }

        return response()->json(['message' => 'Logged out successfully']);
    }

    /**
     * HEADER API for AdminLayout.vue (global admin)
     * GET /backend/api/admin/header
     */
public function header(Request $request)
{
    // 1) Token resolve
    $token = $request->bearerToken()
        ?? $request->header('authorization')
        ?? $request->input('token')
        ?? $request->query('token');

    if ($token && stripos($token, 'bearer ') === 0) {
        $token = trim(substr($token, 7));
    }

    $user = null;

    if ($token) {
        $user = DB::table('users')
            ->where('api_token', $token)
            ->where('status', 1)
            ->whereIn('usertype', [1, 2]) // 1 = Super Admin, 2 = Admin
            ->first();
    }

    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    // 2) Role text
    if ((string) $user->usertype === '1') {
        $roleText = 'Super Admin';
    } elseif ((string) $user->usertype === '2') {
        $roleText = 'Admin';
    } else {
        $roleText = '';
    }
    $user->role_text = $roleText;

    $menu = [];

    /*
    |--------------------------------------------------------------------------
    | Common menu for usertype 1 & 2 (same as CI3 big if-block)
    |--------------------------------------------------------------------------
    */
    if (in_array((string) $user->usertype, ['1', '2'], true)) {

        // Dashboard
        $menu[] = [
            'label' => 'Dashboard',
            'icon'  => 'fa fa-home',
            'to'    => '/admin/dashboard',
        ];

        // General Info (Logo, Continents, Whatsapp, etc.)
        $menu[] = [
            'label' => 'General Info',
            'icon'  => 'fa fa-cogs',
            'children' => [
                [ 'label' => 'Logo',                           'icon' => 'fa fa-file-image-o', 'to' => '/admin/logo' ],
                [ 'label' => 'Continents',                     'icon' => 'fa fa-globe',        'to' => '/admin/v4/continents' ],
                [ 'label' => 'Whatsapp And Facebook Number',   'icon' => 'fa fa-whatsapp',     'to' => '/admin/v4/whatsapp' ],
                [ 'label' => 'Live Chat',                      'icon' => 'fa fa-comments-o',   'to' => '/admin/v4/chat-script' ],
                [ 'label' => 'Facebook Messenger',             'icon' => 'fa fa-comments-o',   'to' => '/admin/v4/messenger-script' ],
                [ 'label' => 'Subjects',                       'icon' => 'fa fa-book',         'to' => '/admin/v4/subjects' ],
                [ 'label' => 'Abstract Categories',            'icon' => 'fa fa-sitemap',      'to' => '/admin/v4/categories' ],
                [ 'label' => 'Registration Categories',        'icon' => 'fa fa-user-plus',    'to' => '/admin/v4/regcategories' ],
                [ 'label' => 'Speaker Categories',             'icon' => 'fa fa-user-plus',    'to' => '/admin/v4/regcategories_new' ],
                [ 'label' => 'Registration Payment Categories','icon' => 'fa fa-usd',          'to' => '/admin/v4/regcategories_reg' ],
                [ 'label' => 'Universities',                   'icon' => 'fa fa-university',   'to' => '/admin/v4/universities_reg' ],
                [ 'label' => 'Hospitals',                      'icon' => 'fa fa-h-square',     'to' => '/admin/v4/hospitals_reg' ],
                [ 'label' => 'Society Associations',           'icon' => 'fa fa-users',        'to' => '/admin/v4/societies_reg' ],
            ],
        ];

        // Bank Information (USD / GBP / EUR)
        $menu[] = [
            'label' => 'Bank Information',
            'icon'  => 'fa fa-bars',
            'children' => [
                [ 'label' => 'USD', 'icon' => 'fa fa-money', 'to' => '/admin/v4/company_info/2/edit' ], // map CI3 bankInfo1/2
                [ 'label' => 'GBP', 'icon' => 'fa fa-money', 'to' => '/admin/v4/company_info/1/edit' ], // bankInfo1/1
                [ 'label' => 'EUR', 'icon' => 'fa fa-money', 'to' => '/admin/v4/company_info/3/edit' ], // bankInfo1/3
            ],
        ];

        // Single items from that block
        $menu[] = [
            'label' => 'Conferences',
            'icon'  => 'fa fa-sitemap',
            'to'    => '/admin/conferences', // was /home/conferences
        ];

      $menu[] = [
    'label' => 'Social Links',
    'icon'  => 'fa fa-share-alt',
    'to'    => '/admin/social',
];

        $menu[] = [
            'label' => 'Advisory Board',
            'icon'  => 'fa fa-user-plus',
            'to'    => '/admin/v4/board_12', // was /home/board
        ];

$menu[] = [
    'label' => 'Speaker Guidelines',
    'icon'  => 'fa fa-file-text-o',
    'to'    => '/admin/v4/guidelines_7',   // 👈 v4 route for page_type=5
];

$menu[] = [
    'label' => 'Membership Content',
    'icon'  => 'fa fa-file-text-o',
    'to'    => '/admin/v4/guidelines_15',   // 👈 v4 route for page_type=5
];


        $menu[] = [
            'label' => 'Invoice & Bank Info',
            'icon'  => 'fa fa-building-o',
            'to'    => '/admin/bank-info', // was /home/bankinfo
        ];

  $menu[] = [
    'label' => 'Online Payments',
    'icon'  => 'fa fa-credit-card',      // more accurate icon
    'to'    => '/admin/v4/regusers_6',   // ✅ v4 route for type=6
];

$menu[] = [
    'label' => 'Registered Users',
    'icon'  => 'fa fa-users',
    'to'    => '/admin/v4/regusers_3',   // ✅ v4 route for type=3 (default)
];

$menu[] = [
    'label' => 'Group Registered Users',
    'icon'  => 'fa fa-users',
    'to'    => '/admin/v4/regusers_8',   // ✅ v4 route for type=8
];

$menu[] = [
    'label' => 'Discount Users',
    'icon'  => 'fa fa-percent',
    'to'    => '/admin/v4/regusers_4',   // ✅ v4 route for type=4
];

    $menu[] = [
  'label' => 'Previous Conference Gallery',
  'to'    => '/admin/v4/gallery',
  'icon'  => 'fa fa-picture-o',
  'children' => [],
  'menu'  => 'gallery', // the slug used in menuMapV4
];
    $menu[] = [
  'label' => 'Previous Conference Videos',
  'to'    => 'admin/v4/videos',
  'icon'  => 'fa fa-picture-o',
  'children' => [],
  'menu'  => 'videos', // the slug used in menuMapV4
];

       // in /admin/header response



        // v4 – Subscriptions
        $menu[] = [
            'label' => 'Subscriptions',
            'icon'  => 'fa fa-envelope',
            'to'    => '/admin/v4/subscriptions', // was /home/subscriptions
            'menu'  => 'subscriptions',            // ⭐ router key
        ];

        // v4 – Meta
        $menu[] = [
            'label' => 'Meta',
            'icon'  => 'fa fa-globe',
            'to'    => '/admin/v4/meta', // was /home/meta
            'menu'  => 'meta',           // ⭐ router key
        ];

        // v4 – Invoices
        $menu[] = [
            'label' => 'Invoices',
            'icon'  => 'fa fa-usd',
            'to'    => '/admin/v4/userinvoices', // was /home/userInvoices
            'menu'  => 'userinvoices',           // ⭐ router key
        ];
    }

    /*
    |--------------------------------------------------------------------------
    | Super Admin (usertype == 1) extra menu (Admins / Assigned Admins / etc.)
    |--------------------------------------------------------------------------
    */
    if ((string) $user->usertype === '1') {

        // Group like "Admin Management" / "Conference Admin"
        $menu[] = [
            'label' => 'Admin Management',
            'icon'  => 'fa fa-bars',
            'children' => [
                [
                    'label' => 'Admins',
                    'icon'  => 'fa fa-user',
                    'to'    => '/admin/v4/admusers',   // was /home/admusers
                    'menu'  => 'admusers',             // ⭐ router key
                ],
                [
                    'label' => 'Assigned Admins',
                    'icon'  => 'fa fa-user',
                    'to'    => '/admin/v4/aadmins',    // was /home/aadmins
                    'menu'  => 'aadmins',              // ⭐ router key
                ],
                [
                    'label' => 'Super Admins',
                    'icon'  => 'fa fa-user-secret',
                    'to'    => '/admin/v4/admusers1',  // was /home/admusers1
                    'menu'  => 'admusers1',            // ⭐ router key
                ],
                [
                    'label' => 'Assigned Super Admins',
                    'icon'  => 'fa fa-user-secret',
                    // 🔗 new SPA + v4 pattern (user_confs1)
                    'to'    => '/admin/v4/aadmins1',   // replaces CI3 /home/aadmins1
                    'menu'  => 'aadmins1',             // ⭐ router key
                ],
            ],
        ];
    }
    /*
    |--------------------------------------------------------------------------
    | Normal Admin (usertype == 2) – Manuscript Reviews instead of admin mgmt
    |--------------------------------------------------------------------------
    */
    elseif ((string) $user->usertype === '2') {
        $menu[] = [
            'label' => 'Manuscript Reviews',
            'icon'  => 'fa fa-file-pdf-o',
            'to'    => '/admin/manuscripts', // was /home/manuscripts
        ];
    }

    return response()->json([
        'user'      => $user,
        'siteTitle' => 'Scisynopsis Conferences',
        'menu'      => $menu,
    ]);
}



    /**
     * FOOTER API
     * GET /backend/api/admin/footer
     */
    public function footer()
    {
        return response()->json([
            'name'      => 'Scisynopsis Conferences',
            'year'      => date('Y'),
            'copyright' => 'Â© ' . date('Y') . ' Scisynopsis Conferences',
        ]);
    }

    /**
     * DASHBOARD DATA (if you want JSON for cards etc.)
     */
    public function dashboard(Request $request)
    {
        return response()->json([
            'welcome_message' => 'Welcome To Scisynopsis Conferences Admin Panel',
            'user'            => null, // or resolved user if needed
        ]);
    }
    
        public function logo(Request $request)
    {
        $user = $this->resolveAdminFromToken($request);
        if (!$user) {
            return response()->json(['message' => 'Unauthorized'], 401);
        }

        $logoPath    = public_path('includes/images/logo.png');
        $favIconPath = public_path('includes/images/fav.png');

        $logoUrl    = file_exists($logoPath)    ? url('includes/images/logo.png') : null;
        $favIconUrl = file_exists($favIconPath) ? url('includes/images/fav.png')  : null;

        return response()->json([
            'logo'     => $logoUrl,
            'fav_icon' => $favIconUrl,
        ]);
    }

    /**
     * POST /api/admin/logo/save â€“ CI3 savelogo()
     */
    public function saveLogo(Request $request)
    {
        $user = $this->resolveAdminFromToken($request);
        if (!$user) {
            return response()->json(['message' => 'Unauthorized'], 401);
        }

        $folder = public_path('includes/images');
        if (!is_dir($folder)) {
            mkdir($folder, 0777, true);
        }

        $updated = [];

        if ($request->hasFile('mainImage')) {
            $file = $request->file('mainImage');
            if ($file->isValid()) {
                $file->move($folder, 'logo.png');
                $updated['logo'] = 'logo.png';
                Log::info('Admin logo.png updated');
            }
        }

        if ($request->hasFile('mainImage1')) {
            $file = $request->file('mainImage1');
            if ($file->isValid()) {
                $file->move($folder, 'fav.png');
                $updated['fav_icon'] = 'fav.png';
                Log::info('Admin fav.png updated');
            }
        }

        return response()->json([
            'message' => 'Logo Changed sucessfully.',
            'updated' => $updated,
        ]);
    }
}
