<?php

namespace App\Helpers;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

trait AuditLogger
{
    /**
     * Log audit information
     */
  public function logAudit(
    string $table,
    int|string $rowId,
    string $event,
    ?array $old = null,
    ?array $new = null
): void {
    $req = request();

    /** ---------------------------------------------------------
     * 1) Field-level DIFF కోసం default
     * ---------------------------------------------------------
     */
    $diff = [];

    // Updated → compute diff
    if ($event === 'updated' && $old && $new) {
        foreach ($new as $key => $newVal) {
            $oldVal = $old[$key] ?? null;

            if ($oldVal != $newVal) {
                $diff[$key] = [
                    'old' => $oldVal,
                    'new' => $newVal,
                ];
            }
        }
    }

    // Created → అన్ని fields ను new గా చూపించాలి
    if ($event === 'created' && $new) {
        foreach ($new as $key => $val) {
            $diff[$key] = [
                'old' => null,
                'new' => $val,
            ];
        }
    }

    // Deleted → అన్ని fields ను old గా చూపించి new=null
    if ($event === 'deleted' && $old) {
        foreach ($old as $key => $val) {
            $diff[$key] = [
                'old' => $val,
                'new' => null,
            ];
        }
    }

    /** ---------------------------------------------------------
     * 2) FILE columns కోసం ప్రత్యేక diff block
     * ---------------------------------------------------------
     */
    $fileColumns = [
        'speaker_img', 'acceptance_letter', 'cms_img',
        'venue_img', 'venue_img2', 'venue_img3',
        'journal_img', 'gallery_img', 'workshop_img',
        'testimonial_image', 'track_icon',
        'cover_img', 'program_cover_img',
        'brouchure_name',
    ];

    $fileChanges = [];

    if (in_array($event, ['created', 'updated', 'deleted'], true)) {
        foreach ($fileColumns as $col) {
            // null-safe గా values తీసుకోవడం
            $oldVal = is_array($old) ? ($old[$col] ?? null) : null;
            $newVal = is_array($new) ? ($new[$col] ?? null) : null;

            // created: सिर्फ new ఉన్నప్పుడు
            if ($event === 'created' && $newVal !== null) {
                $fileChanges[$col] = [
                    'old' => null,
                    'new' => $newVal,
                ];
            }
            // deleted: सिर्फ old ఉన్నప్పుడు
            elseif ($event === 'deleted' && $oldVal !== null) {
                $fileChanges[$col] = [
                    'old' => $oldVal,
                    'new' => null,
                ];
            }
            // updated: change ఐతే మాత్రమే
            elseif ($event === 'updated' && $oldVal != $newVal) {
                $fileChanges[$col] = [
                    'old' => $oldVal,
                    'new' => $newVal,
                ];
            }
        }
    }

    // fileChanges ఉన్నప్పుడే diff లో special block గా add చేయాలి
    if (!empty($fileChanges)) {
        $diff['files'] = $fileChanges;
    }

    /** ---------------------------------------------------------
     * 3) Save to DB
     * ---------------------------------------------------------
     */
    DB::table('audit_logs')->insert([
        'table_name'     => $table,
        'row_id'         => $rowId,
        // request లో user_id రాకపోతే Auth::id() fallback optional గా use చేయొచ్చు
        'user_id'        => $req->input('user_id', \Auth::id()),
        'event'          => $event,
        'diff'           => !empty($diff) ? json_encode($diff) : null,
        'conference_id'  => $req->input('conference_id'),

        'ip_address'     => $req->ip(),
        'url'            => $req->fullUrl(),

        'latitude'       => $req->input('lat'),
        'longitude'      => $req->input('lng'),
        'location_label' => $req->input('location_label'),

        'created_at'     => now(),
        'updated_at'     => now(),
    ]);
}

}
