<?php

namespace App\Helpers;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

trait AdminAuditLogger
{
    /**
     * Admin-specific audit logger
     * - conference_id is optional, defaults to 0
     */
    public function logAdminAudit(
        string $table,
        int|string $rowId,
        string $event,
        ?array $old = null,
        ?array $new = null,
        ?int $conferenceId = null
    ): void {
        $req = request();

        // -----------------------------
        // 1) Field-level DIFF
        // -----------------------------
        $diff = [];

        if ($event === 'updated' && $old && $new) {
            foreach ($new as $key => $newVal) {
                $oldVal = $old[$key] ?? null;

                if ($oldVal != $newVal) {
                    $diff[$key] = [
                        'old' => $oldVal,
                        'new' => $newVal,
                    ];
                }
            }
        }

        if ($event === 'created' && $new) {
            foreach ($new as $key => $val) {
                $diff[$key] = [
                    'old' => null,
                    'new' => $val,
                ];
            }
        }

        if ($event === 'deleted' && $old) {
            foreach ($old as $key => $val) {
                $diff[$key] = [
                    'old' => $val,
                    'new' => null,
                ];
            }
        }

        // -----------------------------
        // 2) File columns special diff
        // (same pattern as Conference AuditLogger)
        // -----------------------------
        $fileColumns = [
            'speaker_img', 'acceptance_letter', 'cms_img',
            'venue_img', 'venue_img2', 'venue_img3',
            'journal_img', 'gallery_img', 'workshop_img',
            'testimonial_image', 'track_icon',
            'cover_img', 'program_cover_img',
            'brouchure_name',
        ];

        $fileChanges = [];

        if (in_array($event, ['created', 'updated', 'deleted'], true)) {
            foreach ($fileColumns as $col) {
                $oldVal = is_array($old) ? ($old[$col] ?? null) : null;
                $newVal = is_array($new) ? ($new[$col] ?? null) : null;

                if ($event === 'created' && $newVal !== null) {
                    $fileChanges[$col] = [
                        'old' => null,
                        'new' => $newVal,
                    ];
                } elseif ($event === 'deleted' && $oldVal !== null) {
                    $fileChanges[$col] = [
                        'old' => $oldVal,
                        'new' => null,
                    ];
                } elseif ($event === 'updated' && $oldVal != $newVal) {
                    $fileChanges[$col] = [
                        'old' => $oldVal,
                        'new' => $newVal,
                    ];
                }
            }
        }

        if (!empty($fileChanges)) {
            $diff['files'] = $fileChanges;
        }

        // -----------------------------
        // 3) Insert into audit_logs
        // -----------------------------
        try {
            DB::table('audit_logs')->insert([
                'table_name'     => $table,
                'row_id'         => $rowId,
                'user_id'        => $req->input('user_id', Auth::id()),
                'event'          => $event,
                'diff'           => !empty($diff) ? json_encode($diff) : null,

                'conference_id'  => $conferenceId ?? (int) $req->input('conference_id', 0),

                'ip_address'     => $req->ip(),
                'url'            => $req->fullUrl(),

                'latitude'       => $req->input('lat'),
                'longitude'      => $req->input('lng'),
                'location_label' => $req->input('location_label'),

                'created_at'     => now(),
                'updated_at'     => now(),
            ]);
        } catch (\Throwable $e) {
            // Don't break main flow if audit fails
            \Log::error('AdminAuditLogger failed', [
                'message' => $e->getMessage(),
            ]);
        }
    }
}
