<?php

namespace App\Http\Controllers\Company;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Company\{
    CompanyMetaData,
    CompanyHomepageScript,
    CompanyConfiguration,
    CompanyWebCms,
    CompanyChatScript,
    CompanyWhatsApp,
    CompanyPageNavigation,
    CompanyBanner,
    CompanyPreviousConferenceGallery
};

class CompanyGalleryController extends Controller
{
    /**
     * GET /company/conference-gallery
     * or GET /company/conference-gallery/{alias_name}
     */
public function getGallery(Request $request, $alias_name = null, $year_id = null)
{
    $pageType = 8;

    // Header data
    $meta = CompanyMetaData::where('page_type', $pageType)->latest('id')->first();
    $homepage_script = CompanyHomepageScript::where('status', 1)->latest('id')->first();
    $banners = CompanyBanner::where('status', 1)
        ->where('page_type', $pageType)
        ->orderByDesc('id')
        ->get();

    // ===== CASE 1: Main gallery page =====
    if ($alias_name === null) {
        $years = CompanyPreviousConferenceGallery::select('year_id')
            ->where('status', 1)
            ->where('parent_id', 0)
            ->distinct()
            ->orderByDesc('year_id')
            ->get();

        $yearsList = [];
        foreach ($years as $year) {
            $galleryImages = CompanyPreviousConferenceGallery::getGalleryImages($year->year_id, 0);
            $yearsList[] = [
                'year_id' => $year->year_id,
                'galleryImages' => $galleryImages,
            ];
        }

        return response()->json(array_merge([
            'meta' => $meta,
            'homepage_script' => $homepage_script?->script_content,
            'banners' => $banners,
            'yearsList' => $yearsList,
        ], $this->getFooterData()));
    }

    // ===== CASE 2: Gallery Details page =====
    $query = CompanyPreviousConferenceGallery::where('status', 1)
        ->where('alias_title', trim($alias_name))
        ->where('parent_id', 0);

    if ($year_id) {
        $query->where('year_id', $year_id);
    }

    $gallery = $query->first();

    $galleryInfo = [];
    if ($gallery) {
        $galleryInfo = CompanyPreviousConferenceGallery::where('status', 1)
            ->where('parent_id', $gallery->id)
            ->orderBy('id', 'ASC')
            ->get();
    }

    return response()->json(array_merge([
        'meta' => $meta,
        'homepage_script' => $homepage_script?->script_content,
        'banners' => $banners,
        'gallery' => $gallery ? [$gallery] : [],
        'galleryInfo' => $galleryInfo,
    ], $this->getFooterData()));
}


    /**
     * 🔹 Shared footer, social, nav data for all company pages
     */
    private function getFooterData()
    {
        $address = CompanyWebCms::where('status', 1)
            ->where('page_type', 10)
            ->latest('id')
            ->first();

        $sociallinks = CompanyWebCms::where('status', 1)
            ->where('page_type', 11)
            ->latest('id')
            ->first();

        $whatsapp = CompanyWhatsApp::where('status', 1)->latest('id')->first();
        $chatScript = CompanyChatScript::where('status', 1)->latest('id')->first();

        $journalsUrl = CompanyConfiguration::where('status', 1)->latest('id')->first();
        $conferenceUrl = CompanyConfiguration::where('status', 1)->latest('id')->first();

        $headerNavigation = CompanyPageNavigation::where('status', 1)
            ->where('page_type', 1)
            ->where('parent_id', 0)
            ->orderBy('orderpos', 'ASC')
            ->with('children')
            ->get();

        $footerQuickLinks = CompanyPageNavigation::where('status', 1)
            ->where('page_type', 2)
            ->where('parent_id', 0)
            ->orderBy('orderpos', 'ASC')
            ->get();

        $footerUsefulLinks = CompanyPageNavigation::where('status', 1)
            ->where('page_type', 3)
            ->where('parent_id', 0)
            ->orderBy('orderpos', 'ASC')
            ->get();

        return [
            // FOOTER
            'address' => $address,
            'sociallinks' => $sociallinks,
            'whatsapp' => $whatsapp,
            'chatScript' => $chatScript?->script_content,
            'journalsUrl' => $journalsUrl,
            'conferenceUrl' => $conferenceUrl,

            // NAVIGATION
            'headerNavigation' => $headerNavigation,
            'footerQuickLinks' => $footerQuickLinks,
            'footerUsefulLinks' => $footerUsefulLinks,
        ];
    }
}
