<?php

namespace App\Http\Controllers\Api;

use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Stripe\Stripe;
use Stripe\PaymentIntent;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;

class RegistrationApiController extends Controller
{
    public function index($id, $currency = 3)
    {
        $currency = in_array($currency, [1, 2, 3]) ? $currency : 3;

        $conferenceInfo = DB::table('conferences')->where('id', $id)->first();
        if (!$conferenceInfo) {
            return response()->json(['success' => false, 'message' => 'Conference not found'], 404);
        }

        $designations = DB::table('designations')->where('status', 1)->get();
        $countries = DB::table('countries')->where('status', 1)->orderBy('country_name', 'ASC')->get();

        $dates = DB::table('c_registrations_dates')
            ->where(['status' => 1, 'conference_id' => $id])
            ->first();
            

        // Main Categories (page_type = 1)
        $regCategories = DB::table('c_registration_categories')
            ->where(['status' => 1, 'page_type' => 1])
            ->get();
             $categories = DB::table('categories')
            ->where(['status' => 1])
            ->get();
            

        $regCatArray = [];
        foreach ($regCategories as $cat) {
            $amounts = DB::table('c_registrations_amount')
                ->where([
                    'status' => 1,
                    'currency_id' => $currency,
                    'conference_id' => $id,
                    'category_id' => $cat->id
                ])
                ->get();

            $regCatArray[] = [
                'id' => $cat->id,
                'category_name' => $cat->category_name,
                'registrationAmount' => $amounts
            ];
        }

        // Add-On Categories (page_type = 2)
        $addOnCategories = DB::table('c_registration_categories')
            ->where(['status' => 1, 'page_type' => 2])
            ->get();

        $addOnArray = [];
        foreach ($addOnCategories as $cat) {
            $amounts = DB::table('c_accommodation')
                ->where([
                    'status' => 1,
                    'currency_id' => $currency,
                    'conference_id' => $id,
                    'category_id' => $cat->id
                ])
                ->get();

            $addOnArray[] = [
                'id' => $cat->id,
                'category_name' => $cat->category_name,
                'addOnAmount' => $amounts
            ];
        }

        $onsiteStatus = DB::table('c_onsite_reg_status')->where('conference_id', $id)->first();
        $settings = DB::table('c_settings')->where('conference_id', $id)->first();
        $captcha = rand(100000, 999999);
  $registrationContent = DB::table('c_registration_content')->where(['conference_id' => $id])->get();
        return response()->json([
    'success' => true,
    'data' => [
        'conferenceInfo' => $conferenceInfo,
        'designations' => $designations,
        'countries' => $countries,
        'categories'=>$categories,
        'dates' => $dates,
        'regCategories' => $regCatArray,
        'addOnCategories' => $addOnArray,
        'onsiteStatus' => $onsiteStatus,
        'settings' => $settings,
        'newcaptcha' => $captcha,
        'symb' => $this->getCurrencySymbol($currency),
        'registrationContent' => @$registrationContent
    ]
]);
    }
   
    private function getCountryCode($countryName)
{
    if (!$countryName) return '';

    $country = DB::table('countries')->where('country_name', $countryName)->first();

    return $country ? ($country->phonecode ?? $country->phone_code ?? $country->country_code ?? $country->code ?? '') : '';
}
 
 private function getCurrencySymbol($id)
{
    return match((int)$id) {
        1 => '£', // GBP
        2 => '$', // USD
        3 => '€', // EUR
        default => '€'
    };
}   
    public function submit(\Illuminate\Http\Request $request)
{
    

    DB::beginTransaction();
 $order_id ='SCS-' . strtoupper(uniqid());
        // Insert into c_reg_users
        $userId = DB::table('c_reg_users')->insertGetId([
            'conference_id' => $request->conferenceId,
            'designation' => $request->designation,
            'firstname' => $request->firstname,
            'lastname' => $request->lastname,
            'usr_email' => $request->usr_email,
            'country' => $request->country,
            'country_code' => $this->getCountryCode($request->country),
            'user_phone' => $request->user_phone,
            'usr_whatsapp' => $request->whatsapp_number,
            'company' => $request->company,
            'category' => $request->category,
            'address' => $request->address,
            'payment_method' => $request->payment_meathod,
            'total_amount' => $request->totalAmount,
            'currency' => $request->currsel,
            'created_date' => now(),
        ]);

        // Insert shipping
        DB::table('c_registration_shipping')->insert([
            'regid' => $userId,
            'conference_id' => $request->conferenceId,
            'shipDesign' => $request->shipdesignation,
            'shipFirstname' => $request->shipfirstname,
            'shipLastname' => $request->shiplastname,
            'shipCompany' => $request->shipcompany,
            'shipCountry' => $request->shipcountry,
            'shipEmail' => $request->shipusr_email,
            'shipPhone' => $request->shipphone,
            'shipWhatsapp' => $request->shipwhatsapp_number,
            'shipAddress' => $request->shipaddress,
            'createDate' => now(),
        ]);

        // Insert main registration
        $regId = DB::table('c_user_registrations_dates')->insertGetId([
            'user_id' => $userId,
            'order_id' => $order_id,
            'earlyType' => null,
            'registration_id' => $request->selectedCatId,
            'radio_type' => $request->selectedItemId,
            'reg_qty' => $request->selectedQty,
            'reg_total_amount' => $request->selectedAmount * $request->selectedQty,
            'total_amount' => $request->totalAmount,
            'curency_type' => $request->currsel,
            'payment_meathod' => $request->payment_meathod,
            'reg_date' => now(),
            'created_date' => now(),
        ]);

        // Insert add-ons (loop if needed)
        foreach ($request->add_ons ?? [] as $addOn) {
            DB::table('c_user_addon_amounts')->insert([
                'order_id' => $order_id,
                'user_id' => $userId,
                'reg_id' => $regId,
                'addon_catid' => $addOn['cat_id'],
                'addon_subcatid' => $addOn['item_id'],
                'addon_qty' => $addOn['qty'],
                'addon_amount' => $addOn['amount'],
                'addon_total_amount' => $addOn['qty'] * $addOn['amount'],
                'currency_type' => $request->currsel,
                'reg_date' => now(),
                'created_date' => now(),
            ]);
        }

        DB::commit();

        return response()->json([
            'success' => true,
            'registration_id' => $userId,
            'order_id' => $order_id,
            'total' => $request->totalAmount,
            'currency' => $this->getCurrencyCode($request->currsel),
            'symbol' => $this->getCurrencySymbol($request->currsel),
        ]);

  

}

private function getCurrencyCode($id)
{
    return match((int)$id) {
        1 => 'GBP',
        2 => 'USD',
        3 => 'EUR',
        default => 'EUR'
    };
}

public function createPaymentIntent(Request $request)
{
    $request->validate([
        'order_id' => 'required|string',
        'amount' => 'required|integer|min:1',
        'currency' => 'required|string|in:usd,eur,gbp',
        'user.name' => 'required|string',
        'user.email' => 'required|email',
        'conference_id' => 'required|integer',
        'transaction_id' => 'nullable|string|max:255', // ✅ Allow null
    ]);

    Stripe::setApiKey(env('STRIPE_SECRET_KEY'));

    // Fetch conference info
    $conferenceInfo = DB::table('conferences')
        ->where('id', $request->conference_id)
        ->first();

    if (!$conferenceInfo) {
        return response()->json(['error' => 'Conference not found'], 404);
    }

    // Build description
    $conferenceName = trim("{$conferenceInfo->conference_first_title} {$conferenceInfo->conference_name}");
    $description = "Registration: {$conferenceName}";
    if ($request->transaction_id) {
        $description .= " (Item ID: {$request->transaction_id})";
    }

    // Optional: Add more to metadata
    try {
        $intent = PaymentIntent::create([
            'amount' => $request->amount,
            'currency' => $request->currency,
            'description' => $description,
            'metadata' => [
                'order_id' => $request->order_id,
                'user_name' => $request->user['name'],
                'user_email' => $request->user['email'],
                'conference_id' => $request->conference_id,
                'conference_name' => $conferenceName,
                'transaction_id' => $request->transaction_id ?? '',
            ],
            'payment_method_types' => ['card'],
        ]);

        return response()->json(['clientSecret' => $intent->client_secret]);
    } catch (\Exception $e) {
        \Log::error('Stripe PaymentIntent failed: ' . $e->getMessage());
        return response()->json(['error' => 'Payment setup failed'], 500);
    }
}

public function updatePaymentSuccess(Request $request, $order_id)
{
    $request->validate([
        'payment_intent' => 'required|string',
        'status' => 'required|in:paid,cancelled'
    ]);

    $updated = DB::table('c_user_registrations_dates')
        ->where('order_id', $order_id)
        ->update([
            'payment_status' => $request->status,
            'payment_intent_id' => $request->payment_intent,
        ]);
                    $users = DB::table('c_user_registrations_dates')->where('order_id', $order_id)->first();

 DB::table('c_reg_users')
        ->where('id', $users->user_id)
        ->update([
            'payment_status' => $request->status,
            'transaction_id' => $order_id,
      
        ]);
   
    if (!$updated) {
        return response()->json(['success' => false, 'message' => 'Order not found'], 404);
    }

    return response()->json(['success' => true]);
}
public function getPaymentSuccessData($order_id)
{
    $data = DB::table('c_reg_users as u')
        ->join('c_user_registrations_dates as r', 'u.id', '=', 'r.user_id')
        ->where('r.order_id', $order_id)
        ->select(
            'u.*',
            'r.order_id',
            'r.payment_status',
            'r.total_amount',
            'r.created_date',
            'r.payment_meathod'
        )
        ->first();

    if (!$data) {
        return response()->json(['success' => false, 'message' => 'Order not found'], 404);
    }

    return response()->json([
        'success' => true,
        'data' => (array) $data
    ]);
}

public function summary($orderId)
{
    $data = DB::table('c_reg_users as u')
        ->join('c_user_registrations_dates as r', 'u.id', '=', 'r.user_id')
        ->where('r.order_id', $orderId)
        ->select(
            'u.*',
            'r.order_id',
            'r.payment_status',
            'r.total_amount',
            'r.created_date',
            'r.payment_meathod'
        )
        ->first();

    if (!$data) {
        return response()->json(['success' => false, 'message' => 'Order not found'], 404);
    }

    return response()->json([
        'success' => true,
        'data' => (array) $data
    ]);
}


 // init endpoint for Vue Online.vue
    public function init()
    {
        try {
            $designations = DB::table('designations')
                ->where('status', 1)
                ->orderBy('id', 'ASC')
                ->get();

            $categories = DB::table('categories')
                ->where('status', 1)
                ->orderBy('id', 'ASC')
                ->get();

            $countries = DB::table('countries')
                ->where('status', 1)
                ->orderBy('country_name', 'ASC')
                ->get();

            return response()->json([
                'success' => true,
                'designations' => $designations,
                'categories' => $categories,
                'countries' => $countries,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load registration data',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    public function getPhoneCode($name)
{
    $country = DB::table('countries')
        ->where('country_name', $name)
        ->where('status', 1)
        ->first();

    if ($country && $country->country_code) {
        return response()->json([
            'success' => true,
            'phonecode' => '+' . $country->country_code,
        ]);
    }

    return response()->json([
        'success' => false,
        'message' => 'Country code not found'
    ], 404);
}

public function indexV4($id, $currency = 3)
{
    // Copy logic from index() — or call it if no customization needed
    return $this->index($id, $currency);
}

public function submitV4(Request $request)
{
    // Copy logic from submit() — customize if needed for v4
    return $this->submit($request);
}

public function createPaymentIntentV4(Request $request)
{
    return $this->createPaymentIntent($request);
}

public function updatePaymentSuccessV4(Request $request, $order_id)
{
    return $this->updatePaymentSuccess($request, $order_id);
}



public function summaryV4($order_id)
{
    return $this->summary($order_id);
}

public function initV4()
{
    return $this->init();
}

public function getPhoneCodeV4($name)
{
    return $this->getPhoneCode($name);
}
public function getPaymentSuccessDataV4($order_id)
{
    $data = DB::table('c_reg_users as u')
        ->join('c_user_registrations_dates as r', 'u.id', '=', 'r.user_id')
        ->where('r.order_id', $order_id)
        ->select(
            'u.firstname',
            'u.lastname',
            'u.usr_email',
            'u.user_phone',
            'u.company',
            'u.country',
            'u.invoiceid',
            'r.order_id',
            'r.total_amount',
            'r.curency_type as currency_id',
            'r.payment_meathod as payment_method',
            'r.payment_status',
            'u.transaction_id',
            'r.created_date'
        )
        ->first();

    if (!$data) {
        return response()->json(['success' => false, 'message' => 'Order not found'], 404);
    }

    $currencyCode = match ((int)($data->currency_id ?? 3)) {
        1 => 'GBP',
        2 => 'USD',
        3 => 'EUR',
        default => 'EUR',
    };

    return response()->json([
        'success' => true,
        'data' => [
            'order_id'       => $data->order_id,
            'firstname'      => $data->firstname,
            'lastname'       => $data->lastname,
            'usr_email'      => $data->usr_email,
            'user_phone'     => $data->user_phone,
            'company'        => $data->company,
            'country'        => $data->country,
            'total_amount'   => $data->total_amount,
            'currency'       => $currencyCode,
            'payment_method' => (int) $data->payment_method,
            'payment_status' => $data->payment_status,
            'transaction_id' => $data->transaction_id,
            'invoiceid'      => $data->invoiceid,
            'created_date'   => $data->created_date,
        ],
    ]);
}

public function stripeReturnV4(Request $request)
{
    // Stripe sends ?payment_intent=...&payment_intent_client_secret=...&redirect_status=succeeded
    $piId = $request->query('payment_intent');
    if (!$piId) {
        return redirect(config('app.frontend_url', env('FRONTEND_URL')).'/stripe-payment-failed');
    }

    \Stripe\Stripe::setApiKey(env('STRIPE_SECRET_KEY'));

    try {
        /** @var \Stripe\PaymentIntent $pi */
        $pi = \Stripe\PaymentIntent::retrieve($piId);
        $orderId = $pi->metadata['order_id'] ?? null;

        if (!$orderId) {
            return redirect(config('app.frontend_url', env('FRONTEND_URL')).'/stripe-payment-failed');
        }

        // Derive status from Stripe (don’t trust client)
        $newStatus = $pi->status === 'succeeded' ? 'paid' : 'cancelled';

        // Update DB (same logic as your updatePaymentSuccess but server-driven)
        $updated = DB::table('c_user_registrations_dates')
            ->where('order_id', $orderId)
            ->update([
                'payment_status'    => $newStatus,
                'payment_intent_id' => $pi->id,
            ]);

        if ($updated) {
            $userRow = DB::table('c_user_registrations_dates')->where('order_id', $orderId)->first();
            if ($userRow) {
                DB::table('c_reg_users')
                    ->where('id', $userRow->user_id)
                    ->update([
                        'payment_status' => $newStatus,
                        'transaction_id' => $orderId,
                    ]);
            }
        }

        // Put order_id in a short-lived cookie so the SPA can fetch details without URL params
        $cookie = cookie(
            'scs_order',             // name
            $orderId,                // value
            5,                       // minutes
            null, null,              // path, domain
            false,                   // secure (set true on HTTPS)
            false,                   // httpOnly = false (SPA must read it)
            false,                   // raw
            'Lax'                    // sameSite
        );

        // Redirect to the SEO-friendly SPA route
        $frontend = rtrim(config('app.frontend_url', env('FRONTEND_URL')), '/');
        return redirect($frontend.'/stripe-payment-success-v4')->withCookie($cookie);

    } catch (\Exception $e) {
        \Log::error('Stripe return verify failed: '.$e->getMessage());
        $frontend = rtrim(config('app.frontend_url', env('FRONTEND_URL')), '/');
        return redirect($frontend.'/stripe-payment-failed');
    }
}

}