<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Stripe\Stripe;
use Stripe\PaymentIntent;

class OnlineRegistrationApiController extends Controller
{
    /**
     * Load initial data (designations, categories, countries)
     */
    public function init()
    {
        $designations = DB::table('designations')->where('status', 1)->get();
        $categories   = DB::table('categories')->where('status', 1)->get();
        $countries    = DB::table('countries')->where('status', 1)->get();

        return response()->json([
            'success'      => true,
            'designations' => $designations,
            'categories'   => $categories,
            'countries'    => $countries
        ]);
    }

    /**
     * Get country code by name
     */
    public function countryCode($country)
    {
        $row = DB::table('countries')->where('country_name', $country)->first();
        if ($row) {
            return response()->json(['success' => true, 'phonecode' => $row->phonecode]);
        }
        return response()->json(['success' => false]);
    }

    /**
     * Checkout – inserts registration, returns payment instructions
     */
    public function checkout(Request $request)
    {
        $data = $request->all();

        $orderId = rand(100000, 99999999);
        $transactionId = "TXN" . rand(999, 99999) . date("ymdHis");
        $created = now();

        // insert user
        $userId = DB::table('c_reg_users')->insertGetId([
            'conference_id' => $data['conference_id'] ?? null,
            'conference_name' => $data['conference_name'] ?? '',
            'designation' => $data['designation'] ?? '',
            'firstname' => $data['firstname'] ?? '',
            'lastname' => $data['lastname'] ?? '',
            'category' => $data['category'] ?? '',
            'company' => $data['company'] ?? '',
            'country' => $data['country'] ?? '',
            'usr_email' => $data['usr_email'] ?? '',
            'user_phone' => $data['user_phone'] ?? '',
            'usr_whatsapp' => $data['whatsapp_number'] ?? '',
            'currency' => $data['currency'] ?? '',
            'total_amount' => $data['total_amount'] ?? 0,
            'payment_meathod' => $data['payment_meathod'] ?? 0,
            'created_date' => $created,
            'invoiceid' => date("ymdHis"),
            'transaction_id' => $transactionId
        ]);

        // insert registration order
        DB::table('c_user_registrations_dates')->insert([
            'order_id' => $orderId,
            'registration_name' => $data['conference_name'] ?? '',
            'user_id' => $userId,
            'curency_type' => $data['currency'] ?? '',
            'total_amount' => $data['total_amount'] ?? 0,
            'payment_meathod' => $data['payment_meathod'] ?? 0,
            'created_date' => $created
        ]);

        // Stripe
        if ($data['payment_meathod'] == 1) {
            Stripe::setApiKey(env('STRIPE_SECRET'));
            $intent = PaymentIntent::create([
                'amount' => intval($data['total_amount'] * 100), // cents
                'currency' => strtolower($this->mapCurrency($data['currency'])),
                'metadata' => [
                    'order_id' => $orderId,
                    'user_id' => $userId
                ]
            ]);

            return response()->json([
                'success' => true,
                'payment' => 'stripe',
                'client_secret' => $intent->client_secret,
                'order_id' => $orderId
            ]);
        }

        // PayPal
        if ($data['payment_meathod'] == 2) {
            $paypalAmount = $data['total_amount'] + ($data['total_amount'] * 3 / 100);

            $paypalUrl = url("/payment/return") . "?order_id=" . $orderId;

            return response()->json([
                'success' => true,
                'payment' => 'paypal',
                'paypal_url' => $paypalUrl,
                'order_id' => $orderId
            ]);
        }

        return response()->json(['success' => false, 'message' => 'Invalid payment method']);
    }

    /**
     * Success page summary
     */
    public function summary($orderId)
    {
        $order = DB::table('c_user_registrations_dates')->where('order_id', $orderId)->first();
        if (!$order) {
            return response()->json(['success' => false, 'message' => 'Order not found']);
        }

        $user = DB::table('c_reg_users')->where('id', $order->user_id)->first();

        return response()->json([
            'success' => true,
            'data' => [
                'order_id' => $order->order_id,
                'total_amount' => $order->total_amount,
                'currency' => $order->curency_type,
                'payment_meathod' => $order->payment_meathod,
                'usr_email' => $user->usr_email ?? '',
                'user_phone' => $user->user_phone ?? '',
                'created_date' => $order->created_date
            ]
        ]);
    }

    private function mapCurrency($id)
    {
        if ($id == 1) return 'gbp';
        if ($id == 2) return 'usd';
        if ($id == 3) return 'eur';
        return 'usd';
    }
}
