<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ConferenceRegistrationController extends Controller
{
    public function checkoutOnline(Request $request)
    {
        $request->validate([
            'firstname' => 'required|string|max:255',
            'lastname' => 'required|string|max:255',
            'usr_email' => 'required|email|max:255',
            'user_phone' => 'required|string|max:20',
            'country' => 'required|string',
            'designation' => 'required|string',
            'company' => 'required|string',
            'category' => 'required|string',
            'currency' => 'required|integer|in:1,2,3',
            'total_amount' => 'required|numeric|min:0.01',
            'payment_method' => 'required|integer|in:1,2',
            'cat_type' => 'required|integer',
            'conference_name' => 'required|string',
        ]);

        try {
            $onConferenceId = $request->input('onconference_id');
            if (!$onConferenceId) {
                $conferenceInfo = DB::table('conferences')
                    ->where('conference_url', $request->input('c_url'))
                    ->first();
                $onConferenceId = $conferenceInfo ? $conferenceInfo->id : null;
            }

            if (!$onConferenceId) {
                return response()->json(['success' => false, 'message' => 'Invalid conference'], 400);
            }

            $invoiceId = now()->format('ymdHis');
            $orderId = rand(9999, 9999999);
            $transactionId = 'SCS' . rand(999, 99999) . now()->format('ymdHis');

            $userId = DB::table('c_reg_users')->insertGetId([
                'conference_id' => $onConferenceId,
                'cat_type' => $request->input('cat_type', 1),
                'conference_name' => $request->input('conference_name'),
                'designation' => $request->input('designation'),
                'firstname' => $request->input('firstname'),
                'lastname' => $request->input('lastname'),
                'category' => $request->input('category'),
                'company' => $request->input('company'),
                'country' => $request->input('country'),
                'usr_email' => $request->input('usr_email'),
                'user_phone' => $request->input('user_phone'),
                'currency' => $request->input('currency'),
                'total_amount' => $request->input('total_amount'),
                'payment_method' => $request->input('payment_method'),
                'payment_status' => 'pending',
                'created_date' => now(),
                'invoiceid' => $invoiceId,
                'transaction_id' => $transactionId,
            ]);

            if ($userId) {
                DB::table('c_registration_shipping')->insert([
                    'regid' => $userId,
                    'shipFirstname' => $request->input('shipfirstname'),
                    'shipLastname' => $request->input('shiplastname'),
                    'shipEmail' => $request->input('shipusr_email'),
                    'shipPhone' => $request->input('shipphone'),
                    'shipAddress' => $request->input('shipaddress'),
                    'createDate' => now(),
                ]);

                $currencyCode = ['1' => 'GBP', '2' => 'USD', '3' => 'EUR'][$request->input('currency')] ?? 'USD';
                $finalAmount = $request->payment_method == 2 ? 
                    $request->total_amount * 1.03 : 
                    $request->total_amount;

                return response()->json([
                    'success' => true,
                    'payment_method' => $request->input('payment_method'),
                    'data' => [
                        'userId' => $userId,
                        'transactionId' => $transactionId,
                        'amount' => $finalAmount,
                        'currencyCode' => $currencyCode,
                        'conferenceName' => $request->input('conference_name'),
                        'firstname' => $request->input('firstname'),
                        'lastname' => $request->input('lastname'),
                        'email' => $request->input('usr_email'),
                    ]
                ]);
            }

            return response()->json([
                'success' => false, 
                'message' => 'Registration failed'
            ], 500);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function handlePayPalIPN(Request $request)
    {
        $paypalUrl = config('services.paypal.sandbox') ? 
            'https://www.sandbox.paypal.com/cgi-bin/webscr' : 
            'https://www.paypal.com/cgi-bin/webscr';

        $req = 'cmd=_notify-validate&' . http_build_query($request->all());

        $ch = curl_init($paypalUrl);
        curl_setopt($ch, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_1);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $req);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 1);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
        $res = curl_exec($ch);
        curl_close($ch);

        if (strcmp($res, "VERIFIED") == 0) {
            DB::table('c_reg_users')
                ->where('id', $request->item_number1)
                ->update([
                    'payment_status' => $request->payment_status === 'Completed' ? 'completed' : 'pending',
                    'transaction_id' => $request->txn_id,
                ]);
            return response('OK', 200);
        }

        return response('INVALID', 400);
    }

    public function getSuccessDetails($userId)
    {
        $registration = DB::table('c_reg_users')->where('id', $userId)->first();

        if (!$registration) {
            return response()->json(['success' => false, 'message' => 'Not found'], 404);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'userId' => $registration->id,
                'firstname' => $registration->firstname,
                'lastname' => $registration->lastname,
                'email' => $registration->usr_email,
                'conferenceName' => $registration->conference_name,
                'amount' => $registration->total_amount,
                'currency' => ['1' => 'GBP', '2' => 'USD', '3' => 'EUR'][$registration->currency] ?? 'USD',
                'transactionId' => $registration->transaction_id,
                'paymentStatus' => $registration->payment_status,
            ]
        ]);
    }
        public function getConferenceData($id)
    {
        $conferenceInfo = DB::table('conferences')
            ->where('id', $id)
            ->first();

        $countries = DB::table('countries')
            ->where('status', 1)
            ->orderBy('country_name', 'ASC')
            ->get();

        $designations = DB::table('designations')
            ->where('status', 1)
            ->orderBy('id', 'ASC')
            ->get();

        $categories = DB::table('categories')
            ->where('status', 1)
            ->orderBy('id', 'ASC')
            ->get();

        return response()->json([
            'conferenceInfo' => $conferenceInfo,
            'countries' => $countries,
            'designations' => $designations,
            'categories' => $categories,
        ]);
    }

}