<?php

namespace App\Http\Controllers\Api;

use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Stripe\Stripe;
use Stripe\PaymentIntent;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;

class XyzRegistrationController extends Controller
{
    public function index($id, $currency = 3)
    {
        $currency = in_array($currency, [1, 2, 3]) ? $currency : 3;

        $conferenceInfo = DB::table('conferences')->where('id', $id)->first();
        if (!$conferenceInfo) {
            return response()->json(['success' => false, 'message' => 'Conference not found'], 404);
        }

        $designations = DB::table('designations')->where('status', 1)->orderBy('id', 'ASC')->get();
        $countries = DB::table('countries')->where('status', 1)->orderBy('country_name', 'ASC')->get();
        $categories = DB::table('categories')->where('status', 1)->orderBy('id', 'ASC')->get();
        $captcha = rand(100000, 999999);

        return response()->json([
            'success' => true,
            'data' => [
                'conferenceInfo' => $conferenceInfo,
                'designations' => $designations,
                'countries' => $countries,
                'categories' => $categories,
                'newcaptcha' => $captcha,
                'symb' => $this->getCurrencySymbol($currency),
            ],
        ]);
    }

    private function getCountryCode($countryName)
    {
        if (!$countryName) return '';

        $country = DB::table('countries')->where('country_name', $countryName)->where('status', 1)->first();
        return $country ? ($country->country_code ?? '') : '';
    }

    private function getCurrencySymbol($id)
    {
        return match((int)$id) {
            1 => '£', // GBP
            2 => '$', // USD
            3 => '€', // EUR
            default => '€'
        };
    }

    private function getCurrencyCode($id)
    {
        return match((int)$id) {
            1 => 'GBP',
            2 => 'USD',
            3 => 'EUR',
            default => 'EUR'
        };
    }

    public function submit(Request $request)
    {
        $request->validate([
            'conferenceId' => 'required|integer',
            'designation' => 'required|string',
            'firstname' => 'required|string',
            'lastname' => 'required|string',
            'usr_email' => 'required|email',
            'country' => 'required|string',
            'user_phone' => 'required|string',
            'category' => 'required|string',
            'totalAmount' => 'required|numeric|min:0',
            'paymentMethod' => 'required|in:1,2',
            'currencyId' => 'required|in:1,2,3',
            'scingroup' => 'required|integer|min:0',
            'termschk' => 'required|boolean',
            'captcha' => 'required|string',
        ]);

        DB::beginTransaction();
        try {
            $order_id = 'SCS-' . strtoupper(uniqid());
            $invoiceid = date('ymdHis');
            $created_date = now();
            $cat_type = $request->cat_type == 2 ? 'Conference' : 'Journal';

            $userId = DB::table('c_reg_users')->insertGetId([
                'conference_id' => $request->conferenceId,
                'cat_type' => $cat_type,
                'conference_name' => $request->conference_name,
                'designation' => $request->designation,
                'firstname' => $request->firstname,
                'lastname' => $request->lastname,
                'category' => $request->category,
                'sessions_id' => $request->sessions_id,
                'company' => $request->company,
                'country' => $request->country,
                'country_code' => $this->getCountryCode($request->country),
                'usr_email' => $request->usr_email,
                'user_phone' => $request->user_phone,
                'usr_whatsapp' => $request->whatsapp_number,
                'user_type' => 6,
                'currency' => $request->currencyId,
                'total_amount' => $request->totalAmount,
                'payment_method' => $request->paymentMethod,
                'created_date' => $created_date,
                'invoiceid' => $invoiceid,
                'transaction_id' => $order_id,
            ]);

            DB::table('c_registration_shipping')->insert([
                'regid' => $userId,
                'cat_type' => $cat_type,
                'conference_name' => $request->conference_name,
                'shipDesign' => $request->shipdesignation,
                'shipFirstname' => $request->shipfirstname,
                'shipLastname' => $request->shiplastname,
                'shipCompany' => $request->shipcompany,
                'shipCountry' => $request->shipcountry,
                'shipEmail' => $request->shipusr_email,
                'shipPhone' => $request->shipphone,
                'shipWhatsapp' => $request->shipwhatsapp_number,
                'shipAddress' => $request->shipaddress,
                'createDate' => $created_date,
            ]);

            DB::table('c_user_registrations_dates')->insert([
                'order_id' => $order_id,
                'registration_name' => $request->conference_name,
                'user_id' => $userId,
                'curency_type' => $request->currencyId,
                'total_amount' => $request->totalAmount,
                'payment_meathod' => $request->paymentMethod,
                'reg_date' => $created_date,
                'created_date' => $created_date,
            ]);

            // Send email notification
            $to = DB::table('web_cms')->where('page_type', 10)->where('status', 1)->value('short_desc');
            $from = $request->usr_email;
            $subject = 'User Registration Details';
            $message = '<table border="1" style="text-align:center;">
                <tr><td style="padding:15px;">Name:</td><td style="padding:15px;">' . $request->firstname . '</td></tr>
                <tr><td style="padding:15px;">Email:</td><td style="padding:15px;">' . $request->usr_email . '</td></tr>
                <tr><td style="padding:15px;">Mobile:</td><td style="padding:15px;">' . $request->user_phone . '</td></tr>
                <tr><td style="padding:15px;">University:</td><td style="padding:15px;">' . $request->company . '</td></tr>
                <tr><td style="padding:15px;">Category Type:</td><td style="padding:15px;">' . $cat_type . '</td></tr>
                <tr><td style="padding:15px;">Conference:</td><td style="padding:15px;">' . $request->conference_name . '</td></tr>
                <tr><td style="padding:15px;">Total Amount:</td><td style="padding:15px;">' . $request->totalAmount . '</td></tr>
                <tr><td style="padding:15px;">Registration Date:</td><td style="padding:15px;">' . $created_date . '</td></tr>
            </table>';
            // Implement email sending logic here (e.g., using Laravel Mail)

            DB::commit();
            return response()->json(['success' => true, 'order_id' => $order_id]);
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Registration failed: ' . $e->getMessage());
            return response()->json(['success' => false, 'message' => 'Failed to register'], 500);
        }
    }

    public function createPaymentIntent(Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric|min:50',
            'currency' => 'required|string|in:usd,eur,gbp',
            'user.name' => 'required|string',
            'user.email' => 'required|email',
            'conference_id' => 'required|integer',
            'order_id' => 'required|string',
        ]);

        Stripe::setApiKey(env('STRIPE_SECRET_KEY'));

        $conferenceInfo = DB::table('conferences')->where('id', $request->conference_id)->first();
        if (!$conferenceInfo) {
            return response()->json(['error' => 'Conference not found'], 404);
        }

        $conferenceName = trim("{$conferenceInfo->conference_first_title} {$conferenceInfo->conference_name}");
        $description = "Registration: {$conferenceName}";

        try {
            $intent = PaymentIntent::create([
                'amount' => $request->amount,
                'currency' => $request->currency,
                'description' => $description,
                'metadata' => [
                    'order_id' => $request->order_id,
                    'user_name' => $request->user['name'],
                    'user_email' => $request->user['email'],
                    'conference_id' => $request->conference_id,
                    'conference_name' => $conferenceName,
                ],
                'payment_method_types' => ['card'],
            ]);

            return response()->json(['clientSecret' => $intent->client_secret]);
        } catch (\Exception $e) {
            \Log::error('Stripe PaymentIntent failed: ' . $e->getMessage());
            return response()->json(['error' => 'Payment setup failed'], 500);
        }
    }

    public function updatePaymentSuccess(Request $request, $order_id)
    {
        $request->validate([
            'payment_intent' => 'required|string',
            'status' => 'required|in:paid,cancelled',
        ]);

        $updated = DB::table('c_user_registrations_dates')
            ->where('order_id', $order_id)
            ->update([
                'payment_status' => $request->status,
                'payment_intent_id' => $request->payment_intent,
                'updated_at' => now(),
            ]);

        $users = DB::table('c_user_registrations_dates')->where('order_id', $order_id)->first();
        DB::table('c_reg_users')
            ->where('id', $users->user_id)
            ->update([
                'payment_status' => $request->status,
                'transaction_id' => $order_id,
            ]);

        if (!$updated) {
            return response()->json(['success' => false, 'message' => 'Order not found'], 404);
        }

        return response()->json(['success' => true]);
    }

    public function getSuccessData($order_id)
    {
        $data = DB::table('c_reg_users as u')
            ->join('c_user_registrations_dates as r', 'u.id', '=', 'r.user_id')
            ->where('r.order_id', $order_id)
            ->select(
                'u.*',
                'r.order_id',
                'r.payment_status',
                'r.total_amount',
                'r.created_date',
                'r.payment_meathod',
                'r.curency_type as currency_id'
            )
            ->first();

        if (!$data) {
            return response()->json(['success' => false, 'message' => 'Order not found'], 404);
        }

        $currency = match ((int)($data->currency_id ?? 3)) {
            1 => 'GBP',
            2 => 'USD',
            3 => 'EUR',
            default => 'EUR',
        };

        return response()->json([
            'success' => true,
            'data' => array_merge((array) $data, ['currency' => $currency]),
        ]);
    }

    public function getPhoneCode(Request $request)
    {
        $request->validate([
            'country_name' => 'required|string',
        ]);

        $country = DB::table('countries')
            ->where('country_name', $request->country_name)
            ->where('status', 1)
            ->first();

        if ($country && $country->country_code) {
            return response()->json([
                'success' => true,
                'phonecode' => '+' . $country->country_code,
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => 'Country code not found',
        ], 404);
    }
}