<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Models\Registration;
use App\Models\UserRegistrationDate; // ✅ This is where order_id and payment_status live
use Stripe\Checkout\Session;
use Stripe\Stripe;
use Log;

class StripeLatestController extends Controller
{
  public function latestcreateCheckout(Request $request): \Illuminate\Http\RedirectResponse
{
    Log::info('=== STRIPE CHECKOUT REQUEST RECEIVED ===');
    Log::info('Request Data:', $request->all());

    try {
        $request->validate([
            'order_id' => 'required|string|max:255',
            'amount' => 'required|integer|min:50',
            'currency_id' => 'required|integer|in:1,2,3',
            'user.name' => 'required|string',
            'user.email' => 'required|email',
        ]);

        Log::info('Validation passed.');

        $stripeSecret = env('STRIPE_SECRET_KEY');
        if (!$stripeSecret) {
            Log::error('STRIPE_SECRET_KEY is not set in .env');
            return redirect()->back()->with('error', 'Server misconfiguration.');
        }

        Stripe::setApiKey($stripeSecret);

        $frontendUrl = env('VITE_FRONTEND_URL');
        if (!$frontendUrl) {
            Log::error('VITE_FRONTEND_URL is not set in .env');
            return redirect()->back()->with('error', 'Server misconfiguration.');
        }

        // Convert currency_id to code for Stripe
        $currencyCode = match((int) $request->currency_id) {
            1 => 'gbp',
            2 => 'usd',
            3 => 'eur',
            default => 'eur'
        };
        
        // Get conference_url from request (sent by Vue form)
$conferenceUrl = $request->input('conference_url', 'conference'); // fallback to 'conference'

// Build dynamic success URL
//$successUrl = rtrim($frontendUrl, '/') . '/' . trim($conferenceUrl, '/') . '/latest-stripe-success?session_id={CHECKOUT_SESSION_ID}';
$successUrl = rtrim($frontendUrl, '/') .'/latest-stripe-success?session_id={CHECKOUT_SESSION_ID}';

        $session = Session::create([
            'payment_method_types' => ['card'],
            'line_items' => [[
                'price_data' => [
                    'currency' => $currencyCode,
                    'product_data' => [
                        'name' => 'Conference Registration',
                    ],
                    'unit_amount' => $request->amount,
                ],
                'quantity' => 1,
            ]],
            'mode' => 'payment',
            'success_url' => $successUrl,
            'cancel_url' => $frontendUrl . '/registration?cancelled=true',
            'customer_email' => $request->user['email'],
            'metadata' => [
                'order_id' => $request->order_id,
                'user_name' => $request->user['name'],
            ],
        ]);

        Log::info('Stripe session created successfully.', [
            'session_id' => $session->id,
            'checkout_url' => $session->url,
        ]);

        // Save to DB
        $updated = UserRegistrationDate::where('order_id', $request->order_id)->update([
            'stripe_session_id' => $session->id,
            'curency_type' => (int) $request->currency_id,
            'total_amount' => $request->amount / 100,
        ]);

        if (!$updated) {
            Log::warning("No UserRegistrationDate record found for order_id: " . $request->order_id);
        }

        // ✅ THIS IS THE KEY: Redirect the browser to Stripe
        return redirect($session->url);

    } catch (\Exception $e) {
        Log::error('🚨 STRIPE CHECKOUT FAILED 🚨', [
            'message' => $e->getMessage(),
            'file' => $e->getFile(),
            'line' => $e->getLine(),
        ]);

        return redirect()->back()->with('error', 'Payment setup failed. Please try again.');
    }
}

    public function stripeSuccess($sessionId): JsonResponse
    {
        Log::info('=== STRIPE SUCCESS CALLED ===', ['session_id' => $sessionId]);

        Stripe::setApiKey(env('STRIPE_SECRET_KEY'));

        try {
            $session = Session::retrieve($sessionId);

            if ($session->payment_status !== 'paid') {
                return response()->json([
                    'success' => false,
                    'message' => 'Payment not completed.'
                ]);
            }

            // ✅ Find order by stripe_session_id in user_registration_dates
            $userRegistrationDate = UserRegistrationDate::where('stripe_session_id', $sessionId)->first();

            if (!$userRegistrationDate) {
                Log::error("No record found for stripe_session_id: $sessionId");
                return response()->json([
                    'success' => false,
                    'message' => 'Order not found.'
                ]);
            }

            // ✅ Load user/registration data
            $registration = Registration::find($userRegistrationDate->user_id);

            if (!$registration) {
                Log::error("No registration found for user_id: " . $userRegistrationDate->user_id);
                return response()->json([
                    'success' => false,
                    'message' => 'User data not found.'
                ]);
            }

            // ✅ Update payment status
            $userRegistrationDate->payment_status = 'paid';
            $userRegistrationDate->save();

            $registration->payment_status = 'paid';
            $registration->transaction_id = $userRegistrationDate->order_id; // or $sessionId if preferred
            $registration->save();

            // ✅ Return data to Vue success page
            return response()->json([
                'success' => true,
                'data' => [
                    'firstname'      => $registration->firstname,
                    'lastname'       => $registration->lastname,
                    'usr_email'      => $registration->usr_email,
                    'total_amount'   => $userRegistrationDate->total_amount,
                    'currency'       => $this->getCurrencyCode($userRegistrationDate->curency_type),
                    'session_id'     => $session->id,
                    'order_id'       => $userRegistrationDate->order_id,
                    'payment_status' => $userRegistrationDate->payment_status,
                    'invoiceid'      => $registration->invoiceid ?? '',
                    'conference_name'=> 'Conference', // You can join Conference model if needed
                ]
            ]);

        } catch (\Exception $e) {
            Log::error('Stripe Success Lookup Failed', [
                'session_id' => $sessionId,
                'error' => $e->getMessage()
            ]);
            return response()->json(['success' => false, 'message' => 'Server error'], 500);
        }
    }

    private function getCurrencyCode($id)
    {
        return match((int)$id) {
            1 => 'GBP',
            2 => 'USD',
            3 => 'EUR',
            default => 'EUR'
        };
    }
    
    // === V7 STRIPE CHECKOUT HANDLERS ===
  public function latestcreateCheckoutV7(Request $request): \Illuminate\Http\RedirectResponse
{
    Log::info('=== STRIPE CHECKOUT REQUEST RECEIVED ===');
    Log::info('Request Data:', $request->all());

    try {
        $request->validate([
            'amount' => 'required|integer|min:50',
            'currency_id' => 'required|integer|in:1,2,3',
            'user.name' => 'required|string',
            'user.email' => 'required|email',
        ]);

        Log::info('Validation passed.');

     $stripeSecret = env('PAYMENT_SANDBOX_V7')
    ? env('STRIPE_SECRET_TEST_V7')
    : env('STRIPE_SECRET_LIVE_V7');

        if (!$stripeSecret) {
            Log::error('STRIPE_SECRET_KEY is not set in .env');
         //   return redirect()->back()->with('error', 'Server misconfiguration.');
        }

        Stripe::setApiKey($stripeSecret);

        $frontendUrl = env('VITE_FRONTEND_URL');
        if (!$frontendUrl) {
            Log::error('VITE_FRONTEND_URL is not set in .env');
           // return redirect()->back()->with('error', 'Server misconfiguration.');
        }

        // Convert currency_id to code for Stripe
        $currencyCode = match((int) $request->currency_id) {
            1 => 'gbp',
            2 => 'usd',
            3 => 'eur',
            default => 'eur'
        };
        
        // Get conference_url from request (sent by Vue form)
$conferenceUrl = $request->input('conference_url', 'conference'); // fallback to 'conference'

// Build dynamic success URL
//$successUrl = rtrim($frontendUrl, '/') . '/' . trim($conferenceUrl, '/') . '/latest-stripe-success?session_id={CHECKOUT_SESSION_ID}';
$successUrl = rtrim($frontendUrl, '/') .'/latest-stripe-success?session_id={CHECKOUT_SESSION_ID}';

        $session = Session::create([
            'payment_method_types' => ['card'],
            'line_items' => [[
                'price_data' => [
                    'currency' => $currencyCode,
                    'product_data' => [
                        'name' => 'Conference Registration',
                    ],
                    'unit_amount' => $request->amount,
                ],
                'quantity' => 1,
            ]],
            'mode' => 'payment',
            'success_url' => $successUrl,
            'cancel_url' => $frontendUrl . '/registration?cancelled=true',
            'customer_email' => $request->user['email'],
            'metadata' => [
                'order_id' => $request->order_id,
                'user_name' => $request->user['name'],
            ],
        ]);

  
      $query = UserRegistrationDate::where('order_id', $request->order_id);

$updated = $query->update([
    'stripe_session_id' => $session->id,
    'curency_type' => (int) $request->currency_id,
    'total_amount' => $request->amount / 100,
]);

        if (!$updated) {
            Log::warning("No UserRegistrationDate record found for order_id: " . $request->order_id);
        }

        // ✅ THIS IS THE KEY: Redirect the browser to Stripe
        return redirect($session->url);

    } catch (\Exception $e) {
        Log::error('🚨 STRIPE CHECKOUT FAILED 🚨', [
            'message' => $e->getMessage(),
            'file' => $e->getFile(),
            'line' => $e->getLine(),
        ]);

        return redirect()->back()->with('error', 'Payment setup failed. Please try again.');
    }
}

public function stripeSuccessV7($sessionId): JsonResponse
{
    return $this->stripeSuccess($sessionId);
}

}