<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Conference;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Stripe\Checkout\Session;
use Stripe\Stripe;

class StripeController extends Controller
{
    public function createCheckout(Request $request): JsonResponse
    {
        $request->validate([
            'order_id' => 'required',
            'amount' => 'required|integer|min:50', // min 50 cents
            'currency' => 'required|in:usd,eur,gbp',
            'user.name' => 'required',
            'user.email' => 'required|email',
            'conference_id' => 'required|integer|exists:conferences,id',
        ]);

        Stripe::setApiKey(env('STRIPE_SECRET'));

        try {
            $conference = Conference::find($request->conference_id);
            $conferenceName = $conference ? $conference->short_name : 'Conference';

            $session = Session::create([
                'payment_method_types' => ['card'],
                'line_items' => [[
                    'price_data' => [
                        'currency' => $request->currency,
                        'product_data' => [
                            'name' => 'Conference Registration',
                            'description' => 'Registration for ' . $conferenceName,
                        ],
                        'unit_amount' => $request->amount,
                    ],
                    'quantity' => 1,
                ]],
                'mode' => 'payment',
                'success_url' => env('APP_URL') . '/payment-success?session_id={CHECKOUT_SESSION_ID}',
                'cancel_url' => env('APP_URL') . '/payment-cancelled',
                'customer_email' => $request->user['email'],
                'metadata' => [
                    'order_id' => $request->order_id,
                    'user_name' => $request->user['name'],
                    'conference_id' => $request->conference_id,
                    'user_email' => $request->user['email'],
                ],
            ]);

            // Optional: Log session to database
            // \App\Models\StripeSession::create([
            //     'session_id' => $session->id,
            //     'order_id' => $request->order_id,
            //     'conference_id' => $request->conference_id,
            //     'user_name' => $request->user['name'],
            //     'user_email' => $request->user['email'],
            //     'amount' => $request->amount,
            //     'currency' => $request->currency,
            //     'status' => 'created',
            //     'created_at' => now(),
            // ]);

            return response()->json(['url' => $session->url]);
        } catch (\Exception $e) {
            \Log::error('Stripe session creation failed: ' . $e->getMessage());
            return response()->json(['error' => 'Payment setup failed'], 500);
        }
    }
}