<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;   // ✅ import the base controller
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class PaypalApiController extends Controller
{
    /**
     * Helper: resolve user_id from our order_id
     */
    private function userIdForOrder(string $orderId): ?int
    {
        return DB::table('c_user_registrations_dates')
            ->where('order_id', $orderId)
            ->value('user_id');
    }

    /**
     * PayPal "return" URL (customer browser). Do NOT fully trust for final status.
     * We show the success page and mark method, but we let IPN set txn/status.
     */
    public function returnFromPayPal(Request $request)
    {
        // Prefer 'custom' (we posted our order_id in it). Fall back to item_number1 or query.
        $orderId = $request->input('custom')
            ?: $request->input('item_number1')
            ?: $request->query('order_id');

        if (!$orderId) {
            return redirect('/registration?error=missing_order');
        }

        $userId = $this->userIdForOrder($orderId);
        if (!$userId) {
            return redirect('/registration?error=unknown_order');
        }

        // Mark payment method and keep status as-is (IPN will finalize)
        DB::table('c_user_registrations_dates')
            ->where('order_id', $orderId)
            ->update([
                'payment_meathod' => 2,           // 2 = PayPal
            
            ]);

        // If PayPal sent any txn id in the return (often it doesn't), stash it
      

        // Go to SPA success page that reads by order_id
        return redirect('/paypalpaymentsuccess?order_id=' . urlencode($orderId));
    }

    /**
     * PayPal IPN (server-to-server). This is where we trust and finalize payment.
     * Make sure this route is CSRF-exempt and publicly reachable.
     */
    public function handleIPN(Request $request)
    {
        Log::info('PayPal IPN', $request->all());

        $orderId = $request->input('custom')
            ?: $request->input('item_number1')
            ?: $request->input('invoice');

        if (!$orderId) {
            return response('No order id', 400);
        }

        $status    = $request->input('payment_status');           // e.g., Completed
        $txnId     = $request->input('txn_id');                   // PayPal transaction id
        $gross     = $request->input('mc_gross') ?: $request->input('mc_gross_1');
        $currency  = $request->input('mc_currency');

        $userId = $this->userIdForOrder($orderId);
        if (!$userId) {
            return response('Unknown order', 404);
        }

        if ($status === 'Completed') {
            DB::beginTransaction();

            // Save definitive status + txn on the registration row keyed by order_id
            DB::table('c_user_registrations_dates')
                ->where('order_id', $orderId)
                ->update([
                    'payment_status'  => 'Completed',
                    'payment_meathod' => 2,          // PayPal
                    ]);

            // Mirror a simple status on the user row; also set invoice if empty
            DB::table('c_reg_users')
                ->where('id', $userId)
                ->update([
                    'payment_status' => 'Completed',
                    'transaction_id' => $txnId,
                ]);

            $invoice = DB::table('c_reg_users')->where('id', $userId)->value('invoiceid');
            if (!$invoice) {
                DB::table('c_reg_users')
                    ->where('id', $userId)
                    ->update(['invoiceid' => 'INV-' . $orderId]);
            }

            // Optional: store an orders row
            DB::table('c_user_orders')->updateOrInsert(
                ['order_id' => $orderId],
                [
                    'user_id'       => $userId,
                    'tracking_id'   => $txnId,
                    'payment_status'=> 'Completed',
                    'user_type'     => 2,
                    'status'        => 1,
                    'createdDate'   => now(),
                ]
            );

            DB::commit();
        }

        return response('OK', 200);
    }

    /**
     * JSON for your Vue success page (reads by order_id)
     */
    public function getSuccessData(string $orderId)
    {
        $rec = DB::table('c_reg_users as u')
            ->join('c_user_registrations_dates as r', 'u.id', '=', 'r.user_id')
            ->where('r.order_id', $orderId)
            ->select(
                'u.firstname',
                'u.lastname',
                'u.usr_email',
                'u.user_phone',
                'u.invoiceid',
                'r.order_id',
                'r.total_amount',
                'r.curency_type as currency_id',
                'r.payment_meathod as payment_method',
                'r.payment_status',
                'u.transaction_id',
                'r.created_date'
            )
            ->first();

        if (!$rec) {
            return response()->json(['success' => false, 'message' => 'Order not found'], 404);
        }

        $currency = match ((int)($rec->currency_id ?? 3)) {
            1 => 'GBP',
            2 => 'USD',
            3 => 'EUR',
            default => 'EUR',
        };

        return response()->json([
            'success' => true,
            'data' => [
                'order_id'       => $rec->order_id,
                'usr_email'      => $rec->usr_email,
                'user_phone'     => $rec->user_phone,
                'total_amount'   => $rec->total_amount,
                'currency'       => $currency,
                'payment_method' => (int) $rec->payment_method,
                'payment_status' => $rec->payment_status,
                'transaction_id' => $rec->transaction_id,
                'invoiceid'      => $rec->invoiceid,
                'created_date'   => $rec->created_date,
            ],
        ]);
    }
        public function getSuccessDatav4(string $orderId)
    {
        $rec = DB::table('c_reg_users as u')
            ->join('c_user_registrations_dates as r', 'u.id', '=', 'r.user_id')
            ->where('r.order_id', $orderId)
            ->select(
                'u.firstname',
                'u.lastname',
                'u.usr_email',
                'u.user_phone',
                'u.invoiceid',
                'r.order_id',
                'r.total_amount',
                'r.curency_type as currency_id',
                'r.payment_meathod as payment_method',
                'r.payment_status',
                'u.transaction_id',
                'r.created_date'
            )
            ->first();

        if (!$rec) {
            return response()->json(['success' => false, 'message' => 'Order not found'], 404);
        }

        $currency = match ((int)($rec->currency_id ?? 3)) {
            1 => 'GBP',
            2 => 'USD',
            3 => 'EUR',
            default => 'EUR',
        };

        return response()->json([
            'success' => true,
            'data' => [
                'order_id'       => $rec->order_id,
                'usr_email'      => $rec->usr_email,
                'user_phone'     => $rec->user_phone,
                'total_amount'   => $rec->total_amount,
                'currency'       => $currency,
                'payment_method' => (int) $rec->payment_method,
                'payment_status' => $rec->payment_status,
                'transaction_id' => $rec->transaction_id,
                'invoiceid'      => $rec->invoiceid,
                'created_date'   => $rec->created_date,
            ],
        ]);
    }
    
    // In app/Http/Controllers/Api/PaypalApiController.php

public function handleIPNV4(Request $request)
{
    \Log::info('PayPal v4 IPN received', $request->all());

    $orderId = $request->input('custom')
        ?: $request->input('item_number1')
        ?: $request->input('invoice');

    if (!$orderId) {
        return response('No order id', 400);
    }

    $status    = $request->input('payment_status');           // e.g., Completed
    $txnId     = $request->input('txn_id');                   // PayPal transaction id
    $gross     = $request->input('mc_gross') ?: $request->input('mc_gross_1');
    $currency  = $request->input('mc_currency');
    $payerId   = $request->input('payer_id');

    $userId = $this->userIdForOrder($orderId);
    if (!$userId) {
        return response('Unknown order', 404);
    }

    if ($status === 'Completed') {
        DB::beginTransaction();

        // Update registration
        DB::table('c_user_registrations_dates')
            ->where('order_id', $orderId)
            ->update([
                'payment_status'  => 'Completed',
                'payment_meathod' => 2,
            ]);

        // Update user
        DB::table('c_reg_users')
            ->where('id', $userId)
            ->update([
                'payment_status' => 'Completed',
                'transaction_id' => $txnId,
                'payer_id'       => $payerId, // ← Optional but useful
            ]);

        DB::commit();
    }

    return response('OK', 200);
}

public function returnFromPayPalvfour(Request $request)
{
    \Log::info('PayPal v4 return received', $request->all());

    $orderId = $request->input('custom')
        ?: $request->input('item_number1')
        ?: $request->query('order_id');

    if (!$orderId) {
        return redirect('/registration-v4?error=missing_order');
    }

    $userId = $this->userIdForOrder($orderId);
    if (!$userId) {
        return redirect('/registration-v4?error=unknown_order');
    }

    // Get payment status from PayPal
    $paymentStatus = $request->input('payment_status');
    $txnId = $request->input('txn_id');
    $payerId = $request->input('payer_id');
    $mcGross = $request->input('mc_gross');
    $mcCurrency = $request->input('mc_currency');

    DB::beginTransaction();

    try {
        // Update registration
        DB::table('c_user_registrations_dates')
            ->where('order_id', $orderId)
            ->update([
                'payment_meathod' => 2,
                'payment_status' => $paymentStatus === 'Completed' ? 'Completed' : 'Pending',
            ]);

        // Update user
        DB::table('c_reg_users')
            ->where('id', $userId)
            ->update([
                'payment_status' => $paymentStatus === 'Completed' ? 'Completed' : 'Pending',
                'transaction_id' => $txnId,
                'payer_id' => $payerId,
                'total_amount' => $mcGross, // optional
                'currency' => $mcCurrency, // optional
            ]);

        DB::commit();

        // Redirect to Vue.js success page
        return redirect('/payment-success-v4?order_id=' . urlencode($orderId));

    } catch (\Exception $e) {
        DB::rollBack();
        \Log::error('PayPal v4 return update failed', ['error' => $e->getMessage()]);
        return redirect('/registration-v4?error=update_failed');
    }
}
public function handleIPNV5(Request $request)
{
    \Log::info('PayPal v5 IPN (group)', $request->all());

    $orderId = $request->input('custom') ?: $request->input('item_number1') ?: $request->input('invoice');
    if(!$orderId) return response('No order id', 400);

    $status   = $request->input('payment_status');
    $txnId    = $request->input('txn_id');
    $payerId  = $request->input('payer_id');

    $userId = DB::table('c_user_group_registrations_dates')->where('order_id',$orderId)->value('user_id');
    if(!$userId) return response('Unknown order', 404);

    if ($status === 'Completed') {
        DB::beginTransaction();
        DB::table('c_user_group_registrations_dates')->where('order_id',$orderId)->update([
            'payment_status'=>'Completed',
            'payment_meathod'=>2,
        ]);
        DB::table('c_reg_users')->where('id',$userId)->update([
            'payment_status'=>'Completed',
            'transaction_id'=>$txnId,
            'payer_id'=>$payerId,
        ]);
        DB::commit();
    }
    return response('OK', 200);
}

public function paypalIpnV5(Request $request)
{
    // 1) Capture raw payload
    $payload = $request->all();
    \Log::info('PayPal IPN hit', $payload);

    // 2) Choose endpoint based on env
    $sandbox = filter_var(env('PAYPAL_SANDBOX', true), FILTER_VALIDATE_BOOLEAN);
    $verifyUrl = $sandbox
        ? 'https://ipnpb.sandbox.paypal.com/cgi-bin/webscr'
        : 'https://ipnpb.paypal.com/cgi-bin/webscr';

    // 3) Validate with PayPal (cmd=_notify-validate)
    $verifyBody = 'cmd=_notify-validate&' . http_build_query($payload);
    $ch = curl_init($verifyUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST           => true,
        CURLOPT_POSTFIELDS     => $verifyBody,
        CURLOPT_HTTPHEADER     => ['Content-Type: application/x-www-form-urlencoded', 'Connection: Close'],
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_SSL_VERIFYHOST => 2,
    ]);
    $response = curl_exec($ch);
    $curlErr  = curl_error($ch);
    curl_close($ch);

    if ($response !== 'VERIFIED') {
        \Log::warning('PayPal IPN not VERIFIED', ['resp'=>$response, 'err'=>$curlErr]);
        return response('INVALID', 400);
    }

    // 4) Basic fraud checks (strongly recommended)
    $receiverEmail = $payload['receiver_email'] ?? $payload['business'] ?? '';
    $expectedEmail = $sandbox ? env('PAYPAL_BUSINESS_SANDBOX','') : env('PAYPAL_BUSINESS_LIVE','');
    if ($expectedEmail && strcasecmp(trim($receiverEmail), trim($expectedEmail)) !== 0) {
        \Log::warning('IPN receiver_email mismatch', compact('receiverEmail','expectedEmail'));
        return response('BAD_RECEIVER', 400);
    }

    $paymentStatus = $payload['payment_status'] ?? '';       // e.g. Completed, Pending
    $txnId         = $payload['txn_id'] ?? '';
    $mcGross       = (float)($payload['mc_gross'] ?? 0);
    $mcCurrency    = $payload['mc_currency'] ?? '';
    $orderId       = $payload['custom'] ?? ($payload['item_number_1'] ?? $payload['item_number'] ?? '');

    if (!$orderId) {
        \Log::warning('IPN missing order_id', $payload);
        return response('NO_ORDER', 400);
    }

    // 5) Look up the order
    $row = DB::table('c_user_group_registrations_dates')->where('order_id', $orderId)->first();
    if (!$row) {
        \Log::warning('IPN order not found', compact('orderId'));
        return response('NO_MATCH', 404);
    }

    // (Optional) amount/currency checks
    // Your UI adds +3% for PayPal. If your DB "total_amount" stores the base total,
    // you may want to compare mc_gross ≈ total_amount * 1.03, with tiny tolerance:
    // $expected = round(($row->total_amount ?? 0) * 1.03, 2);
    // if ($paymentStatus === 'Completed' && abs($mcGross - $expected) > 0.05) { ... }

    // 6) Update status
    $newStatus = ($paymentStatus === 'Completed') ? 'Completed' : 'Pending';

    DB::beginTransaction();
    try {
        DB::table('c_user_group_registrations_dates')
            ->where('order_id', $orderId)
            ->update([
                'payment_status'  => $newStatus,
                'payment_meathod' => 2,            // PayPal
                'transaction_id'  => $txnId,       // store PayPal txn id here if you have this column
            ]);

        // mirror to user table like you do for Stripe
        $row = DB::table('c_user_group_registrations_dates')->where('order_id',$orderId)->first();
        if ($row) {
            DB::table('c_reg_users')->where('id',$row->user_id)->update([
                'payment_status'  => $newStatus,
                'transaction_id'  => $txnId ?: $orderId, // fall back to order id
            ]);
        }

        DB::commit();
    } catch (\Throwable $e) {
        DB::rollBack();
        \Log::error('IPN DB update failed: '.$e->getMessage());
        return response('DB_ERR', 500);
    }

    return response('OK', 200);
}


}
