<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Conference;
use App\Models\Country;
use App\Models\Registration;
use App\Models\AbstractCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\JsonResponse;

class AbstractSubmissionApiController extends Controller
{
    public function show($id): JsonResponse
    {
        $conference = Conference::with([
            'brochures' => fn($q) => $q->where('status', 1)->where('page_type', 2)->orderBy('id'),
            'sessionTracks',
            'headerContent',
            'navigations',
            'socialLinks',
            'address',
            'AbstractmetaData'
        ])->find($id);

        if (!$conference) {
            return response()->json(['error' => 'Conference not found'], 404);
        }

        $meta = $conference->AbstractmetaData;
        $abstractCategories = AbstractCategory::where('status', 1)->orderBy('id')->get();
        $countries = Country::where('status', 1)->orderBy('country_name')->get();

        return response()->json([
            'success' => true,
            'conferenceInfo' => $conference,
            'headerData' => $this->getHeaderData($conference),
            'footerData' => $this->getFooterData($conference),
            'sampleabstract' => $conference->brochures->first(),
            'sessionTracks' => $conference->sessionTracks,
            'abstractCategories' => $abstractCategories,
            'countriesList' => $countries,
            'meta' => [
                'title' => $meta?->meta_title ?? $meta?->title ?? '',
                'description' => $meta?->meta_description ?? '',
                'keywords' => $meta?->meta_keywords ?? ''
            ],
            'favicon' => $conference->fav_icon ?? ''
        ]);
    }

    private function getHeaderData(Conference $conference)
    {
        $headerNav = $conference->navigations
            ->where('page_type', 1)
            ->where('status', 1)
            ->where('parent_id', 0)
            ->sortBy('orderpos')
            ->map(function ($item) use ($conference) {
                $item->headerSubNav = $conference->navigations
                    ->where('parent_id', $item->id)
                    ->where('status', 1)
                    ->sortBy('orderpos');
                return $item;
            });

        return [
            'conferenceInfo' => $conference,
            'headercontent' => $conference->headerContent,
            'headerNavigation' => $headerNav,
            'socialLinks' => $conference->socialLinks,
        ];
    }

    private function getFooterData(Conference $conference)
    {
        return [
            'quickLinks' => $conference->navigations
                ->where('page_type', 2)
                ->where('status', 1)
                ->sortBy('orderpos'),
            'usefulLinks' => $conference->navigations
                ->where('page_type', 3)
                ->where('status', 1)
                ->sortBy('orderpos'),
            'socialLinks' => $conference->socialLinks,
            'mainaddress' => \DB::table('web_cms')->where('page_type', 10)->where('status', 1)->first(),
            'contactInfo' => $conference->address,
            'whatsappNumber' => $conference->address?->whatsapp_number,
        ];
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'conference_id' => 'required|integer|exists:conferences,id',
            'designation' => 'required|string|max:100',
            'abstractcategory' => 'required|string|max:255',
            'abstract_title' => 'required|string|max:255',
            'firstname' => 'required|string|max:100',
            'company' => 'required|string|max:255',
            'country' => 'required|string|max:100',
            'usr_email' => 'required|email',
            'user_phone' => 'required|string|max:50',
            'abstract_message' => 'nullable|string',
            'session_id' => 'required|string|max:255',
            'user_file' => 'nullable',

        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        if ((int)$request->captcha_answer !== (int)$request->captcha_expected) {
            return response()->json(['error' => 'Invalid captcha'], 422);
        }

        $conference = Conference::findOrFail($request->conference_id);

if ($request->hasFile('user_file')) {
    $file = $request->file('user_file');
    $docfile = time() . '.' . $file->getClientOriginalExtension();

    // Absolute path like in your old code
    $directory = "/home/scisynopsisconfe/public_html/uploads/conferences/{$conference->folder_name}_{$conference->id}";

    if (!file_exists($directory)) {
        mkdir($directory, 0777, true);
    }

    // Move uploaded file
    $file->move($directory, $docfile);

    // Public URL
    $fileUrl = "https://scisynopsisconferences.com/uploads/conferences/{$conference->folder_name}_{$conference->id}/{$docfile}";
}




        Registration::create([
            'unique_id' => now()->format('smdiYH'),
            'conference_id' => $conference->id,
            'designation' => $request->designation,
            'absCategory' => $request->abstractcategory,
            'abstract_title' => $request->abstract_title,
            'firstname' => $request->firstname,
            'lastname' => $request->lastname,
            'country' => $request->country,
            'company' => $request->company,
            'usr_email' => strtolower($request->usr_email),
            'user_phone' => $request->user_phone,
            'usr_whatsapp' => $request->whatsapp_no,
            'sessions_id' => $request->session_id,
            'user_type' => 2,
            'coupon_value' => 0, 
               'invoiceid'     => '', 
               'aditional_details' => '',
               'extra_info' => '',
               'total_extra_amount' => '',
               'transaction_id' => '',
            'user_file' => $docfile,
            'queries' => $request->abstract_message,
            'created_date' => now(),
        ]);

        return response()->json([
            'message' => 'Thank you for submitting your abstract. We will get back to you soon.'
        ]);
    }

    public function getPhonecode($countryName): JsonResponse
    {
        $countryCode = Country::where('country_name', $countryName)->value('country_code');
        return response()->json($countryCode ?: '');
    }
}