<?php

namespace App\Http\Controllers\Api;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Stripe\Checkout\Session;
use Stripe\Stripe;

class StripeController extends Controller
{
    public function createCheckout(Request $request)
    {
        // Validate request
        $request->validate([
            'order_id' => 'required',
            'amount' => 'required|integer|min:50', // min 50 cents
            'currency' => 'required|in:usd,eur,gbp',
            'user.name' => 'required',
            'user.email' => 'required|email',
        ]);

        // Set Stripe API key
        Stripe::setApiKey(env('STRIPE_SECRET'));

        try {
            // Create Checkout Session
            $session = Session::create([
                'payment_method_types' => ['card'],
                'line_items' => [[
                    'price_data' => [
                        'currency' => $request->currency,
                        'product_data' => [
                            'name' => 'Conference Registration',
                            'description' => 'Registration for ' . $this->getConferenceName($request->conference_id),
                        ],
                        'unit_amount' => $request->amount,
                    ],
                    'quantity' => 1,
                ]],
                'mode' => 'payment',
                'success_url' => env('APP_URL') . '/payment-success?session_id={CHECKOUT_SESSION_ID}',
                'cancel_url' => env('APP_URL') . '/payment-cancelled',
                'customer_email' => $request->user['email'],
                'metadata' => [
                    'order_id' => $request->order_id,
                    'user_name' => $request->user['name'],
                    'conference_id' => $request->conference_id,
                    'user_email' => $request->user['email'],
                ],
            ]);

            return response()->json(['url' => $session->url]);
        } catch (\Exception $e) {
            \Log::error('Stripe session creation failed: ' . $e->getMessage());
            return response()->json(['error' => 'Payment setup failed'], 500);
        }
    }

    private function getConferenceName($id)
    {
        $conference = DB::table('conferences')->where('id', $id)->first();
        return $conference ? $conference->short_name : 'Conference';
    }
}