<?php

namespace App\Http\Controllers\Api;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;
use App\Http\Controllers\Controller;

class BrochureApiController extends Controller
{
    /**
     * GET /api/brochure/{conferenceId}
     * Returns all data the download page needs, including countries.
     */
    public function show($id)
    {
        $conference = DB::table('conferences')->find($id);
        if (!$conference) {
            return response()->json(['error' => 'Conference not found'], 404);
        }

        $meta = DB::table('c_meta_data')
            ->where(['conference_id' => $id, 'page_type' => 3])
            ->first();

        // Countries: ensure we return fields the UI expects
        $countries = DB::table('countries')
            ->select('id', 'country_name', 'country_code') // phonecode column assumed
            ->where('status', 1)
            ->orderBy('country_name', 'ASC')
            ->get();

        return response()->json([
            'success' => true,
            'conferenceInfo' => $conference,
            'headerData' => $this->getHeaderData($id),
            'footerData' => $this->getFooterData($id),
            // Provide a consistent "commonData" object so the UI can read easily
            'commonData' => [
                'countriesList' => $countries,
            ],
            'meta' => [
                'title'       => $meta->meta_title ?? $meta->title ?? '',
                'description' => $meta->meta_description ?? '',
                'keywords'    => $meta->meta_keywords ?? ''
            ],
            'favicon' => $conference->fav_icon ?? ''
        ]);
    }

    private function getHeaderData($id)
    {
        $conf = DB::table('conferences')->find($id);

        $nav = DB::table('c_page_navigations')
            ->where(['conference_id' => $id, 'page_type' => 1, 'status' => 1, 'parent_id' => 0])
            ->orderBy('orderpos')
            ->get()
            ->map(function ($item) use ($id) {
                $item->headerSubNav = DB::table('c_page_navigations')
                    ->where(['conference_id' => $id, 'parent_id' => $item->id, 'status' => 1])
                    ->orderBy('orderpos')->get();
                return $item;
            });

        return [
            'conferenceInfo'  => $conf,
            'headercontent'   => DB::table('c_header_content')->where('conference_id', $id)->first(),
            'headerNavigation'=> $nav,
            'socialLinks'     => DB::table('c_social')->where(['conference_id' => $id, 'status' => 1])->get(),
        ];
    }

    private function getFooterData($id)
    {
        return [
            'footerQuickLinks' => DB::table('c_page_navigations')
                ->where(['conference_id' => $id, 'page_type' => 2, 'status' => 1])
                ->orderBy('orderpos')->get(),
            'footerUsefulLinks' => DB::table('c_page_navigations')
                ->where(['conference_id' => $id, 'page_type' => 3, 'status' => 1])
                ->orderBy('orderpos')->get(),
            'socialLinks'   => DB::table('c_social')->where(['conference_id' => $id, 'status' => 1])->get(),
            'mainaddress'   => DB::table('web_cms')->where(['page_type' => 10, 'status' => 1])->first(),
            'contactInfo'   => DB::table('c_address')->where(['conference_id' => $id, 'status' => 1])->first(),
            'whatsappNumber'=> DB::table('c_address')->where('conference_id', $id)->value('whatsapp_number'),
        ];
    }

    /**
     * GET /api/phonecode/{countryName}
     * Returns the phone code for a given country name.
     */
    public function phoneCode($countryName)
    {
        $name = urldecode($countryName);
        $code = DB::table('countries')
            ->where('country_name', $name)
            ->value('country_code');

        if (!$code) {
            return response()->json(['code' => ''], 404);
        }
        return response()->json(['code' => (string)$code]);
    }

    /**
     * POST /api/brochure
     * Validates, stores user, emails admin + user (with brochure attachment if exists).
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'conference_id'    => 'required|integer',
            'designation'      => 'required|string|max:100',
            'firstname'        => 'required|string|max:100',
            'company'          => 'required|string|max:255',
            'usr_email'        => 'required|email',
            'usr_address'      => 'required|string',
            'country'          => 'required|string',
            'user_phone'       => 'required|string',
            'cp'               => 'required|string',
            'queries'          => 'nullable|string',
            'download_type'    => 'nullable|integer',
            // Server-side captcha (required by your rules)
            'captcha_answer'   => 'required|numeric',
            'captcha_expected' => 'required|numeric',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        if ((int)$request->captcha_answer !== (int)$request->captcha_expected) {
            return response()->json(['error' => 'Invalid captcha'], 422);
        }

        $conferenceId = (int)$request->conference_id;
        $conference = DB::table('conferences')->where('id', $conferenceId)->first();
        if (!$conference) {
            return response()->json(['error' => 'Invalid Conference'], 404);
        }

        // Persist user request
        $userId = DB::table('c_reg_users')->insertGetId([
            'conference_id'       => $conferenceId,
            'designation'         => $request->designation,
            'firstname'           => $request->firstname,
            'company'             => $request->company,
            'usr_email'           => strtolower($request->usr_email),
            'address'             => $request->usr_address,
            'country'             => $request->country,
            'user_phone'          => $request->user_phone,
            'usr_whatsapp'        => $request->usr_whatsapp,
            'cp'                  => $request->cp,
            'queries'             => $request->queries,
            'download_type'       => $request->download_type ?? 1,
            'user_type'           => 1,
            'created_date'        => now(),
            'invoiceid'           => '',
            'abstract_title'      => '',
            'transaction_id'      => '',
            'extra_info'          => '',
            'total_extra_amount'  => '',
            'coupon_value'        => '',
            'aditional_details'   => '',
        ]);

        // Admin email
        $admin = DB::table('c_address')->where('conference_id', $conferenceId)->first();
        $adminEmail = $admin->primary_mail ?? null;

        if ($adminEmail) {
            $subject = "Brochure Downloaded from " . ($conference->short_name ?? $conference->suname ?? 'Conference');
            $msg  = "Name: {$request->firstname}<br>";
            $msg .= "Organization: {$request->company}<br>";
            $msg .= "Phone: {$request->user_phone}<br>";
            $msg .= "Mail: {$request->usr_email}<br>";
            $msg .= "Query: {$request->queries}<br>";

            Mail::send([], [], function ($message) use ($adminEmail, $conference, $msg, $subject) {
                $message->to($adminEmail)
                    ->from($conference->smtp_user, $conference->suname)
                    ->bcc('kmlalnehru9@gmail.com')
                    ->subject($subject)
                    ->html($msg);
            });
        }

        // User email with brochure attachment if exists
        $brochure = DB::table('c_brouchers')
            ->where(['conference_id' => $conferenceId, 'page_type' => 1])
            ->first();

        if ($brochure && $request->usr_email) {
            $filePath = public_path("uploads/conferences/{$conference->unique_id}/{$brochure->brouchure_name}");

            $subjectUser = "Thank you for your download.";
            $message  = "Dear {$request->firstname},<br><br>";
            $message .= "Greetings<br><br>";
            $message .= "Thank you for downloading the brochure.<br><br>";
            $message .= "Revert us for any further queries.<br><br>";
            $message .= "With thanks,<br>Conference Coordinator";

            Mail::send([], [], function ($mail) use ($request, $conference, $message, $subjectUser, $filePath) {
                $mail->to($request->usr_email)
                    ->from($conference->smtp_user, $conference->suname)
                    ->subject($subjectUser);

                if (is_string($filePath) && file_exists($filePath)) {
                    $mail->attach($filePath);
                }

                $mail->html($message);
            });
        }

        return response()->json(['message' => 'Form submitted successfully!']);
    }
}
