<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Conference;
use App\Models\Subscription;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class SubscriptionApiController extends Controller
{
    // Show Subscribe/Unsubscribe Page
    public function show($id): JsonResponse
    {
        $conference = Conference::find($id);

        if (!$conference) {
            return response()->json([
                'success' => false,
                'message' => 'Conference not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'meta' => [
                'title' => 'Subscribe – Conference',
                'description' => 'Stay updated with our latest news and announcements.'
            ],
            'conferenceInfo' => [
                'id' => $conference->id,
                'conference_url' => $conference->conference_url,
                'unique_id' => $conference->unique_id
            ]
        ]);
    }

    // Handle Subscribe
    public function subscribe(Request $request, $id): JsonResponse
    {
        $request->validate([
            'email' => 'required|email',
            'name' => 'required|string|max:255',
        ]);

        $current_page = $request->input('current_page', "https://{$id}");
        $curDate = now();

        $conference = Conference::find($id);
        if (!$conference) {
            return response()->json(['success' => false, 'message' => 'Conference not found'], 404);
        }

        $existing = Subscription::where('usr_email', $request->email)
            ->where('conference_id', $id)
            ->first();

        if ($existing) {
            // Reactivate
            $existing->update([
                'status_s' => 1,
                'name' => $request->name
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Thank you for subscribing again!'
            ]);
        } else {
            // Insert new
            $subscription = Subscription::create([
                'conference_id' => $id,
                'usr_email' => $request->email,
                'name' => $request->name,
                'page_url' => $current_page,
                'created_date' => $curDate,
                'status_s' => 1
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Thank you for subscribing us. We will get back to you shortly.'
            ]);
        }
    }

    // Handle Unsubscribe
    public function unsubscribe(Request $request, $id): JsonResponse
    {
        $request->validate([
            'email' => 'required|email',
            'name' => 'required|string|max:255',
            'reason' => 'nullable|string'
        ]);

        $updated = Subscription::where('usr_email', $request->email)
            ->where('conference_id', $id)
            ->update([
                'status_s' => 0,
                'unsub_date' => now(),
                'reason_des' => $request->reason,
                'name' => $request->name
            ]);

        return response()->json([
            'success' => true,
            'message' => 'We are sorry to see you go. Thank you.'
        ]);
    }
}