<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Conference;
use App\Models\Cspeaker;
use Illuminate\Http\JsonResponse;

class SpeakerDetailsApiController extends Controller
{
    public function show($id, $alias_name = null): JsonResponse
    {
        // Read query params
        $speaker_type = request()->query('speaker_type'); 
        $alias_name   = $alias_name ?? request()->query('alias_name');

        // Build query
        $query = Cspeaker::with(['country', 'conference.SpeakerDetailsmetaData'])
            ->where('conference_id', $id)
            ->where('status', 1);

        if ($alias_name) {
            // Case-insensitive alias match
            $query->whereRaw('LOWER(alias_name) = ?', [strtolower($alias_name)]);
        }

        if ($speaker_type) {
            $query->where('speaker_type', $speaker_type);
        }

        $speaker = $query->first();

        if (!$speaker) {
            return response()->json(['error' => 'Speaker not found'], 404);
        }

        // Speaker details
        $speakerData = [
            'id'              => $speaker->id,
            'speaker_img'     => $speaker->speaker_img_url, // accessor
            'alias_name'      => $speaker->alias_name,
            'speaker_name'    => $speaker->speaker_name,
            'speaker_title'   => $speaker->speaker_title,
            'affliation'      => $speaker->affliation,
            'biography'       => $speaker->biography,
            'researchIntrest' => $speaker->researchIntrest,
            'acceptance_letter'=> $speaker->acceptance_letter,
            'countryInfo'     => $speaker->country
                ? ['country_name' => $speaker->country->country_name]
                : null,
        ];

        // Speaker title & URL
        switch ($speaker->speaker_type) {
            case 1:
                $speakerTitle = 'Scientific Committee';
                $pageUrl      = 'scientific-committee';
                break;
            case 2:
                $speakerTitle = 'Keynote Speaker';
                $pageUrl      = 'keynote-speakers';
                break;
            case 6:
                $speakerTitle = 'Eminent Speaker';
                $pageUrl      = 'eminent-speakers';
                break;
            default:
                $speakerTitle = 'Speaker';
                $pageUrl      = 'all-speakers';
                break;
        }

        $meta = $speaker->conference->SpeakerDetailsmetaData;

        // Response
        return response()->json([
            'success'           => true,
            'speakerDetailsData'=> $speakerData,
            'speakerTitle'      => $speakerTitle,
            'pageUrl'           => $pageUrl,
            'meta'              => [
                'title'       => "{$speaker->speaker_name} – {$speakerTitle}",
                'description' => "Profile of {$speaker->speaker_name}, {$speaker->speaker_title} at the conference.",
                'keywords'    => "{$speaker->speaker_name}, {$speakerTitle}, {$speaker->affliation}",
            ],
        ]);
    }
}
