<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\UserRegistrationDate;
use App\Models\Registration;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;

class OnlinePaypalApiController extends Controller
{
    /**
     * PayPal return redirect (user is sent here after payment)
     */
    public function returnFromPayPal(Request $request): RedirectResponse
    {
        $orderId = $request->get('order_id');

        if (!$orderId) {
            return redirect('/online-registration?error=missing_order');
        }

        // Mark as paid
        UserRegistrationDate::where('order_id', $orderId)
            ->update(['payment_status' => 'paid']);

        return redirect("/online-payment-success/$orderId");
    }

    /**
     * PayPal IPN (Instant Payment Notification)
     * PayPal server calls this silently
     */
    public function handleIPN(Request $request): JsonResponse
    {
        $rawData = file_get_contents('php://input');
        Log::info('PayPal IPN received (raw)', ['raw' => $rawData]);

        // Verify IPN with PayPal
        $verifyUrl = config('services.paypal.sandbox', true)
            ? 'https://ipnpb.sandbox.paypal.com/cgi-bin/webscr'
            : 'https://ipnpb.paypal.com/cgi-bin/webscr';

        // Prepare verification request
        $verifyData = 'cmd=_notify-validate&' . $rawData;

        try {
            $response = Http::timeout(30)->post($verifyUrl, $verifyData);

            if ($response->body() === 'VERIFIED') {
                // Parse IPN data
                parse_str($rawData, $ipnData);
                Log::info('PayPal IPN verified', $ipnData);

                // Extract invoice/order_id
                $orderId = $ipnData['invoice'] ?? $ipnData['custom'] ?? null;

                if ($orderId) {
                    // Update payment status
                    $updated = UserRegistrationDate::where('order_id', $orderId)
                        ->update([
                            'payment_status' => 'paid',
                            'transaction_id' => $ipnData['txn_id'] ?? null
                        ]);

                    if ($updated) {
                        Log::info('PayPal IPN: Order marked as paid', ['order_id' => $orderId]);
                    } else {
                        Log::warning('PayPal IPN: Order not found', ['order_id' => $orderId]);
                    }
                } else {
                    Log::warning('PayPal IPN: No order_id found in IPN data', $ipnData);
                }
            } else {
                Log::warning('PayPal IPN verification failed', [
                    'response' => $response->body(),
                    'raw_data' => $rawData
                ]);
                return response('INVALID', 400);
            }
        } catch (\Exception $e) {
            Log::error('PayPal IPN verification error', [
                'error' => $e->getMessage(),
                'raw_data' => $rawData
            ]);
            return response('ERROR', 500);
        }

        return response('OK', 200);
    }

    /**
     * Fetch success data (used by Vue success page)
     */
    public function getSuccessData($orderId): JsonResponse
    {
        $userRegistrationDate = UserRegistrationDate::with('registration')
            ->where('order_id', $orderId)
            ->first();

        if (!$userRegistrationDate) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found'
            ], 404);
        }

        $user = $userRegistrationDate->registration;

        return response()->json([
            'success' => true,
            'data' => [
                'order_id' => $userRegistrationDate->order_id,
                'total_amount' => $userRegistrationDate->total_amount,
                'currency' => $userRegistrationDate->curency_type,
                'payment_meathod' => $userRegistrationDate->payment_meathod,
                'usr_email' => $user->usr_email ?? '',
                'user_phone' => $user->user_phone ?? '',
                'created_date' => $userRegistrationDate->created_date
            ]
        ]);
    }
}