<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Conference;
use App\Models\Category;
use App\Models\Cspeaker;
use Illuminate\Http\JsonResponse;
use Carbon\Carbon;
use Carbon\CarbonPeriod;

class HomepageApiController extends Controller
{
    public function index($id): JsonResponse
    {
        $conference = Conference::with([
            'banners',
            'aboutUs',
            'quickLinks',
            'homePastGalleries',
            'sessionTracks',
            'importantDates',
            'ocmSpeakers.country.city',
            'previousSpeakersMapping.previousConference',
            'keynoteSpeakers.country.city',
            'eminentSpeakers.country.city',
            'supportJournals',
            'earlyBirds',
            'workshops',
            'venue',
            'globalTargets',
            'skeletonSchedules',
            'skeletonScheduleContent',
            'pastConferences',
            'welcomeMessages',
            'sponsors',
            'mediaPartners',
            'testimonials',
            'fieldWisePrograms',
            'homepageContent',
            'HomepageMetaData'
          
        ])->find($id);

        if (!$conference) {
            return response()->json(['error' => 'Conference not found'], 404);
        }

        // OCM Speakers
        $ocmSpeakers = $conference->ocmSpeakers->map(function ($speaker) {
            return [
                'id' => $speaker->id,
                'speaker_name' => $speaker->speaker_name,
                'speaker_title' => $speaker->speaker_title,
                'alias_name' => $speaker->alias_name,
                'affliation' => $speaker->affliation,
                'speaker_img' => $speaker->speaker_img,
                'country_id' => $speaker->country_id,
                'countryInfo' => $speaker->country ? [['country_name' => $speaker->country->country_name]] : [],
                'cityInfo' => $speaker->city ? [['city_name' => $speaker->city->city_name]] : [],
            ];
        });

        // Previous Speakers
        $prevArray = $this->getPreviousSpeakers($conference);

        // Keynote Speakers
        $keynoteArray = $this->getSpeakerArray($conference->keynoteSpeakers);

        // Eminent Speakers
        $eminentArray = $this->getSpeakerArray($conference->eminentSpeakers);

        // Skeleton Schedule
        $skeletonschedule = $this->buildSkeletonSchedule($conference);

        // Past Conference Reports
        $pastConferenceReport = $this->getPastConferenceReports($conference);

        // Past Conference Gallery
        [$homeprevArray, $pastvideosInfoname] = $this->getPastGalleryData($conference);

        // Our Past Events
        $homeprevAllArrayonlyfour = $this->getPastEvents($conference);

        // Testimonial Speakers
        $testimonialspeakers = $conference->testimonials->take(1);

        return response()->json([
            'success' => true,
            'homepageData' => [
                'banners' => $conference->banners,
                'aboutus' => $conference->aboutUs ? $conference->aboutUs : [],
                'pastConf' => $conference->pastConf->take(1)->first(),
                'quickLinks' => $conference->quickLinks,
                'sessionTracks' => $conference->sessionTracks,
                'impDates' => $conference->importantDates,
                'keynoteSpeakers' => $keynoteArray,
                'prevArray' => $prevArray,
                'ocmSpeakers' => $ocmSpeakers,
                'eminentSpeakers' => $eminentArray,
                'earlyBird' => $conference->earlyBirds,
                'workshops' => $conference->workshops,
                'supportJournals' => $conference->supportJournals,
                'venueInfo' => $conference->venue,
                'globaltarget' => $conference->globalTargets,
                'skeletonschedule' => $skeletonschedule->values(),
                'schedulecontent' => $conference->skeletonScheduleContent ? [$conference->skeletonScheduleContent] : [],
                'homeprevAllArray' => [], // You can implement this if needed
                'welcomeMessage' => $conference->welcomeMessages,
                'sponsors' => $conference->sponsors,
                'mediaPartners' => $conference->mediaPartners,
                'testimonials' => $conference->testimonials,
                'testimonialspeakers' => $testimonialspeakers,
                'homeprevArray' => $homeprevArray,
                'homeprevAllArrayonlyfour' => $homeprevAllArrayonlyfour,
                'conferenceVideos' => $this->getConferenceVideos($id),
                'pastvideosInfoname' => $pastvideosInfoname,
                'pastconferencerepports' => $pastConferenceReport,
            ]
        ]);
    }

    private function getPreviousSpeakers($conference)
    {
        $prevArray = [];

        foreach ($conference->previousSpeakersMapping as $row) {
            $prevConf = $row->previousConference;
            if (!$prevConf) continue;

            $speakers = Cspeaker::with('country.city')
                ->where([
                    'status' => 1,
                    'conference_id' => $row->previous_conference_id,
                    'speaker_type' => 6,
                ])
                ->orderBy('orderpos', 'ASC')
                ->limit(4)
                ->get();

            if ($speakers->isEmpty()) {
                $prevArray[] = [
                    'id' => $row->id,
                    'conference_id' => $row->conference_id,
                    'previous_conference_id' => $row->previous_conference_id,
                    'orderpos' => $row->orderpos,
                    'status' => $row->status,
                    'prevConfInfo' => collect([$prevConf]),
                    'speakersInfo' => collect(),
                ];
                continue;
            }

            $speakersInfo = $speakers->map(function ($s) {
                return [
                    'id' => $s->id,
                    'conference_id' => $s->conference_id,
                    'speaker_title' => $s->speaker_title,
                    'speaker_name' => $s->speaker_name,
                    'alias_name' => $s->alias_name,
                    'speaker_img' => $s->speaker_img,
                    'affliation' => $s->affliation,
                    'continent_id' => $s->continent_id,
                    'country_id' => $s->country_id,
                    'city_id' => $s->city_id,
                    'hospital_id' => $s->hospital_id,
                    'university_id' => $s->university_id,
                    'continentsInfo' => [],
                    'countryInfo' => $s->country ? [['id' => $s->country->id, 'country_name' => $s->country->country_name]] : [],
                    'cityInfo' => $s->city ? [['id' => $s->city->id, 'city_name' => $s->city->city_name]] : [],
                    'universityInfo' => [],
                    'hospitalInfo' => [],
                ];
            });

            $prevArray[] = [
                'id' => $row->id,
                'conference_id' => $row->conference_id,
                'previous_conference_id' => $row->previous_conference_id,
                'orderpos' => $row->orderpos,
                'status' => $row->status,
                'prevConfInfo' => collect([$prevConf]),
                'speakersInfo' => $speakersInfo,
            ];
        }

        return $prevArray;
    }

    private function getSpeakerArray($speakers)
    {
        return $speakers->map(function ($s) {
            return [
                'id' => $s->id,
                'speaker_name' => $s->speaker_name,
                'speaker_title' => $s->speaker_title,
                'alias_name' => $s->alias_name,
                'affliation' => $s->affliation,
                'speaker_img' => $s->speaker_img,
                'country_id' => $s->country_id,
                'countryInfo' => $s->country ? [['country_name' => $s->country->country_name]] : [],
                'cityInfo' => $s->city ? [['city_name' => $s->city->city_name]] : [],
            ];
        })->values()->all();
    }

    private function buildSkeletonSchedule($conference)
    {
        $skeletonschedule = collect();

        if ($conference->from_date && $conference->to_date) {
            $period = CarbonPeriod::create(
                Carbon::parse($conference->from_date)->startOfDay(),
                Carbon::parse($conference->to_date)->startOfDay()
            );

            $schDays = collect($period)->values();

            $skeletonschedule = $schDays->map(function ($schDate, $idx) use ($conference) {
                $dayIndex = $idx + 1;

                $speakersInfo = $conference->skeletonSchedules
                    ->where('sch_day', $dayIndex)
                    ->values();

                return [
                    'day' => 'Day-' . $dayIndex,
                    'schDate' => $schDate->format('Y-m-d'),
                    'schId' => $idx,
                    'speakersInfo' => $speakersInfo,
                ];
            });
        }

        return $skeletonschedule;
    }

    private function getPastConferenceReports($conference)
    {
        $pastConfIds = $conference->pastConferences->pluck('past_conference_id')->filter()->unique();

        if ($pastConfIds->isEmpty()) {
            return [];
        }

        $pastReports = \DB::table('c_past_conference_report')
            ->where('status', 1)
            ->whereIn('conference_id', $pastConfIds)
            ->orderBy('id', 'ASC')
            ->get();

        $reportsByConf = $pastReports->groupBy('conference_id');

        $pastConferenceReport = [];
        foreach ($pastConfIds as $confId) {
            $reports = $reportsByConf->get($confId, collect());
            if ($reports->isNotEmpty()) {
                $pastConferenceReport[] = $reports->values();
            }
        }

        return $pastConferenceReport;
    }

    private function getPastGalleryData($conference)
    {
        $homeprevGallery = $conference->homePastGalleries
            ->where('page_type', 1)
            ->first();

        $homeprevArray = collect();
        $pastvideosInfoname = collect();

        if ($homeprevGallery) {
            $targetPastConfId = $homeprevGallery->past_conference_id;

            $homeprevArray = \DB::table('previous_conference_gallery')
                ->where('status', 1)
                ->where('conference_id', $targetPastConfId)
                ->where('parent_id', '<>', 0)
                ->orderBy('id', 'ASC')
                ->limit(8)
                ->get(['id','gallery_img','gallery_title']);

            $pastvideosInfoname = \DB::table('conferences')
                ->where(['status' => 1, 'id' => $targetPastConfId])
                ->limit(1)
                ->get(['id','short_alias_name']);
        }

        return [$homeprevArray, $pastvideosInfoname];
    }

    private function getPastEvents($conference)
    {
        $homePrevAllGalleryonlyfour = $conference->homePastGalleries
            ->where('page_type', 2)
            ->sortByDesc('id')
            ->take(4);

        $homeprevAllArrayonlyfour = $homePrevAllGalleryonlyfour->map(function ($hp) {
            $conf = \DB::table('conferences')
                ->where('id', $hp->past_conference_id)
                ->first(['id','short_alias_name']);

            $gallery = \DB::table('previous_conference_gallery')
                ->where(['status' => 1, 'conference_id' => $hp->past_conference_id])
                ->orderBy('id', 'ASC')
                ->limit(1)
                ->get(['gallery_img','gallery_title']);

            return [
                'conference' => $conf,
                'gallery' => $gallery,
            ];
        })->values();

        return $homeprevAllArrayonlyfour;
    }

    private function getConferenceVideos($id)
    {
        return \DB::table('previous_conference_videos')
            ->where(['status' => 1, 'parent_id' => $id])
            ->limit(3)
            ->get(['id','video_url','video_title']);
    }

    public function indexv1($id): JsonResponse
    {
        $conference = Conference::with([
            'banners',
            'sessionTracks',
            'importantDates',
            'aboutUs',
            'welcomeMessages',
            'quickLinks',
            'ocmSpeakers.country.city',
            'sponsors',
            'mediaPartners',
            'venue',
            'globalTargets',
            'homepageContent',
            'fieldWisePrograms',
            'pastConferences',
            'skeletonScheduleContent',
            'testimonials',
            'HomepageMetaData'
        ])->find($id);

        if (!$conference) {
            return response()->json(['error' => 'Conference not found'], 404);
        }

        // OCM Speakers
        $ocmSpeakers = $conference->ocmSpeakers->map(function ($r) {
            return [
                'id' => $r->id,
                'speaker_name' => $r->speaker_name,
                'speaker_title' => $r->speaker_title,
                'alias_name' => $r->alias_name,
                'affliation' => $r->affliation,
                'speaker_img' => $r->speaker_img,
                'country_id' => $r->country_id,
                'countryInfo' => $r->country ? [['country_name' => $r->country->country_name]] : [],
                'cityInfo' => $r->city ? [['city_name' => $r->city->city_name]] : [],
            ];
        })->values();

        // Eminent Speakers by Category
        $categoriesResult = Category::where('status', 1)->orderBy('sort_order', 'ASC')->get(['id', 'category_name']);
        $groupedSpeakers = [];

        foreach ($categoriesResult as $category) {
            $speakers = Cspeaker::with('country.city')
                ->where([
                    'status' => 1,
                    'conference_id' => $id,
                    'speaker_type' => $category->id
                ])
                ->orderBy('orderpos', 'ASC')
                ->get();

            $groupedSpeakers[$category->id] = $speakers->map(function ($r) {
                return [
                    'id' => $r->id,
                    'speaker_name' => $r->speaker_name,
                    'speaker_title' => $r->speaker_title,
                    'alias_name' => $r->alias_name,
                    'affliation' => $r->affliation,
                    'speaker_img' => $r->speaker_img,
                    'country_id' => $r->country_id,
                    'countryInfo' => $r->country ? [['country_name' => $r->country->country_name]] : [],
                    'cityInfo' => $r->city ? [['city_name' => $r->city->city_name]] : [],
                ];
            })->values();
        }

        // Past Conference Reports
        $pastConfIds = $conference->pastConferences->pluck('past_conference_id')->filter();
        $pastConferenceReport = [];

        if ($pastConfIds->isNotEmpty()) {
            $reports = \DB::table('c_past_conference_report')
                ->where('status', 1)
                ->whereIn('conference_id', $pastConfIds)
                ->orderBy('id', 'ASC')
                ->get(['id', 'conf_title', 'conf_desc']);

            foreach ($pastConfIds as $confId) {
                $filtered = $reports->where('conference_id', $confId);
                if ($filtered->isNotEmpty()) {
                    $pastConferenceReport[] = $filtered->values();
                }
            }
        }

        // Past Conference Gallery
        $homeprevGallery = $conference->homePastGalleries
            ->where('page_type', 1)
            ->first();

        $homeprevArray = collect();
        if ($homeprevGallery) {
            $homeprevArray = \DB::table('previous_conference_gallery')
                ->where([
                    'status' => 1,
                    'conference_id' => $homeprevGallery->past_conference_id,
                    'parent_id' => 0
                ])
                ->orderBy('id', 'ASC')
                ->limit(8)
                ->get(['id', 'gallery_img', 'gallery_title']);
        }

        // Our Past Events (only 4)
        $homePrevAllGalleryonlyfour = $conference->homePastGalleries
            ->where('page_type', 2)
            ->sortByDesc('id')
            ->take(4);

        $homeprevAllArrayonlyfour = $homePrevAllGalleryonlyfour->map(function ($item) {
            $conference = \DB::table('conferences')
                ->where('id', $item->past_conference_id)
                ->first(['id', 'short_alias_name']);

            $gallery = \DB::table('previous_conference_gallery')
                ->where('status', 1)
                ->where('conference_id', $item->past_conference_id)
                ->orderBy('id', 'ASC')
                ->limit(1)
                ->get(['gallery_img', 'gallery_title']);

            return [
                'conference' => $conference,
                'gallery' => $gallery
            ];
        })->values();

        // Conference Videos
        $conferenceVideos = \DB::table('previous_conference_videos')
            ->where(['status' => 1, 'parent_id' => $id])
            ->limit(3)
            ->get(['id', 'video_url', 'video_title']);

        $pastvideosInfoname = collect();
        if ($conferenceVideos->isNotEmpty()) {
            $confId = $conferenceVideos[0]->conference_id ?? null;
            $pastvideosInfoname = \DB::table('conferences')
                ->where('id', $confId)
                ->limit(1)
                ->get(['id', 'short_alias_name']);
        }

        // Testimonial Speakers
        $testimonialspeakers = \DB::table('testimonialspeakersays')
            ->where('status', 1)
            ->orderBy('id', 'DESC')
            ->limit(1)
            ->get(['id', 'testimonial_title', 'testimonial_desc']);

        // Skeleton Schedule
        $skeletonschedule = collect();
        if ($conference->from_date && $conference->to_date) {
            $from = Carbon::parse($conference->from_date);
            $to = Carbon::parse($conference->to_date);
            $period = CarbonPeriod::create($from, '1 day', $to);

            $dayCounter = 1;
            foreach ($period as $date) {
                $schDate = $date->format('Y-m-d');
                $speakersInfo = $conference->skeletonSchedules
                    ->where('sch_day', $dayCounter)
                    ->values();

                $skeletonschedule->push([
                    'day' => 'Day ' . $dayCounter,
                    'schDate' => $schDate,
                    'schId' => $dayCounter - 1,
                    'speakersInfo' => $speakersInfo
                ]);

                $dayCounter++;
            }
        }

        // Schedule Content
        $schedulecontent = $conference->skeletonScheduleContent ? [$conference->skeletonScheduleContent] : [];

        // Support Journals
        $supportJournals = $conference->supportJournals;

        // Early Bird
        $earlyBird = $conference->earlyBirds;

        return response()->json([
            'success' => true,
            'homepageData' => [
                'conferenceInfo' => $conference,
                'banners' => $conference->banners,
                'sessionTracks' => $conference->sessionTracks,
                'quickLinks' => $conference->quickLinks,
                'impDates' => $conference->importantDates,
                'aboutus' => $conference->aboutUs,
                'welcomeMessage' => $conference->welcomeMessages,
                'ocmSpeakers' => $ocmSpeakers,
                'categoriesResult' => $categoriesResult,
                'groupedSpeakers' => $groupedSpeakers,
                'sponsors' => $conference->sponsors,
                'mediaPartners' => $conference->mediaPartners,
                'venueInfo' => $conference->venue,
                'globaltarget' => $conference->globalTargets,
                'popcontentInfo' => $conference->homepageContent ? [$conference->homepageContent] : [],
                'fieldwise' => $conference->fieldWisePrograms,
                'pastconferencerepports' => $pastConferenceReport,
                'homeprevArray' => $homeprevArray,
                'homeprevAllArrayonlyfour' => $homeprevAllArrayonlyfour,
                'conferenceVideos' => $conferenceVideos,
                'pastvideosInfoname' => $pastvideosInfoname,
                'testimonials' => $conference->testimonials,
                'testimonialspeakers' => $testimonialspeakers,
                'skeletonschedule' => $skeletonschedule->values(),
                'schedulecontent' => $schedulecontent,
                'supportJournals' => $supportJournals,
                'earlyBird' => $earlyBird,
            ]
        ]);
    }
}