<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Conference;
use App\Models\Country;
use App\Models\Designation;
use App\Models\Category;
use App\Models\Registration;
use App\Models\RegistrationShipping;
use App\Models\GroupRegistrationDate;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Stripe\Stripe;
use Stripe\PaymentIntent;
use Illuminate\Support\Facades\Log;

class GroupRegistrationApiController extends Controller
{
    public function indexV5($id, $currency = 3): JsonResponse
    {
        $currency = in_array((int)$currency, [1,2,3]) ? (int)$currency : 3;

        $conference = Conference::with([
            'registrationDates',
            'groupRegistrationAmounts' => fn($q) => $q->where('currency_id', $currency)->orderBy('id'),
            'onsiteRegistrationStatus'
        ])->find($id);

        if (!$conference) {
            return response()->json(['success' => false, 'message' => 'Conference not found'], 404);
        }

        $design = Designation::where('status', 1)->orderBy('id')->get();
        $countries = Country::where('status', 1)->orderBy('country_name')->get();
        $categories = Category::where('status', 1)->orderBy('id')->get();
        $captcha = rand(100000, 999999);

        return response()->json([
            'success' => true,
            'data' => [
                'conferenceId' => $id,
                'currency' => $currency,
                'dates' => $conference->registrationDates,
                'registrationAmount' => $conference->groupRegistrationAmounts,
                'onsiteStatus' => $conference->onsiteRegistrationStatus,
                'designations' => $design,
                'countries' => $countries,
                'newcaptcha' => $captcha,
                'categories' => $categories,
            ]
        ]);
    }

    public function getPhoneCodeV5($name): JsonResponse
    {
        $country = Country::where(['status' => 1, 'country_name' => $name])->first();
        if (!$country || empty($country->country_code)) {
            return response()->json(['success' => false, 'message' => 'Country code not found'], 404);
        }
        return response()->json(['success' => true, 'phonecode' => '+' . $country->country_code]);
    }

    public function submitV5(Request $r): JsonResponse
    {
        $orderId = 'GRP-' . strtoupper(uniqid());
        $currency = in_array((int)($r->currsel ?? 3), [1,2,3]) ? (int)$r->currsel : 3;

        $sel = is_array($r->selection ?? null) ? $r->selection : [];
        $qty = (int)($sel['qty'] ?? $r->selected_qty ?? $r->qty ?? 0);
        $planId = (int)($sel['plan_id'] ?? $r->selected_id ?? $r->plan_id ?? 0);
        $totalAmt = (float)($r->totalAmount ?? $r->total_Amount ?? $r->total_amount ?? 0.0);

        if ($qty <= 0 || $planId <= 0) {
            return response()->json(['success' => false, 'message' => 'Invalid selection or quantity'], 422);
        }

        \DB::beginTransaction();
        try {
            $countryCode = Country::where('country_name', $r->country)->value('country_code') ?? '';

            $registration = Registration::create([
                'conference_id' => $r->conferenceId,
                'designation' => $r->designation,
                'firstname' => $r->firstname,
                'lastname' => $r->lastname,
                'usr_email' => $r->usr_email,
                'country' => $r->country,
                'country_code' => $countryCode,
                'user_phone' => $r->user_phone,
                'usr_whatsapp' => $r->whatsapp_number,
                'company' => $r->company,
                'category' => $r->category,
                'address' => $r->address,
                'user_type' => 8, // group
                'currency' => $currency,
                'payment_method' => $r->payment_meathod,
                'total_amount' => $totalAmt,
                'created_date' => now(),
            ]);

            $registration->shipping()->create([
                'conference_id' => $r->conferenceId,
                'shipDesign' => $r->shipdesignation,
                'shipFirstname' => $r->shipfirstname,
                'shipLastname' => $r->shiplastname,
                'shipCompany' => $r->shipcompany,
                'shipCountry' => $r->shipcountry,
                'shipEmail' => $r->shipusr_email,
                'shipPhone' => $r->shipphone,
                'shipWhatsapp' => $r->shipwhatsapp_number,
                'shipAddress' => $r->shipaddress,
                'createDate' => now(),
            ]);

            $now = date('Y-m-d');
            $plan = GroupRegistrationAmount::where([
                'conference_id' => $r->conferenceId,
                'currency_id' => $currency,
                'id' => $planId,
                'status' => 1
            ])->first();

            $dates = $registration->conference->registrationDates()->first();

            $unit = 0.0;
            if ($plan && $dates) {
                if (!empty($dates->early_date) && strtotime($dates->early_date) >= strtotime($now)) {
                    $unit = (float)$plan->early_bird_amount;
                } elseif (!empty($dates->standard_date) && strtotime($dates->standard_date) >= strtotime($now)) {
                    $unit = (float)$plan->first_call_amount;
                } elseif (!empty($dates->onsite_date) && strtotime($dates->onsite_date) >= strtotime($now)) {
                    $unit = (float)$plan->final_call_amount;
                } elseif (!empty($dates->onsite_reg_date) && strtotime($dates->onsite_reg_date) >= strtotime($now)) {
                    $unit = (float)$plan->onsite_registration;
                }
            }
            $regTotal = round($qty * $unit, 2);

            GroupRegistrationDate::create([
                'order_id' => $orderId,
                'user_id' => $registration->id,
                'earlyType' => null,
                'registration_id' => $planId,
                'reg_qty' => $qty,
                'reg_total_amount' => $regTotal,
                'total_amount' => $totalAmt,
                'curency_type' => $currency,
                'payment_meathod' => $r->payment_meathod,
                'reg_date' => now(),
                'created_date' => now(),
            ]);

            \DB::commit();

            return response()->json([
                'success' => true,
                'order_id' => $orderId,
                'total' => $totalAmt,
                'currency' => $this->code($currency),
                'symbol' => $this->sym($currency),
            ]);
        } catch (\Throwable $e) {
            \DB::rollBack();
            Log::error('group submit v5 failed: ' . $e->getMessage());
            return response()->json(['success' => false, 'message' => 'Submit failed'], 500);
        }
    }

    public function createPaymentIntentV5(Request $r): JsonResponse
    {
        $r->validate([
            'order_id' => 'required|string',
            'amount' => 'required|integer|min:1',
            'currency' => 'required|string|in:usd,eur,gbp',
            'user.name' => 'required|string',
            'user.email' => 'required|email',
            'conference_id' => 'required|integer',
        ]);

        Stripe::setApiKey(env('STRIPE_SECRET_KEY'));
        $conference = Conference::find($r->conference_id);
        if (!$conference) {
            return response()->json(['error' => 'Conference not found'], 404);
        }

        $confName = trim(($conference->conference_first_title ?? '') . ' ' . ($conference->conference_name ?? ''));
        $intent = PaymentIntent::create([
            'amount' => $r->amount,
            'currency' => $r->currency,
            'description' => "Group Registration: {$confName}",
            'metadata' => [
                'order_id' => $r->order_id,
                'user_name' => $r->user['name'],
                'user_email' => $r->user['email'],
                'conference_id' => $r->conference_id,
                'conference_name' => $confName,
            ],
            'payment_method_types' => ['card'],
        ]);
        return response()->json(['clientSecret' => $intent->client_secret]);
    }

    public function stripeReturnV5(Request $request): JsonResponse
    {
        Stripe::setApiKey(env('STRIPE_SECRET_KEY'));
        $piId = $request->query('payment_intent');
        if (!$piId) {
            $frontend = rtrim(config('app.frontend_url', env('FRONTEND_URL','/')), '/');
            return redirect($frontend.'/group-payment-failed-v5');
        }

        try {
            $pi = PaymentIntent::retrieve($piId);
            $orderId = $pi->metadata['order_id'] ?? null;
            if (!$orderId) {
                $frontend = rtrim(config('app.frontend_url', env('FRONTEND_URL','/')), '/');
                return redirect($frontend.'/group-payment-failed-v5');
            }

            $newStatus = $pi->status === 'succeeded' ? 'Completed' : 'Pending';

            GroupRegistrationDate::where('order_id', $orderId)->update([
                'payment_status' => $newStatus,
                'payment_meathod' => 1, // Stripe
                'payment_intent_id' => $pi->id,
            ]);

            $row = GroupRegistrationDate::where('order_id', $orderId)->first();
            if ($row) {
                Registration::where('id', $row->user_id)->update([
                    'payment_status' => $newStatus,
                    'transaction_id' => $orderId,
                ]);
            }

            $cookie = cookie('scs_group_order', $orderId, 5, null, null, false, false, false, 'Lax');
            $frontend = rtrim(config('app.frontend_url', env('FRONTEND_URL','/')), '/');
            return redirect($frontend.'/group-payment-success-v5')->withCookie($cookie);

        } catch (\Exception $e) {
            Log::error('stripe group v5 return failed: ' . $e->getMessage());
            $frontend = rtrim(config('app.frontend_url', env('FRONTEND_URL','/')), '/');
            return redirect($frontend.'/group-payment-failed-v5');
        }
    }

    public function getPaymentSuccessDataV5($order_id): JsonResponse
    {
        $data = Registration::join('c_user_group_registrations_dates as r', 'c_reg_users.id', '=', 'r.user_id')
            ->where('r.order_id', $order_id)
            ->select(
                'c_reg_users.firstname',
                'c_reg_users.lastname',
                'c_reg_users.usr_email',
                'c_reg_users.user_phone',
                'c_reg_users.company',
                'c_reg_users.country',
                'c_reg_users.invoiceid',
                'r.order_id',
                'r.total_amount',
                'r.curency_type as currency_id',
                'r.payment_meathod as payment_method',
                'r.payment_status',
                'c_reg_users.transaction_id',
                'r.created_date'
            )->first();

        if (!$data) {
            return response()->json(['success' => false, 'message' => 'Order not found'], 404);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'order_id' => $data->order_id,
                'firstname' => $data->firstname,
                'lastname' => $data->lastname,
                'usr_email' => $data->usr_email,
                'user_phone' => $data->user_phone,
                'company' => $data->company,
                'country' => $data->country,
                'total_amount' => $data->total_amount,
                'currency' => $this->code((int)$data->currency_id),
                'payment_method' => (int)$data->payment_method,
                'payment_status' => $data->payment_status,
                'transaction_id' => $data->transaction_id,
                'invoiceid' => $data->invoiceid,
                'created_date' => $data->created_date,
            ]
        ]);
    }

    public function summaryV5($order_id): JsonResponse
    {
        return $this->getPaymentSuccessDataV5($order_id);
    }

    private function code($id) { return [1=>'GBP',2=>'USD',3=>'EUR'][$id] ?? 'EUR'; }
    private function sym($id) { return [1=>'£',2=>'$',3=>'€'][$id] ?? '€'; }

    public function paypalIpnV5(Request $request): JsonResponse
    {
        $raw = file_get_contents('php://input');
        Log::info('PayPal IPN hit', $request->all());

        $sandbox = filter_var(env('PAYPAL_SANDBOX', true), FILTER_VALIDATE_BOOLEAN);
        $verifyUrl = $sandbox
            ? 'https://ipnpb.sandbox.paypal.com/cgi-bin/webscr'
            : 'https://ipnpb.paypal.com/cgi-bin/webscr';

        $ch = curl_init($verifyUrl);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => 'cmd=_notify-validate&'.$raw,
            CURLOPT_HTTPHEADER => ['Content-Type: application/x-www-form-urlencoded', 'Connection: Close'],
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_SSL_VERIFYHOST => 2,
            CURLOPT_TIMEOUT => 30,
        ]);
        $resp = curl_exec($ch);
        $cerr = curl_error($ch);
        curl_close($ch);

        if ($resp !== 'VERIFIED') {
            Log::warning('PayPal IPN not VERIFIED', ['resp' => $resp, 'err' => $cerr]);
            return response('INVALID', 400);
        }

        parse_str($raw, $p);
        if (empty($p)) $p = $request->all();

        $recv = $p['receiver_email'] ?? $p['business'] ?? '';
        $expect = $sandbox ? env('PAYPAL_BUSINESS_SANDBOX', '') : env('PAYPAL_BUSINESS_LIVE', '');
        if ($expect && strcasecmp(trim($recv), trim($expect)) !== 0) {
            Log::warning('IPN receiver_email mismatch', compact('recv','expect'));
            return response('BAD_RECEIVER', 400);
        }

        $status = $p['payment_status'] ?? '';
        $txnId = $p['txn_id'] ?? '';
        $orderId = $p['custom']
            ?? $p['invoice']
            ?? $p['item_number1']
            ?? $p['item_number_1']
            ?? $p['item_number']
            ?? null;

        Log::info('IPN parsed', ['status' => $status, 'orderId' => $orderId, 'txn' => $txnId]);
        if (!$orderId) return response('NO_ORDER', 400);

        $reg = GroupRegistrationDate::where('order_id', $orderId)->first();
        if (!$reg) {
            Log::warning('IPN order not found', compact('orderId'));
            return response('NO_MATCH', 404);
        }

        $newStatus = $status === 'Completed' ? 'Completed' : 'Pending';

        \DB::beginTransaction();
        try {
            $update = ['payment_status' => $newStatus, 'payment_meathod' => 2];
            if (!empty($txnId)) $update['transaction_id'] = $txnId;

            GroupRegistrationDate::where('order_id', $orderId)->update($update);

            Registration::where('id', $reg->user_id)->update([
                'payment_status' => $newStatus,
                'transaction_id' => $txnId ?: $orderId,
            ]);

            \DB::commit();
        } catch (\Throwable $e) {
            \DB::rollBack();
            Log::error('IPN DB update failed', ['orderId' => $orderId, 'err' => $e->getMessage()]);
            return response('DB_ERR', 500);
        }

        Log::info('IPN updated', [
            'orderId' => $orderId,
            'status' => GroupRegistrationDate::where('order_id',$orderId)->value('payment_status')
        ]);

        return response('OK', 200);
    }
}