<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Conference;
use App\Models\Designation;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class ConferenceApiController extends Controller
{
    protected $defaultConferenceId = 128;



public function index()
    {
        // conferences టేబుల్ నుండి id, folder_name మాత్రమే తీసుకోవడం
        $conferences = Conference::select('id', 'folder_name')->get();

        return response()->json($conferences);
    }
    public function show($id = null): JsonResponse
    {
        $conferenceId = $id ?? $this->defaultConferenceId;

        $conference = Conference::with([
            'headerContent',
            'navigations',
            'socialLinks',
            'address',
            'VenueV1metaData', // No page_type filter — will filter in controller if needed
            'sponsors'
        ])->find($conferenceId);

        if (!$conference) {
            return response()->json(['error' => 'Conference not found'], 404);
        }

        $metaData = $conference->VenueV1metaData->first(); // Adjust if you need specific page_type

        return response()->json([
            'success' => true,
            'conferenceInfo' => $conference,
            'meta' => [
                'meta_title' => $metaData?->meta_title ?? $conference->conference_name,
                'meta_desc' => $metaData?->meta_desc ?? '',
                'meta_keywords' => $metaData?->meta_keywords ?? '',
                'blue_color' => $conference->blue_color,
                'pink_color' => $conference->pink_color,
                'unique_id' => $conference->unique_id,
                'favicon' => $conference->fav_icon,
            ],
            'headerData' => $this->getHeaderNavigation($conference),
            'footerData' => [
                'quickLinks' => $this->getFooterLinks($conference, 2),
                'usefulLinks' => $this->getFooterLinks($conference, 3),
                'socialLinks' => $conference->socialLinks,
                'mainaddress' => \DB::table('web_cms')->where('page_type', 10)->where('status', 1)->first(),
                'contactInfo' => $conference->address,
                'whatsappNumber' => $conference->address?->whatsapp_number,
            ],
            'pageData' => [
                'contactData' => [
                    'designations' => Designation::orderBy('status')->get(),
                    'mediaPartners' => $conference->sponsors()
                        ->where('page_type', 2)
                        ->where('status', 1)
                        ->orderBy('id', 'DESC')
                        ->get(),
                ],
                'mediaPartnersData' => [
                    'mediaPartners' => $conference->sponsors()
                        ->where('page_type', 2)
                        ->where('status', 1)
                        ->orderBy('id', 'DESC')
                        ->get(),
                ],
            ],
        ]);
    }

    public function common($id): JsonResponse
    {
        $conference = Conference::with([
            'navigations',
            'socialLinks',
            'address'
        ])->find($id);

        if (!$conference) {
            return response()->json(['error' => 'Conference not found'], 404);
        }

        return response()->json([
            'success' => true,
             'conferenceInfo' => $conference,
            'headerData' => $this->getHeaderNavigation($conference),
            'footerData' => [
                'quickLinks' => $this->getFooterLinks($conference, 2),
                'usefulLinks' => $this->getFooterLinks($conference, 3),
                'socialLinks' => $conference->socialLinks,
                'mainaddress' => \DB::table('web_cms')->where('page_type', 10)->where('status', 1)->first(),
                'contactInfo' => $conference->address,
                'whatsappNumber' => $conference->address?->whatsapp_number,
            ],
        ]);
    }

    private function getHeaderNavigation(Conference $conference)
{
    return $conference->navigations
        ->where('page_type', 1)
        ->where('status', 1)
        ->where('parent_id', 0)
        ->sortBy('orderpos')
        ->map(function ($item) use ($conference) {
            $item->headerSubNav = $conference->navigations
                ->where('parent_id', $item->id)
                ->where('status', 1)
                ->sortBy('orderpos')
                ->values() // reindex
                ->toArray(); // ✅ force array
            return $item;
        })
        ->values() // reindex top-level too
        ->toArray(); // ✅ force array
}


    private function getFooterLinks(Conference $conference, $pageType)
    {
        return $conference->navigations
            ->where('page_type', $pageType)
            ->where('status', 1)
            ->sortBy('orderpos');
    }

    public function findByPath(Request $request): JsonResponse
    {
        $path = trim($request->input('path'), '/');

        if (!$path) {
            return response()->json(['error' => 'Path parameter required'], 400);
        }

        $conferenceInfo = Conference::where('folder_name', $path)->first();

        if (!$conferenceInfo) {
            return response()->json(['error' => 'Conference not found'], 404);
        }

        return response()->json([
            'id' => $conferenceInfo->id,
            'name' => $conferenceInfo->conference_name, // Fixed to match your DB column
            'folder_name' => $conferenceInfo->folder_name,
        ]);
    }
}