<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Conference;
use App\Models\Country;
use App\Models\Registration;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;
use Illuminate\Http\JsonResponse;

class BrochureApiController extends Controller
{
    public function show($id): JsonResponse
    {
        $conference = Conference::with([
            'headerContent',
            'navigations',
            'socialLinks',
            'address',
            'BrochuremetaData'
        ])->find($id);

        if (!$conference) {
            return response()->json(['error' => 'Conference not found'], 404);
        }

        $meta = $conference->BrochuremetaData;

        $countries = Country::select('id', 'country_name', 'country_code')
            ->where('status', 1)
            ->orderBy('country_name')
            ->get();

        return response()->json([
            'success' => true,
            'conferenceInfo' => $conference,
            'headerData' => $this->getHeaderData($conference),
            'footerData' => $this->getFooterData($conference),
            'commonData' => [
                'countriesList' => $countries,
            ],
            'meta' => [
                'title'       => $meta?->meta_title ?? $meta?->title ?? '',
                'description' => $meta?->meta_description ?? '',
                'keywords'    => $meta?->meta_keywords ?? ''
            ],
            'favicon' => $conference->fav_icon ?? ''
        ]);
    }

    private function getHeaderData(Conference $conference)
    {
        $headerNav = $conference->navigations
            ->where('page_type', 1)
            ->where('status', 1)
            ->where('parent_id', 0)
            ->sortBy('orderpos')
            ->map(function ($item) use ($conference) {
                $item->headerSubNav = $conference->navigations
                    ->where('parent_id', $item->id)
                    ->where('status', 1)
                    ->sortBy('orderpos');
                return $item;
            });

        return [
            'conferenceInfo'  => $conference,
            'headercontent'   => $conference->headerContent,
            'headerNavigation'=> $headerNav,
            'socialLinks'     => $conference->socialLinks,
        ];
    }

    private function getFooterData(Conference $conference)
    {
        return [
            'footerQuickLinks' => $conference->navigations
                ->where('page_type', 2)
                ->where('status', 1)
                ->sortBy('orderpos'),
            'footerUsefulLinks' => $conference->navigations
                ->where('page_type', 3)
                ->where('status', 1)
                ->sortBy('orderpos'),
            'socialLinks'   => $conference->socialLinks,
            'mainaddress'   => \DB::table('web_cms')->where('page_type', 10)->where('status', 1)->first(),
            'contactInfo'   => $conference->address,
            'whatsappNumber'=> $conference->address?->whatsapp_number,
        ];
    }

    public function phoneCode($countryName): JsonResponse
    {
        $name = urldecode($countryName);
        $code = Country::where('country_name', $name)->value('country_code');
        return response()->json(['code' => (string)($code ?? '')]);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'conference_id'    => 'required|integer|exists:conferences,id',
            'designation'      => 'required|string|max:100',
            'firstname'        => 'required|string|max:100',
            'company'          => 'required|string|max:255',
            'usr_email'        => 'required|email',
            'usr_address'      => 'required|string',
            'country'          => 'required|string',
            'user_phone'       => 'required|string',
            'cp'               => 'required|string',
            'queries'          => 'nullable|string',
            'download_type'    => 'nullable|integer',
            'captcha_answer'   => 'required|numeric',
            'captcha_expected' => 'required|numeric',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        if ((int)$request->captcha_answer !== (int)$request->captcha_expected) {
            return response()->json(['error' => 'Invalid captcha'], 422);
        }

        $conference = Conference::findOrFail($request->conference_id);

        // Create registration
        $registration = Registration::create([
            'conference_id'       => $conference->id,
            'designation'         => $request->designation,
            'firstname'           => $request->firstname,
            'company'             => $request->company,
            'usr_email'           => strtolower($request->usr_email),
            'address'             => $request->usr_address,
            'country'             => $request->country,
            'user_phone'          => $request->user_phone,
            'usr_whatsapp'        => $request->usr_whatsapp,
            'cp'                  => $request->cp,
            'queries'             => $request->queries,
            'download_type'       => $request->download_type ?? 1,
            'user_type'           => 1,
            'created_date'        => now(),
            'invoiceid'           => '',
            'abstract_title'      => '',
            'transaction_id'      => '',
            'extra_info'          => '',
            'total_extra_amount'  => '',
            'coupon_value'        => '',
            'aditional_details'   => '',
        ]);

        // Send admin email
      //  $this->sendAdminEmail($conference, $request);

        // Send user email with brochure
      //  $this->sendUserEmailWithBrochure($conference, $request);

        return response()->json(['message' => 'Form submitted successfully!']);
    }

    private function sendAdminEmail(Conference $conference, Request $request)
    {
        $admin = $conference->address;
        $adminEmail = $admin?->primary_mail;

        if (!$adminEmail) return;

        $subject = "Brochure Downloaded from " . ($conference->short_name ?? $conference->suname ?? 'Conference');
        $msg  = "Name: {$request->firstname}<br>";
        $msg .= "Organization: {$request->company}<br>";
        $msg .= "Phone: {$request->user_phone}<br>";
        $msg .= "Mail: {$request->usr_email}<br>";
        $msg .= "Query: {$request->queries}<br>";

        Mail::send([], [], function ($message) use ($adminEmail, $conference, $msg, $subject) {
            $message->to($adminEmail)
                ->from($conference->smtp_user, $conference->suname)
                ->bcc('kmlalnehru9@gmail.com')
                ->subject($subject)
                ->html($msg);
        });
    }

    private function sendUserEmailWithBrochure(Conference $conference, Request $request)
    {
        $brochure = $conference->brochures()
            ->where('page_type', 1)
            ->first();

        if (!$brochure || !$request->usr_email) return;

        $filePath = public_path("uploads/conferences/{$conference->unique_id}/{$brochure->brouchure_name}");

        $subjectUser = "Thank you for your download.";
        $message  = "Dear {$request->firstname},<br><br>";
        $message .= "Greetings<br><br>";
        $message .= "Thank you for downloading the brochure.<br><br>";
        $message .= "Revert us for any further queries.<br><br>";
        $message .= "With thanks,<br>Conference Coordinator";

        Mail::send([], [], function ($mail) use ($request, $conference, $message, $subjectUser, $filePath) {
            $mail->to($request->usr_email)
                ->from($conference->smtp_user, $conference->suname)
                ->subject($subjectUser);

            if (is_string($filePath) && file_exists($filePath)) {
                $mail->attach($filePath);
            }

            $mail->html($message);
        });
    }
}